% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vglmer_regression.R
\name{vglmer}
\alias{vglmer}
\title{Variational Inference for Hierarchical Generalized Linear Models}
\usage{
vglmer(formula, data, family, control = vglmer_control())
}
\arguments{
\item{formula}{\code{lme4} style-formula for random effects. Typically,
\code{(1 + z | g)} indicates a random effect for each level of variable
\code{"g"} with a differing slope for the effect of variable \code{"z"} and
an intercept (\code{1}); see "Details" for further discussion and how to
incorporate splines.}

\item{data}{\code{data.frame} containing the outcome and predictors.}

\item{family}{Options are "binomial", "linear", or "negbin" (experimental).
If "binomial", outcome must be either binary (\eqn{\{0,1\}}) or
\code{cbind(success, failure)} as per standard \code{glm(er)} syntax.
Non-integer values are permitted for binomial if \code{force_whole} is set
to \code{FALSE} in \code{vglmer_control}.}

\item{control}{Adjust internal options for estimation. Must use an object
created by \link{vglmer_control}.}
}
\value{
This returns an object of class \code{vglmer}. The available methods
  (e.g. \code{coef}) can be found using \code{methods(class="vglmer")}.
\describe{
\item{beta}{Contains the estimated distribution of the fixed effects
(\eqn{\beta}). It is multivariate normal. \code{mean} contains the means;
\code{var} contains the variance matrix; \code{decomp_var} contains a matrix
\eqn{L} such that \eqn{L^T L} equals the full variance matrix.}
\item{alpha}{Contains the estimated distribution of the random effects
(\eqn{\alpha}). They are all multivariate normal. \code{mean} contains the
means; \code{dia.var} contains the variance of each random effect. \code{var}
contains the variance matrix of each random effect (j,g). \code{decomp_var}
contains a matrix \eqn{L} such that \eqn{L^T L} equals the full variance of
the entire set of random effects.}
\item{joint}{If \code{factorization_method="weak"}, this is a list with one
element (\code{decomp_var}) that contains a matrix \eqn{L} such that \eqn{L^T
L} equals the full variance matrix between the fixed and random effects
\eqn{q(\beta,\alpha)}. The marginal variances are included in \code{beta} and
\code{alpha}. If the factorization method is not \code{"weak"}, this is
\code{NULL}.}
\item{sigma}{Contains the estimated distribution of each random
effect covariance \eqn{\Sigma_j}; all distributions are Inverse-Wishart.
\code{cov} contains a list of the estimated scale matrices. \code{df}
contains a list of the degrees of freedom.}
\item{hw}{If a Huang-Wand prior is used (see Huang and Wand 2013 or Goplerud
2024 for more details), then the estimated distribution. Otherwise, it is
\code{NULL}. All distributions are Inverse-Gamma. \code{a} contains a list of
the scale parameters. \code{b} contains a list of the shape parameters.}
\item{sigmasq}{If \code{family="linear"}, this contains a list of the
estimated parameters for \eqn{\sigma^2}; its distribution is Inverse-Gamma.
\code{a} contains the scale parameter; \code{b} contains the shape
parameter.}
\item{ln_r}{If \code{family="negbin"}, this contains the variational
parameters for the log dispersion parameter \eqn{\ln(r)}. \code{mu} contains
the mean; \code{sigma} contains the variance.}
\item{family}{Family of outcome.}
\item{ELBO}{Contains the ELBO at the termination of the algorithm.}
\item{ELBO_trajectory}{\code{data.frame} tracking the ELBO per iteration.}
\item{control}{Contains the control parameters from \code{vglmer_control}
used in estimation.}
\item{internal_parameters}{Variety of internal parameters used in
post-estimation functions.}
\item{formula}{Contains the formula used for estimation; contains the
original formula, fixed effects, and random effects parts separately for
post-estimation functions. See \code{formula.vglmer} for more details.}
}
}
\description{
This function estimates hierarchical models using mean-field variational
inference. \code{vglmer} accepts standard syntax used for \code{lme4}, e.g.,
\code{y ~ x + (x | g)}. Options are described below. Goplerud (2022; 2024)
provides details on the variational algorithms.
}
\details{
\bold{Estimation Syntax:} The \code{formula} argument takes syntax designed
to be a similar as possible to \code{lme4}. That is, one can specify models
using \code{y ~ x + (1 | g)} where \code{(1 | g)} indicates a random intercept. While
not tested extensively, terms of \code{(1 | g / f)} should work as expected. Terms
of \code{(1 + x || g)} may work, although will raise a warning about duplicated
names of random effects. \code{(1 + x || g)} terms may not work with spline
estimation. To get around this, one can might copy the column \code{g} to
\code{g_copy} and then write \code{(1 | g) + (0 + x | g_copy)}.

\bold{Splines:} Splines can be added using the term \code{v_s(x)} for a
spline on the variable \code{x}. These are transformed into hierarchical
terms in a standard fashion (e.g. Ruppert et al. 2003) and then estimated
using the variational algorithms. At the present, only truncated linear
functions (\code{type = "tpf"}; the default) and O'Sullivan splines (Wand and
Ormerod 2008) are included. The options are described in more detail at
\link{v_s}.

It is possible to have the spline vary across some categorical predictor by
specifying the \code{"by"} argument such as \code{v_s(x, by = g)}. In effect,
this adds additional hierarchical terms for the group-level deviations from
the "global" spline. \emph{Note:} In contrast to the typical presentation of
these splines interacted with categorical variables (e.g., Ruppert et al.
2003), the default use of \code{"by"} includes the lower order interactions
that are regularized, i.e. \code{(1 + x | g)}, versus their unregularized
version (e.g., \code{x * g}); this can be changed using the \code{by_re}
argument described in \link{v_s}. Further, all group-level deviations from
the global spline share the same smoothing parameter (same prior
distribution).

\bold{Default Settings:} By default, the model is estimated using the
"strong" (i.e. fully factorized) variational assumption. Setting
\code{vglmer_control(factorization_method = "weak")} will improve the quality
of the variance approximation but may take considerably more time to
estimate. See Goplerud (2022) for discussion. 

By default, the prior on each random effect variance (\eqn{\Sigma_j}) uses a Huang-Wand prior (Huang
and Wand 2013) with hyper-parameters \eqn{\nu_j = 2} and \eqn{A_{j,k} = 5}.
This is designed to be proper but weakly informative. Other options are
discussed in \link{vglmer_control} under the \code{prior_variance} argument.

By default, estimation is accelerated using SQUAREM (Varadhan and Roland
2008) and (one-step-late) parameter expansion for variational Bayes. Under
the default \code{"strong"} factorization, a "translation" expansion is used;
under other factorizations a "mean" expansion is used. These can be adjusted
using \link{vglmer_control}. See Goplerud (2024) for more discussion of
these methods.
}
\examples{

set.seed(234)
sim_data <- data.frame(
  x = rnorm(100),
  y = rbinom(100, 1, 0.5),
  g = sample(letters, 100, replace = TRUE)
)

# Run with defaults
est_vglmer <- vglmer(y ~ x + (x | g), data = sim_data, family = "binomial")

# Simple prediction
predict(est_vglmer, newdata = sim_data)

# Summarize results
summary(est_vglmer)

# Extract parameters
coef(est_vglmer); vcov(est_vglmer)

# Comparability with lme4,
# although ranef is formatted differently.
ranef(est_vglmer); fixef(est_vglmer)

\donttest{
# Run with weaker (i.e. better) approximation
vglmer(y ~ x + (x | g),
  data = sim_data,
  control = vglmer_control(factorization_method = "weak"),
  family = "binomial")
}

\donttest{
# Use a spline on x with a linear outcome
vglmer(y ~ v_s(x),
  data = sim_data,
  family = "linear")
}

}
\references{
Goplerud, Max. 2022. "Fast and Accurate Estimation of Non-Nested Binomial
Hierarchical Models Using Variational Inference." \emph{Bayesian Analysis}. 17(2):
623-650.

Goplerud, Max. 2024. "Re-Evaluating Machine Learning for MRP Given the
Comparable Performance of (Deep) Hierarchical Models." \emph{American
Political Science Review}. 118(1): 529-536.

Huang, Alan, and Matthew P. Wand. 2013. "Simple Marginally Noninformative
Prior Distributions for Covariance Matrices." \emph{Bayesian Analysis}.
8(2):439-452.

Ruppert, David, Matt P. Wand, and Raymond J. Carroll. 2003.
\emph{Semiparametric Regression}. Cambridge University Press.

Varadhan, Ravi, and Christophe Roland. 2008. "Simple and Globally Convergent
Methods for Accelerating the Convergence of any EM Algorithm." \emph{Scandinavian
Journal of Statistics}. 35(2): 335-353.

Wand, Matt P. and Ormerod, John T. 2008. "On Semiparametric Regression with
O'Sullivan Penalized Splines". \emph{Australian & New Zealand Journal of Statistics}.
50(2): 179-198.
}
