#' Hashing
#'
#' @description
#' Map an arbitrary character string to a shorter string of hexadecimal
#' characters highly likely to be unique. It typically has a fixed width.
#'
#' **Arguments listed below are not validated for efficiency.**
#'
#' @details
#' Hashes generated by [hash()] uniquely identify the `lang` and `text` pair.
#' Values passed to these arguments are concatenated with a colon character for
#' hashing purposes.
#'
#' @param text A non-NA character string. It can be empty.
#'
#' @template param-lang
#'
#' @template param-algorithm
#'
#' @returns
#' [hash()] returns a character string, or `NULL` if `algorithm` is not
#' supported.
#'
#' @seealso
#' [`Translator`][Translator],
#' [`Text`][Text],
#' [normalize()],
#' [algorithms()]
#'
#' @keywords internal
hash <- function(lang = "", text = "", algorithm = "") {
    x <- sprintf("%s:%s", lang, text)

    return(
        switch(algorithm,
            sha1 = digest::digest(
                charToRaw(x),
                algo      = "sha1",
                serialize = FALSE),
            utf8 = as.character(sum(cumsum(utf8ToInt(x)))),
            NULL))
}

#' Hashing Algorithms
#'
#' These algorithms map a character string to another character string of
#' hexadecimal characters highly likely to be unique. The latter is used
#' to uniquely identify a source text (and the underlying source language).
#'
#' ## Secure Hash Algorithm 1
#'
#' Method `sha1` corresponds to SHA-1 (Secure Hash Algorithm version 1), a
#' cryptographic hashing function. While it is now superseded by more secure
#' variants (SHA-256, SHA-512, etc.), it is still useful for non-sensitive
#' purposes. It is fast, collision-resistant, and may handle very large inputs.
#' It emits strings of 40 hexadecimal characters.
#'
#' ## Cumulative UTF-8 Sum
#'
#' `r lifecycle::badge("experimental")`
#'
#' **This method is experimental. Use with caution.**
#'
#' Method `utf8` is a simple method derived from cumulative sums of UTF-8 code
#' points (converted to integers). It is slightly faster than method `sha1` for
#' small inputs and emits hashes with a width porportional to the underlying
#' input's length. It is used for testing purposes internally.
#'
#' @rdname constants
#' @keywords internal
algorithms <- function() {
    return(c("sha1", "utf8"))
}
