% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CI.R
\name{CI}
\alias{CI}
\alias{calc_H}
\alias{calc_e}
\alias{calc_Gamma}
\alias{calc_GammaINF}
\alias{calc_L_And}
\alias{calc_D}
\alias{calc_SE}
\title{Internal functions for the computation of confidence intervals}
\usage{
calc_H(smoothed_obj)

calc_e(smoothed_obj, H)

calc_Gamma(e, l)

calc_GammaINF(e, L)

calc_L_And(e, AR.method = c("yule-walker", "burg", "ols", "mle", "yw"))

calc_D(smoothed_obj)

calc_SE(
  smoothed_obj,
  h,
  AR.method = c("yule-walker", "burg", "ols", "mle", "yw")
)
}
\arguments{
\item{smoothed_obj}{an object created with \code{\link{calc_rho}}.}

\item{H}{an object created with \code{calc_H}.}

\item{e}{an object created with \code{calc_e}.}

\item{l}{a scalar indicating a number of time points.}

\item{L}{a scalar indicating a bandwidth parameter.}

\item{AR.method}{character string specifying the method to fit the autoregressive model used to compute \eqn{\hat{\gamma}_1} in \eqn{L_{And}} (see \code{\link[stats:ar]{stats::ar}} for details).}

\item{h}{a scalar indicating the bandwidth used by the smoothing function.}
}
\value{
\itemize{
\item \code{calc_H()} returns a 5 x 5 x \eqn{t} array of elements of class numeric, which corresponds to \eqn{\hat{H_t}} in Choi & Shin (2021).
\item \code{calc_e()} returns a \eqn{t} x 5 matrix of elements of class numeric storing the residuals, which corresponds to \eqn{\hat{e}_t} in Choi & Shin (2021).
\item \code{calc_Gamma()} returns a 5 x 5 matrix of elements of class numeric, which corresponds to \eqn{\hat{\Gamma}_l} in Choi & Shin (2021).
\item \code{calc_GammaINF()} returns a 5 x 5 matrix of elements of class numeric, which corresponds to \eqn{\hat{\Gamma}^\infty} in Choi & Shin (2021).
\item \code{calc_L_And()} returns a scalar of class numeric, which corresponds to \eqn{L_{And}} in Choi & Shin (2021).
\item \code{calc_D()} returns a \eqn{t} x 5 matrix of elements of class numeric storing the residuals, which corresponds to \eqn{D_t} in Choi & Shin (2021).
\item \code{calc_SE()} returns a vector of length \eqn{t} of elements of class numeric, which corresponds to \eqn{se(\hat{\rho}_t(h))} in Choi & Shin (2021).
}
}
\description{
These functions compute the different terms required for \code{\link[=tcor]{tcor()}} to compute the confidence
interval around the time-varying correlation coefficient. These terms are defined in Choi & Shin (2021).
}
\section{Functions}{
\itemize{
\item \code{calc_H()}: computes the \eqn{\hat{H_t}} array.

\eqn{\hat{H_t}} is a component needed to compute confidence intervals;
\eqn{H_t} is defined in eq. 6 from Choi & Shin (2021).

\item \code{calc_e()}: computes \eqn{\hat{e}_t}.

\eqn{\hat{e}_t} is defined in eq. 9 from Choi & Shin (2021).

\item \code{calc_Gamma()}: computes \eqn{\hat{\Gamma}_l}.

\eqn{\hat{\Gamma}_l} is defined in eq. 9 from Choi & Shin (2021).

\item \code{calc_GammaINF()}: computes \eqn{\hat{\Gamma}^\infty}.

\eqn{\hat{\Gamma}^\infty} is the long run variance estimator, defined in eq. 9 from Choi & Shin (2021).

\item \code{calc_L_And()}: computes \eqn{L_{And}}.

\eqn{L_{And}} is defined in Choi & Shin (2021, p 342).
It also corresponds to \eqn{S_T^*}, eq 5.3 in Andrews (1991).

\item \code{calc_D()}: computes \eqn{D_t}.

\eqn{D_t} is defined in Choi & Shin (2021, p 338).

\item \code{calc_SE()}: computes \eqn{se(\hat{\rho}_t(h))}.

The standard deviation of the time-varying correlation (\eqn{se(\hat{\rho}_t(h))}) is defined in eq. 8 from Choi & Shin (2021).
It depends on \eqn{D_{Lt}}, \eqn{D_{Mt}} & \eqn{D_{Ut}}, themselves defined in Choi & Shin (2021, p 337 & 339).
The \eqn{D_{Xt}} terms are all computed within the function since they all rely on the same components.

}}
\examples{
rho_obj <- with(na.omit(stockprice),
                calc_rho(x = SP500, y = FTSE100, t = DateID, h = 20, kernel = "box"))
head(rho_obj)

## Computing \eqn{\hat{H_t}}

H <- calc_H(smoothed_obj = rho_obj)
H[, , 1:2] # H array for the first two time points

## Computing \eqn{\hat{e}_t}

e <- calc_e(smoothed_obj = rho_obj, H = H)
head(e) # e matrix for the first six time points

## Computing \eqn{\hat{\Gamma}_l}

calc_Gamma(e = e, l = 3)

## Computing \eqn{\hat{\Gamma}^\infty}

calc_GammaINF(e = e, L = 2)

## Computing \eqn{L_{And}}

calc_L_And(e = e)
sapply(c("yule-walker", "burg", "ols", "mle", "yw"),
       function(m) calc_L_And(e = e, AR.method = m)) ## comparing AR.methods

## Computing \eqn{D_t}

D <- calc_D(smoothed_obj = rho_obj)
head(D) # D matrix for the first six time points

## Computing \eqn{se(\hat{\rho}_t(h))}
# nb: takes a few seconds to run

run <- FALSE ## change to TRUE to run the example
if (in_pkgdown() || run) {

SE <- calc_SE(smoothed_obj = rho_obj, h = 50)
head(SE) # SE vector for the first six time points

}


}
\references{
Choi, JE., Shin, D.W. Nonparametric estimation of time varying correlation coefficient.
J. Korean Stat. Soc. 50, 333–353 (2021). \doi{10.1007/s42952-020-00073-6}

Andrews, D. W. K. Heteroskedasticity and autocorrelation consistent covariance matrix estimation.
Econometrica: Journal of the Econometric Society, 817-858 (1991).
}
\seealso{
\code{\link[=tcor]{tcor()}}
}
