## ----style, echo = FALSE, results = 'asis'------------------------------------
BiocStyle::markdown()

## ----options, include = FALSE-------------------------------------------------
knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>"
)

## ----setup--------------------------------------------------------------------
library(tidyGenR)

## -----------------------------------------------------------------------------
# download test raw data (if not downloaded)
raw <-
  system.file("extdata", "raw", package = "tidyGenR")
freads <- list.files(raw,
    pattern = "1.fastq",
    full.names = TRUE)
rreads <- list.files(raw,
    pattern = "2.fastq",
    full.names = TRUE)

## ----primers------------------------------------------------------------------
# data.frame with primers
data("primers")
head(primers, 3)

## ----demultiplex, results = "hide", message=FALSE-----------------------------
p_sh_out <- tempfile(fileext = ".sh")
# demultiplex reads by locus using 3 primer pairs
demultiplex(
    freads = freads,
    rreads = rreads,
    primers = primers[1:3, ],
    sh_out = p_sh_out,
    run = FALSE
)

## ----cutadapt_script----------------------------------------------------------
readLines(p_sh_out)

## ----truncate_one_locus-------------------------------------------------------
# Download per-locus demultiplexed FASTQ
dem <-
  system.file("extdata", "demultiplexed", package = "tidyGenR")
# truncate reads for one locus
p_trun <- file.path(tempdir(), "truncated")
one_locus <-
    trunc_amp(
        loci = "chrna9",
        mode_trun = "pe",
        in_dir = dem,
        fw_pattern = "1.fastq.gz",
        rv_pattern = "2.fastq.gz",
        trunc_fr = c(250, 180),
        max_ee = c(3, 3),
        outdir = p_trun
    )

## ----trunc_fr-----------------------------------------------------------------
# dataframe with locus-specific truncation lengths
data("trunc_fr")

head(trunc_fr, 3)

## ----truncate_all_loci, results="hide", message=FALSE-------------------------
# truncate reads for all loci using locus-specific truncation lengths
all_loci <-
    trunc_amp(
        mode_trun = "pe",
        in_dir = dem,
        fw_pattern = "1.fastq.gz",
        rv_pattern = "2.fastq.gz",
        trunc_fr = trunc_fr,
        max_ee = c(3, 3),
        outdir = p_trun
    )

## ----truncated_fastq----------------------------------------------------------
head(list.files(p_trun))

## ----trunc_in_out-------------------------------------------------------------
head(all_loci, 3)

## ----variant_calling, message=FALSE, results="hide"---------------------------
variants <- variant_call(in_dir = p_trun)
head(variants)

## ----head_variants------------------------------------------------------------
head(variants)

## ----genotype, message = FALSE------------------------------------------------
genotypes <-
    genotype(variants, ADt = 10, ploidy = 2)
head(genotypes)

