% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rake_functions.R
\name{rakesvy}
\alias{rakesvy}
\alias{rakew8}
\title{Flexibly Calculate Rake Weights}
\usage{
rakesvy(
  design,
  ...,
  samplesize = "from.data",
  match.levels.by = "name",
  na.targets = "fail",
  rebase.tol = 0.01,
  control = list(maxit = 10, epsilon = 1, verbose = FALSE)
)

rakew8(
  design,
  ...,
  samplesize = "from.data",
  match.levels.by = "name",
  na.targets = "fail",
  rebase.tol = 0.01,
  control = list(maxit = 10, epsilon = 1, verbose = FALSE)
)
}
\arguments{
\item{design}{A \code{survey.design} object from \code{\link[survey:svydesign]{survey::svydesign()}},
or a data frame that  can be coerced to one. When a data frame is coerced, the
coercion assuming no clustering or design weighting.}

\item{...}{Formulas specifying weight targets, with an object that can be coerced
to class w8margin (see \code{\link[=as.w8margin]{as.w8margin()}}) on the right-hand side, and
(optionally) a matching variable or transformation of it on the left-hand side.
Objects that can be coerced to w8margin include named numeric vectors and matrices,
and data frames in the format accepted by \code{rake}.}

\item{samplesize}{Either a number specifying the desired post-raking sample
size, or a character string "from.data" or "from.targets" specifying how to
calculate the desired sample size (see details).}

\item{match.levels.by}{A character string that specifies how to match levels in
the target with the observed data, either "name" (the default) or "order"
(see details).}

\item{na.targets}{A characters string that specifies how to  handle NAs in \emph{targets},
either "fail" (the default) or "observed" (see details).}

\item{rebase.tol}{Numeric between 0 and 1. If targets are rebased, and
the rebased sample sizes differs from the original sample size by more than
this percentage, generates a warning.}

\item{control}{Parameters passed to the \code{control} argument of \code{\link[survey:rake]{survey::rake()}},
to control the details of convergence in weighting.}
}
\value{
\code{rakesvy()} returns a \code{survey.design} object with rake weights applied. Any changes
made to the variables in \code{design} in order to call \code{rake}, such as
dropping empty factor levels, are temporary and \emph{not} returned in the
output object.

\code{rakew8()} returns a vector of weights. This avoids creating
duplicated \code{survey.design} objects, which can be useful when calculating multiple
sets of weights for the same data.
}
\description{
Calculate rake weights on a data frame, or on a
\code{survey.design} object from \code{\link[survey:svydesign]{survey::svydesign()}}. Targets may be counts or
percentages, in vector, matrix, data frame, or w8margin form. Before
weighting, targets are converted to w8margins, checked for validity, and
matched to variables in observed data, \code{rakesvy} returns a weighted
\code{svydesign} object, while \code{rakew8} returns a vector of weights.
}
\details{
rakesvy and rakew8 wrangles observed data and targets into compatible formats,
before using \code{\link[survey:rake]{survey::rake()}} to make underlying weighting calculations.
The function matches weight targets to observed
variables, cleans both targets and observed variables, and then checks the
validity of weight targets (partially by calling
\code{\link[=w8margin_matched]{w8margin_matched()}}) before raking. It also allows a weight
target of zero, and assigns an automatic weight of zero to cases on this target
level.

Weight target levels can be matched with observed variable levels in
two ways, specified via the \code{match.levels.by} parameter. "name" (the
default) matches based on name, disregarding order (so a "male" level in
the weight target will be matched with a "male" level in the observed
data). "order" matches based on order, disregarding name (so the first
level or row of the target will match with the first level of the observed
factor variable).

By default, with parameter \code{na.targets = "fail"}), an NA in weight targets
will cause an error. With \code{na.targets = "observed"}, rakesvy() and rakew8() will instead
compute a target that matches the observed data. The category with an NA target will
therefore have a similar incidence rate in the pre-raking and post-raking dataset.
This is accomplished by calling \code{\link[=impute_w8margin]{impute_w8margin()}} before raking; see
the impute_w8margin documentation for more details. Note that NAs in \emph{observed}
data (as opposed to targets) will always cause failure, and are not affected by this parameter.

The desired sample size (in other words, the desired sum of weights
after raking)  is specified via the \code{samplesize} parameter. This can
be a numeric value. Alternatively, "from.data" specifies that the observed
sample size before weighting (taken from \code{sum(weights(design))} if
applicable, or \code{nrow()} if not); "from.targets" specifies that the total
sample sizes in target objects should be followed, and should only be used
if all targets specify the same sample size.
}
\examples{
# Computing only rake weights
# EG, for a survey conducted with simple random sampling
gles17$simple_weight <- rakew8(design = gles17, 
    gender ~ c("Male" = .495, "Female" = .505),
    eastwest ~ c("East Germany" = .195, "West Germany" = .805)
)

# Specifying a recode of variable in observed dataset
require(dplyr)
gles17_raked <- rakesvy(design = gles17, 
    gender ~ c("Male" = .495, "Female" = .505),
    dplyr::recode(agecat, `<=29` = "<=39", `30-39` = "<=39") ~ 
        c("<=39" = .31, "40-49" = .15, "50-59" = .19, "60-69" = .15, ">=70" = .21)
)

# Computing rake weights after design weights
# EG, for a survey with complex sampling design
require(survey)
gles17_dweighted <- svydesign(ids = gles17$vpoint, weights = gles17$dweight, 
    strata = gles17$eastwest, data = gles17, nest = TRUE)
gles17_raked <- rakesvy(design = gles17_dweighted, 
    gender ~ c("Male" = .495, "Female" = .505),
    agecat ~ c("<=29" = .16, "30-39" = .15, "40-49" = .15, 
        "50-59" = .19, "60-69" = .15, ">=70" = .21)
)

# With unnamed target levels, using match.levels.by = "order"
rakew8(design = gles17, 
    gender ~ c(.495, .505),
    eastwest ~ c(.195, .805),
    match.levels.by = "order"
)
}
