#' @export

skeleton <- function(x, mapping = NULL, fill = NULL, space = FALSE) {
  if ((!is.character(x)) | length(x) > 1) {
    stop("The parameter x must be a character vector of length 1.")
  }

  if (is.null(mapping)) {
    path    <- system.file("extdata", "etcsl_mapping.txt", package = "sumer")
    mapping <- read.csv2(path, sep = ";", na.strings = "")
  }

  "get_equation" <- function(x) {
    x   <- info(x, mapping = mapping)
    res <- paste0(c(paste0(x$reading, collapse = "-"),
                    paste0(x$name, collapse = "."),
                    paste0(x$sign, collapse = "")), collapse = "=")
    return(res)
  }

  # Convert to cuneiform and extract tokens
  x     <- as.cuneiform(x, mapping = mapping)
  token <- split_sumerian(x)$signs
  N     <- length(token)

  # Validate fill if provided
  if (!is.null(fill)) {
    fill_tokens <- split_sumerian(fill$expr[1])$signs
    if (!identical(fill_tokens, token) || nrow(fill) != N * (N + 1) / 2) {
      stop("The parameter fill does not match x.")
    }
  }

  # Normalize brackets and extract hierarchical structure
  x_marked <- mark_skeleton_entries(x)
  df       <- extract_skeleton_entries(x_marked)
  n_entries <- nrow(df)

  # Add column with indices into the fill data frame
  df$fill_idx <- substr_position(df$start, df$n_tokens, N)

  # Pre-compute type and translation columns from fill
  if (!is.null(fill)) {
    df$type        <- fill$type[df$fill_idx]
    df$translation <- fill$translation[df$fill_idx]
  } else {
    df$type        <- ""
    df$translation <- ""
  }

  # Build header line
  header     <- paste0(info(x_marked, mapping = mapping)$reading, collapse = "-")
  header_idx <- substr_position(1, N, N)

  if (!is.null(fill)) {
    header_type  <- fill$type[header_idx]
    header_trans <- fill$translation[header_idx]
  } else {
    header_type  <- ""
    header_trans <- ""
  }

  result <- paste0("|", header, ":", header_type, ":", header_trans)

  # Build entry lines
  for (i in 1:n_entries) {
    # Insert blank line before depth-1 entries
    if (space && df$depth[i] == 1) {
      result <- c(result, "")
    }

    eq <- get_equation(df$expr[i])

    result <- c(result, paste0("|", strrep("\t", df$depth[i] - 1),
                                eq, ":", df$type[i], ":", df$translation[i]))
  }

  class(result) <- c("skeleton", "character")
  result
}


#' @exportS3Method print skeleton
print.skeleton <- function(x, ...) {
  cat(x, sep = "\n")
}
