% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rspca.R
\name{rspca}
\alias{rspca}
\title{Randomized Sparse Principal Component Analysis (rspca).}
\usage{
rspca(X, k = NULL, alpha = 1e-04, beta = 1e-04, center = TRUE,
  scale = FALSE, max_iter = 1000, tol = 1e-05, o = 20, q = 2,
  verbose = TRUE)
}
\arguments{
\item{X}{array_like; \cr
a real \eqn{(n, p)} input matrix (or data frame) to be decomposed.}

\item{k}{integer; \cr
specifies the target rank, i.e., the number of components to be computed.}

\item{alpha}{float; \cr
Sparsity controlling parameter. Higher values lead to sparser components.}

\item{beta}{float; \cr
Amount of ridge shrinkage to apply in order to improve conditioning.}

\item{center}{bool; \cr
logical value which indicates whether the variables should be
shifted to be zero centered (TRUE by default).}

\item{scale}{bool; \cr
logical value which indicates whether the variables should
be scaled to have unit variance (FALSE by default).}

\item{max_iter}{integer; \cr
maximum number of iterations to perform before exiting.}

\item{tol}{float; \cr
stopping tolerance for the convergence criterion.}

\item{o}{integer; \cr
oversampling parameter (default \eqn{o=20}).}

\item{q}{integer; \cr
number of additional power iterations (default \eqn{q=2}).}

\item{verbose}{bool; \cr
logical value which indicates whether progress is printed.}
}
\value{
\code{spca} returns a list containing the following three components:
\item{loadings}{  array_like; \cr
          sparse loadings (weight) vector;  \eqn{(p, k)} dimensional array.
}

\item{transform}{  array_like; \cr
          the approximated inverse transform; \eqn{(p, k)} dimensional array.
}

\item{scores}{  array_like; \cr
          the principal component scores; \eqn{(n, k)} dimensional array.
}

\item{eigenvalues}{  array_like; \cr
         the approximated eigenvalues; \eqn{(k)} dimensional array.
}

\item{center, scale}{  array_like; \cr
                    the centering and scaling used.
}
}
\description{
Randomized accelerated implementation of SPCA, using variable projection as an optimization strategy.
}
\details{
Sparse principal component analysis is a modern variant of PCA. Specifically, SPCA attempts to find sparse
weight vectors (loadings), i.e., a weight vector with only a few 'active' (nonzero) values. This approach
leads to an improved interpretability of the model, because the principal components are formed as a
linear combination of only a few of the original variables. Further, SPCA avoids overfitting in a
high-dimensional data setting where the number of variables \eqn{p} is greater than the number of
observations \eqn{n}.

Such a parsimonious model is obtained by introducing prior information like sparsity promoting regularizers.
More concreatly, given an \eqn{(n,p)} data matrix \eqn{X}, SPCA attemps to minimize the following
objective function:

\deqn{ f(A,B) = \frac{1}{2} \| X - X B A^\top \|^2_F + \psi(B) }

where \eqn{B} is the sparse weight (loadings) matrix and \eqn{A} is an orthonormal matrix.
\eqn{\psi} denotes a sparsity inducing regularizer such as the LASSO (\eqn{\ell_1}{l1} norm) or the elastic net
(a combination of the \eqn{\ell_1}{l1} and \eqn{\ell_2}{l2} norm). The principal components \eqn{Z} are formed as

\deqn{ Z = X B }{Z = X B}

and the data can be approximately rotated back as

\deqn{ \tilde{X} = Z A^\top }{Xtilde = Z t(A)}

The print and summary method can be used to present the results in a nice format.
}
\note{
This implementation uses randomized methods for linear algebra to speedup the computations.
\eqn{o} is an oversampling parameter to improve the approximation.
A value of at least 10 is recommended, and \eqn{o=20} is set by default.

The parameter \eqn{q} specifies the number of power (subspace) iterations
to reduce the approximation error. The power scheme is recommended,
if the singular values decay slowly. In practice, 2 or 3 iterations
achieve good results, however, computing power iterations increases the
computational costs. The power scheme is set to \eqn{q=2} by default.

If \eqn{k > (min(n,p)/4)}, a the deterministic \code{\link{spca}}
algorithm might be faster.
}
\examples{

# Create artifical data
m <- 10000
V1 <- rnorm(m, 0, 290)
V2 <- rnorm(m, 0, 300)
V3 <- -0.1*V1 + 0.1*V2 + rnorm(m,0,100)

X <- cbind(V1,V1,V1,V1, V2,V2,V2,V2, V3,V3)
X <- X + matrix(rnorm(length(X),0,1), ncol = ncol(X), nrow = nrow(X))

# Compute SPCA
out <- rspca(X, k=3, alpha=1e-3, beta=1e-3, center = TRUE, scale = FALSE, verbose=0)
print(out)
summary(out)

}
\references{
\itemize{

 \item [1] N. B. Erichson, P. Zheng, K. Manohar, S. Brunton, J. N. Kutz, A. Y. Aravkin.
 "Sparse Principal Component Analysis via Variable Projection."
 Submitted to IEEE Journal of Selected Topics on Signal Processing (2018).
 (available at `arXiv \url{https://arxiv.org/abs/1804.00341}).


  \item  [1] N. B. Erichson, S. Voronin, S. Brunton, J. N. Kutz.
         "Randomized matrix decompositions using R."
         Submitted to Journal of Statistical Software (2016).
         (available at `arXiv \url{http://arxiv.org/abs/1608.02148}).
}
}
\seealso{
\code{\link{spca}}, \code{\link{robspca}}
}
\author{
N. Benjamin Erichson, Peng Zheng, and Sasha Aravkin
}
