% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/custom_cache.R
\name{custom_cache}
\alias{custom_cache}
\title{Create a custom cache backend (cachem-like)}
\usage{
custom_cache(get, set, remove, take = NULL, info = NULL)
}
\arguments{
\item{get}{A function(key, missing = NULL) -> value. Required.
Should return the stored value, or the \code{missing} argument if the key is not present.
The \code{missing} parameter is mandatory because both \code{OAuthClient} and
\code{OAuthProvider} validators will pass it explicitly.}

\item{set}{A function(key, value) -> invisible(NULL). Required.
Should store the value under the given key}

\item{remove}{A function(key) -> any. Required.

Deletes the entry for \code{key}. When \verb{$take()} is provided, \verb{$remove()} serves
only as a best-effort cleanup and its return value is ignored. When
\verb{$take()} is not provided, 'shinyOAuth' falls back to
\verb{$get()} + \verb{$remove()} followed by a post-removal absence check via
\verb{$get(key, missing = NA)}. In this fallback path the return value of
\verb{$remove()} is not relied upon; the post-check is authoritative.}

\item{take}{A function(key, missing = NULL) -> value. Optional.

An atomic get-and-delete operation. When provided, 'shinyOAuth' uses
\verb{$take()} instead of separate \verb{$get()} + \verb{$remove()} calls to enforce
single-use state consumption. This prevents TOCTOU (time-of-check /
time-of-use) replay attacks in multi-worker deployments with shared state
stores.

Should return the stored value and atomically remove the entry, or
return the \code{missing} argument (default \code{NULL}) if the key is not present.

If your backend supports atomic get-and-delete natively
(e.g., Redis \code{GETDEL}, SQL \verb{DELETE ... RETURNING}), wire it through this
parameter for replay-safe state stores.

When \code{take} is not provided and the state store is not a per-process cache
(like \code{\link[cachem:cache_mem]{cachem::cache_mem()}}), 'shinyOAuth' will \strong{error} at state
consumption time because non-atomic \verb{$get()} + \verb{$remove()} cannot
guarantee single-use under concurrent access in shared stores.}

\item{info}{Function() -> list(max_age = seconds, ...). Optional

This may be provided to because TTL information from \verb{$info()} is used to
align browser cookie max age in \code{oauth_module_server()}}
}
\value{
An R6 object exposing cachem-like \verb{$get/$set/$remove/$info} methods
}
\description{
Builds a minimal cachem-like cache backend object that exposes cachem-compatible methods:
\verb{$get(key, missing)}, \verb{$set(key, value)}, \verb{$remove(key)}, and \verb{$info()}.

Use this helper when you want to plug a custom state store or JWKS cache
into 'shinyOAuth', when \code{\link[cachem:cache_mem]{cachem::cache_mem()}} is not suitable (e.g.,
multi-process deployments with non-sticky workers).
In such cases, you may want to use a shared external cache (e.g., database,
Redis, Memcached).

The resulting object can be used in both places where 'shinyOAuth' accepts a cache-like object:
\itemize{
\item OAuthClient@state_store (requires \verb{$get}, \verb{$set}, \verb{$remove}; optional \verb{$info})
\item OAuthProvider@jwks_cache (requires \verb{$get}, \verb{$set}; optional \verb{$remove}, \verb{$info})
}

The \verb{$info()} method is optional, but if provided and it returns a list with
\code{max_age} (seconds), shinyOAuth will align browser cookie max-age in
\code{\link[=oauth_module_server]{oauth_module_server()}} to that value.
}
\examples{
mem <- new.env(parent = emptyenv())

my_cache <- custom_cache(
  get = function(key, missing = NULL) {
    base::get0(key, envir = mem, ifnotfound = missing, inherits = FALSE)
  },

  set = function(key, value) {
    assign(key, value, envir = mem)
    invisible(NULL)
  },

  remove = function(key) {
    if (exists(key, envir = mem, inherits = FALSE)) {
      rm(list = key, envir = mem)
    }
    invisible(NULL)
  },

  # Atomic get-and-delete: preferred for state stores in multi-worker
  # deployments to prevent TOCTOU replay attacks. For per-process caches
  # (like cachem::cache_mem()) this is optional; for shared backends (Redis,
  # database) it should map to the backend's atomic primitive (e.g., GETDEL).
  take = function(key, missing = NULL) {
    val <- base::get0(key, envir = mem, ifnotfound = missing, inherits = FALSE)
    if (exists(key, envir = mem, inherits = FALSE)) {
      rm(list = key, envir = mem)
    }
    val
  },

  info = function() list(max_age = 600)
)
}
