% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/see.R
\name{see}
\alias{see}
\alias{see}
\alias{see.formula}
\alias{see.default}
\alias{see.fit}
\title{Stagewise Estimating Equations Implementation}
\usage{
see(y, ...)

\method{see}{formula}(formula, data = list(), clusterID, waves = NULL,
  contrasts = NULL, subset, ...)

\method{see}{default}(y, x, waves = NULL, ...)

\method{see}{fit}(y, x, family, clusterID, waves = NULL,
  corstr = "independence", alpha = NULL, intercept = TRUE, offset = 0,
  control = sgee.control(maxIt = 200, epsilon = 0.05, stoppingThreshold =
  min(length(y), ncol(x)) - intercept, undoThreshold = 0), standardize = TRUE,
  verbose = FALSE, ...)
}
\arguments{
\item{y}{Vector of response measures that corresponds with modeling family
given in 'family' parameter. \code{y} is assumed to be the same length as
\code{clusterID} and is assumed to be organized into clusters as dictated by
\code{clusterID}.}

\item{...}{Not currently used}

\item{formula}{Object of class 'formula'; a symbolic description of
the model to be fitted}

\item{data}{Optional data frame containing the variables in the model.}

\item{clusterID}{Vector of integers that identifies the clusters of response
measures in \code{y}. Data and \code{clusterID} are assumed to
1) be of equal lengths,
2) sorted so that observations of a cluster are in contiguous rows, and 3)
organized so that \code{clusterID} is a vector of consecutive integers.}

\item{waves}{An integer vector which identifies components in clusters.
The length of \code{waves} should be the same as the number of
observations. \code{waves} is automatically generated if none is supplied,
but when using \code{subset} parameter, the \code{waves} parameter must be
provided by the user for proper calculation.}

\item{contrasts}{An optional list provided when using a formula.
similar to \code{contrasts} from \code{glm}.
See the \code{contrasts.arg} of \code{model.matrix.default}.}

\item{subset}{An optional vector specifying a subset of observations to be
used in the fitting process.}

\item{x}{Design matrix of dimension \code{length(y)} x nvar,
the number of variables, where each row is
represents an observation of predictor variables.}

\item{family}{Modeling family that describes the marginal distribution of
the response. Assumed to be an object such as \code{gaussian()} or
\code{poisson()}.}

\item{corstr}{A character string indicating the desired working correlation
structure. The following are implemented : "independence" (default value),
"exchangeable", and "ar1".}

\item{alpha}{An initial guess for the correlation parameter value
between -1 and 1 . If left NULL (the default), the initial estimate is 0.}

\item{intercept}{Binary value indicating where an intercept term is
to be included in the model for estimation. Default is to include an
intercept.}

\item{offset}{Vector of offset value(s) for the linear predictor.
\code{offset}
is assumed to be either of length one, or of the same length as \code{y}.
Default is to have no offset.}

\item{control}{A list of parameters used to contorl the path generation
process; see \code{sgee.control}.}

\item{standardize}{A logical parameter that indicates whether or not
the covariates need to be standardized before fitting.
If standardized before fitting, the unstandardized
path is returned as the default, with a \code{standardizedPath} and
\code{standardizedX} included
separately. Default value is \code{TRUE}.}

\item{verbose}{Logical parameter indicating whether output should be produced
while bisee is running. Default value is FALSE.}
}
\value{
Object of class \code{sgee} containing the path
of coefficient estimates,
the path of scale estimates, the path of correlation parameter
estimates, the iteration at which SEE terminated, and initial regression
values including \code{x}, \code{y}, code{family}, \code{clusterID},
\code{groupID}, \code{offset}, \code{epsilon}, and \code{numIt}.
}
\description{
Function to perform SEE, a  Forward Stagewise regression
approach for model selection / dimension reduction
using Generalized Estimating Equations
}
\details{
Function to implement SEE, a stagewise regression approach
that is designed to perform model selection in the context of
Generalized Estimating Equations. Given a response \code{y} and
a design matrix \code{x}
(excluding intercept) SEE generates a path of stagewise regression
estimates for each covariate based on the provided step size \code{epsilon}.

The resulting path can then be analyzed to determine an optimal
model along the path of coefficient estimates. The
\code{summary.sgee} function provides such
functionality based on various
possible metrics, primarily focused on the Mean Squared Error.
Furthermore, the \code{plot.sgee} function can be used to examine the
path of coefficient estimates versus the iteration number, or some
desired penalty.

A stochastic version of this function can also be called. using the
auxiliary function \code{sgee.control} the parameters \code{stochastic},
\code{reSample}, and \code{withReplacement} can be given to \code{see}
to perform a sub sampling step in the procedure to make the SEE
implementation scalable for large data sets.
}
\examples{


#####################
## Generate test data
#####################

## Initialize covariate values
p <- 50 
beta <- c(rep(2,5),
          c(1, 0, 1.5, 0, .5),
          rep(0.5,5),
          rep(0,p-15))
groupSize <- 1
numGroups <- length(beta)/groupSize


generatedData <- genData(numClusters = 50,
                         clusterSize = 4,
                         clusterRho = 0.6,
                         clusterCorstr = "exchangeable",
                         yVariance = 1,
                         xVariance = 1,
                         numGroups = numGroups,
                         groupSize = groupSize,
                         groupRho = 0.3,
                         beta = beta,
                         family = gaussian(),
                         intercept = 1)



## Perform Fitting by providing formula and data
genDF <- data.frame(generatedData$y, generatedData$x)
names(genDF) <- c("Y", paste0("Cov", 1:p))
coefMat1 <- see(formula(genDF), data = genDF,
                 family = gaussian(),
                 waves = rep(1:4, 50), 
                 clusterID = generatedData$clusterID,
                 groupID = generatedData$groupID, 
                 corstr = "exchangeable",
                 control = sgee.control(maxIt = 50, epsilon = 0.5),
                 verbose = TRUE)

## set parameter 'stochastic' to 0.5 to implement the stochastic
## stagewise approach where a subsmaple of 50\% of the data is taken
## before the path is calculation.
## See sgee.control for more details about the parameters for the
## stochastic stagewise approach

coefMat2 <- see(formula(genDF), data = genDF,
                 family = gaussian(),
                 waves = rep(1:4, 50), 
                 clusterID = generatedData$clusterID,
                 groupID = generatedData$groupID, 
                 corstr = "exchangeable",
                 control = sgee.control(maxIt = 50, epsilon = 0.5,
                                        stochastic = 0.5), 
                 verbose = FALSE)

par(mfrow = c(2,1))
plot(coefMat1)
plot(coefMat2)

}
\references{
Vaughan, G., Aseltine, R., Chen, K., Yan, J., (2017). Stagewise
Generalized Estimating Equations with Grouped Variables. Biometrics 73,
1332-1342. URL: http://dx.doi.org/10.1111/biom.12669,
doi:10.1111/biom.12669.

Wolfson, J. (2011). EEBoost: A general method for prediction
and variable selection based on estimating equations. Journal of the
American Statistical Association 106, 296--305.

Tibshirani, R. J. (2015). A general framework for fast stagewise
algorithms. Journal of Machine Learning Research 16, 2543--2588.
}
\author{
Gregory Vaughan
}
