% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/intra_influ_phyglm.R
\name{intra_influ_phyglm}
\alias{intra_influ_phyglm}
\title{Interaction between intraspecific variability and influential species - Phylogenetic Logistic Regression}
\usage{
intra_influ_phyglm(
  formula,
  data,
  phy,
  Vx = NULL,
  n.intra = 30,
  x.transf = NULL,
  distrib = "normal",
  cutoff = 2,
  btol = 50,
  track = TRUE,
  ...
)
}
\arguments{
\item{formula}{The model formula:  \code{response~predictor}.}

\item{data}{Data frame containing species traits with row names matching tips
in \code{phy}.}

\item{phy}{A phylogeny (class 'phylo') matching \code{data}.}

\item{Vx}{Name of the column containing the standard deviation or the standard error of the predictor 
variable. When information is not available for one taxon, the value can be 0 or \code{NA}}

\item{n.intra}{Number of datasets resimulated taking into account intraspecific variation (see: \code{"intra_phylgm"})}

\item{x.transf}{Transformation for the predictor variable (e.g. \code{"log"} or \code{"sqrt"}). Please use this 
argument instead of transforming data in the formula directly (see also details below).}

\item{distrib}{A character string indicating which distribution to use to generate a random value for the response 
and/or predictor variables. Default is normal distribution: "normal" (function \code{\link{rnorm}}).
Uniform distribution: "uniform" (\code{\link{runif}})
Warning: we recommend to use normal distribution with Vx = standard deviation of the mean.}

\item{cutoff}{The cutoff value used to identify for influential species
(see Details)}

\item{btol}{Bound on searching space. For details see \code{phyloglm}}

\item{track}{Print a report tracking function progress (default = TRUE)}

\item{...}{Further arguments to be passed to \code{phylolm}}
}
\value{
The function \code{intra_influ_phylm} returns a list with the following
components:

\code{cutoff}: The value selected for \code{cutoff}

\code{formula}: The formula

\code{full.model.estimates}: Coefficients, aic and the optimised
value of the phylogenetic parameter (e.g. \code{lambda}) for the full model
without deleted species.

\code{influential_species}: List of influential species, both
based on standardised difference in intercept and in the slope of the
regression. Species are ordered from most influential to less influential and
only include species with a standardised difference > \code{cutoff}.

\code{sensi.estimates}: A data frame with all simulation
estimates. Each row represents a deleted clade for an iteration of resimulated
data. Columns report the calculated regression intercept (\code{intercept}), 
difference between simulation intercept and full model intercept (\code{DIFintercept}), 
the standardised difference (\code{sDIFintercept}), the percentage of change in intercept compared
to the full model (\code{intercept.perc}) and intercept p-value
(\code{pval.intercept}). All these parameters are also reported for the regression
slope (\code{DIFestimate} etc.). Additionally, model aic value (\code{AIC}) and
the optimised value (\code{optpar}) of the phylogenetic parameter
(e.g. \code{kappa} or \code{lambda}, depending on the phylogenetic model used) are
reported.

\code{data}: Original full dataset.

\code{errors}: Species where deletion resulted in errors.
}
\description{
Performs leave-one-out deletion analysis for phylogenetic logistic regression,
and detects influential species, while taking into account potential
interactions with intraspecific variability.
}
\details{
This function fits a phylogenetic linear regression model using \code{\link[phylolm]{phylolm}}, and detects
influential species by sequentially deleting one at a time. The regression is repeated \code{n.intra} times for 
simulated values of the dataset, taking into account intraspecific variation. At each iteration, the function 
generates a random value for each row in the dataset using the standard deviation or errors supplied, and 
detect the influential species within that iteration. 

\code{influ_phylm} detects influential species based on the standardised
difference in intercept and/or slope when removing a given species compared
to the full model including all species. Species with a standardised difference
above the value of \code{cutoff} are identified as influential. The default
value for the cutoff is 2 standardised differences change.

Currently, this function can only implement simple models (i.e. \eqn{trait~
predictor}). In the future we will implement more complex models.

Output can be visualised using \code{sensi_plot}.
}
\section{Warning}{
  
When Vx exceeds X negative (or null) values can be generated, this might cause problems
for data transformation (e.g. log-transformation). In these cases, the function will skip the simulation. This problem can
be solved by increasing \code{n.intra}, changing the transformation type and/or checking the target species in output$sp.pb.

Setting \code{n.intra} at high values can take a long time to execute, since the total number of iterations equals \code{n.intra * nrow(data)}.
}

\examples{
\dontrun{
#Generate data
set.seed(6987)
phy = rtree(100)
x = rTrait(n=1,phy=phy,parameters=list(ancestral.state=2,optimal.value=2,sigma2=1,alpha=1))
X = cbind(rep(1,100),x)
y = rbinTrait(n=1,phy=phy, beta=c(-1,0.5), alpha=.7 ,X=X)
z = rnorm(n = length(x),mean = mean(x),sd = 0.1*mean(x))
dat = data.frame(y, x, z)
# Run sensitivity analysis:
intra_influ <- intra_influ_phyglm(formula = y ~ x, data = dat, phy = phy, 
                       Vx = "z", n.intra = 5,track = TRUE,distrib="normal",x.transf=NULL) 
# To check summary results and most influential species:
summary(intra_influ)
# Visual diagnostics for clade removal:
sensi_plot(intra_influ)
}
\dontshow{
phy = rtree(100)
x = rTrait(n=1,phy=phy,parameters=list(ancestral.state=2,optimal.value=2,sigma2=1,alpha=1))
X = cbind(rep(1,100),x)
y = rbinTrait(n=1,phy=phy, beta=c(-1,0.5), alpha=.7 ,X=X)
z = rnorm(n = length(x),mean = mean(x),sd = 0.1*mean(x))
dat = data.frame(y, x, z)
# Run sensitivity analysis:
intra_influ <- intra_influ_phyglm(formula = y ~ x, data = dat[1:20,], phy = phy, 
                                 Vx = "z", n.intra = 2,track = TRUE,
                                 distrib="normal",x.transf=NULL) 
# To check summary results and most influential species:
summary(intra_influ)
# Visual diagnostics for clade removal:
sensi_plot(intra_influ)
}
}
\references{
Paterno, G. B., Penone, C. Werner, G. D. A. 
\href{http://doi.wiley.com/10.1111/2041-210X.12990}{sensiPhy: 
An r-package for sensitivity analysis in phylogenetic 
comparative methods.} Methods in Ecology and Evolution 
2018, 9(6):1461-1467.  

Ho, L. S. T. and Ane, C. 2014. "A linear-time algorithm for 
Gaussian and non-Gaussian trait evolution models". Systematic Biology 63(3):397-408.
}
\seealso{
\code{\link[phylolm]{phylolm}}, \code{\link{intra_phyglm}},
\code{\link{influ_phyglm}},\code{\link{intra_influ_phylm}},\code{\link{sensi_plot}}.
}
\author{
Gustavo Paterno, Caterina Penone & Gijsbert D.A. Werner
}
