% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/genomic_indices.R
\name{clgsi}
\alias{clgsi}
\title{Combined Linear Genomic Selection Index (CLGSI)}
\usage{
clgsi(
  phen_mat = NULL,
  gebv_mat = NULL,
  pmat,
  gmat,
  wmat,
  wcol = 1,
  P_y = NULL,
  P_g = NULL,
  P_yg = NULL,
  reliability = NULL,
  selection_intensity = 2.063,
  GAY = NULL
)
}
\arguments{
\item{phen_mat}{Matrix of phenotypes (n_genotypes x n_traits). Can be NULL if P_y and P_yg are provided.}

\item{gebv_mat}{Matrix of GEBVs (n_genotypes x n_traits). Can be NULL if P_g and P_yg are provided.}

\item{pmat}{Phenotypic variance-covariance matrix (n_traits x n_traits)}

\item{gmat}{Genotypic variance-covariance matrix (n_traits x n_traits)}

\item{wmat}{Economic weights matrix (n_traits x k), or vector}

\item{wcol}{Weight column to use if wmat has multiple columns (default: 1)}

\item{P_y}{Optional. Phenotypic variance-covariance matrix computed from data (n_traits x n_traits).
If NULL (default), uses pmat or computes from phen_mat using cov().}

\item{P_g}{Optional. GEBV variance-covariance matrix (n_traits x n_traits).
If NULL (default), computed from gebv_mat using cov().}

\item{P_yg}{Optional. Covariance matrix between phenotypes and GEBVs (n_traits x n_traits).
If NULL (default), computed from phen_mat and gebv_mat using cov().}

\item{reliability}{Optional. Reliability of GEBVs (r^2, the squared correlation). See lgsi() for details.}

\item{selection_intensity}{Selection intensity i (default: 2.063 for 10\% selection)}

\item{GAY}{Optional. Genetic advance of comparative trait for PRE calculation}
}
\value{
List with components:
  \itemize{
    \item \code{b_y} - Coefficients for phenotypes
    \item \code{b_g} - Coefficients for GEBVs
    \item \code{b_combined} - Full coefficient vector [b_y; b_g]
    \item \code{P_combined} - Combined variance matrix
    \item \code{Delta_H} - Expected genetic advance per trait
    \item \code{GA} - Overall genetic advance
    \item \code{PRE} - Percent relative efficiency
    \item \code{hI2} - Index heritability
    \item \code{rHI} - Index accuracy
    \item \code{summary} - Data frame with all metrics
  }
}
\description{
Implements the Combined Linear Genomic Selection Index where selection combines
both phenotypic observations and Genomic Estimated Breeding Values (GEBVs).
This is used for selecting candidates with both phenotype and genotype data
(e.g., in a training population).
}
\details{
\strong{Mathematical Formulation:}

The CLGSI combines phenotypic and genomic information:
\deqn{I = \mathbf{b}_y' \mathbf{y} + \mathbf{b}_g' \hat{\mathbf{g}}}

Coefficients are obtained by solving the partitioned system:
\deqn{\begin{bmatrix} \mathbf{b}_y \\ \mathbf{b}_g \end{bmatrix} =
      \begin{bmatrix} \mathbf{P} & \mathbf{P}_{y\hat{g}} \\
                      \mathbf{P}_{y\hat{g}}' & \mathbf{P}_{\hat{g}} \end{bmatrix}^{-1}
      \begin{bmatrix} \mathbf{G} \\ \mathbf{C}_{\hat{g}g} \end{bmatrix} \mathbf{w}}

Where:
- \eqn{\mathbf{P}} = Var(phenotypes)
- \eqn{\mathbf{P}_{\hat{g}}} = Var(GEBVs)
- \eqn{\mathbf{P}_{y\hat{g}}} = Cov(phenotypes, GEBVs)
- \eqn{\mathbf{G}} = Genotypic variance-covariance
- \eqn{\mathbf{C}_{\hat{g}g}} = Cov(GEBV, true BV)
}
\examples{
\dontrun{
# Generate example data
gmat <- gen_varcov(seldata[, 3:9], seldata[, 2], seldata[, 1])
pmat <- phen_varcov(seldata[, 3:9], seldata[, 2], seldata[, 1])

# Simulate phenotypes and GEBVs
set.seed(123)
n_genotypes <- 100
n_traits <- ncol(gmat)

phen_mat <- matrix(rnorm(n_genotypes * n_traits, mean = 15, sd = 3),
  nrow = n_genotypes, ncol = n_traits
)
gebv_mat <- matrix(rnorm(n_genotypes * n_traits, mean = 10, sd = 2),
  nrow = n_genotypes, ncol = n_traits
)
colnames(phen_mat) <- colnames(gmat)
colnames(gebv_mat) <- colnames(gmat)

# Economic weights
weights <- c(10, 5, 3, 3, 5, 8, 4)

# Calculate CLGSI
result <- clgsi(phen_mat, gebv_mat, pmat, gmat, weights, reliability = 0.7)
print(result$summary)
}
}
\references{
Cerón-Rojas, J. J., & Crossa, J. (2018). Linear Selection Indices in Modern Plant Breeding.
Springer International Publishing.
}
