##' Scatter Plot
##'
##' S3 methods generating scatter plot for some objects generated by
##' \code{rrpack} using \code{ggplot2}. An \code{ggplot2} object is returned so
##' that users are allowed to easily further customize the plot.
##'
##' @name plot
##'
##' @param x Some object generated by \code{rrpack}.
##' @param y NULL. Do not need to specify.
##' @param layer The unit-rank layer to plot; cannot be larger than the estimated rank
##' @param xlab Label of X axis.
##' @param ylab Label of Y axis.
##' @param ... Other argumnts for future usage.
##'
##' @return ggplot2 object.
##'
##' @importFrom ggplot2 ggplot geom_point xlab ylab aes
NULL


##' @rdname plot
##' @export
plot.rrr <- function(x, y = NULL, 
                     layer = 1L,
                     xlab = paste("latent predictor ", layer, sep = ""),
                     ylab = paste("latent response ", layer, sep = ""), ...)
{
    if(layer > ncol(x$U)) stop("Layer number exceeded the estimated rank.")
    xVec <- with(x, X %*% U[, layer])
    yVec <- with(x, Y %*% V[, layer])
    ggDat <- data.frame(x = xVec, y = yVec)
    out <- ggplot(ggDat, aes(x = x, y = y)) +
        geom_point() +
        ggplot2::xlab(xlab) +
        ggplot2::ylab(ylab)
    out
}

##' @rdname plot
##' @export
plot.sofar <- plot.rrr

##' @rdname plot
##' @export
plot.cv.sofar <- plot.rrr

##' @rdname plot
##' @export
plot.srrr <- plot.rrr

##' @rdname plot
##' @export
plot.cv.srrr <- plot.rrr

##' @rdname plot
##' @export
plot.rssvd <- plot.rrr
