#' Show the content of an OSM file
#'
#' Displays the content of an OSM file either in `.html`, `.xml` or `.opl`
#' format.
#'
#' @param input_path A string. The path to the OSM file whose content should be
#'   shown. Please see [file_formats] for a list of supported file formats.
#' @param output_format A string. The format in which the content should be
#'   shown. Either `"html"`, `"opl"` or `"xml"`. Please note that the `"html"`
#'   option, although the most human readable format, results in large files
#'   that may be very slow to open and inspect, depending on the size of the
#'   input file.
#' @param object_type A character vector. The object types that should be
#'   included in the output. Please note that using `"all"` is a simple way of
#'   passing all other available types (`"node"`, `"way"`, `"relation"` and
#'   `"changeset"`).
#' @param echo_cmd A logical. Whether to print the Osmium command generated by
#'   the function call to the screen. Defaults to `FALSE`.
#' @param spinner A logical. Whether to show a reassuring spinner while the
#'   Osmium call is being executed. Defaults to `TRUE`.
#' @param preview A logical. Whether to open the temporary file in which the
#'   content of the OSM file was saved.
#'
#' @return The path to the temporary file in which the OSM file content was
#'   saved.
#'
#' @examplesIf identical(tolower(Sys.getenv("NOT_CRAN")), "true")
#' pbf_path <- system.file("extdata/cur.osm.pbf", package = "rosmium")
#'
#' small_pbf <- tags_filter(
#'   pbf_path,
#'   "note",
#'   tempfile(fileext = "osm.pbf"),
#'   omit_referenced = TRUE
#' )
#'
#' # both function calls below result in outputs with the same object types
#'
#' show_content(small_pbf, object_type = "all")
#' show_content(
#'   small_pbf,
#'   object_type = c("node", "way", "relation", "changeset")
#' )
#'
#' # only show nodes and ways
#'
#' show_content(small_pbf, object_type = c("node", "way"))
#'
#' # display all objects in xml format
#'
#' show_content(small_pbf, output_format = "xml")
#'
#' @export
show_content <- function(
  input_path,
  output_format = c("html", "opl", "xml"),
  object_type = c("all", "node", "way", "relation", "changeset"),
  echo_cmd = FALSE,
  spinner = TRUE,
  preview = TRUE
) {
  assert_osmium_is_installed()

  checkmate::assert_file_exists(input_path)
  checkmate::assert_logical(echo_cmd, any.missing = FALSE, len = 1)
  checkmate::assert_logical(spinner, any.missing = FALSE, len = 1)
  checkmate::assert_logical(preview, any.missing = FALSE, len = 1)

  output_format_arg <- assert_and_assign_output_fmt(output_format)
  object_type_arg <- assert_and_assign_type(object_type)

  args <- c(
    "show",
    input_path,
    output_format_arg,
    object_type_arg
  )

  logs <- processx::run(
    "osmium",
    args,
    spinner = spinner,
    echo_cmd = echo_cmd,
    echo = FALSE
  )

  tmpfile <- write_content_to_file(logs$stdout, output_format)
  if (preview && interactive()) utils::browseURL(tmpfile)

  return(normalizePath(tmpfile))
}

assert_and_assign_output_fmt <- function(output_format) {
  possible_choices <- c("html", "opl", "xml")

  if (!identical(output_format, possible_choices)) {
    coll <- checkmate::makeAssertCollection()
    checkmate::assert_string(output_format, add = coll)
    checkmate::assert_names(
      output_format,
      subset.of = possible_choices,
      add = coll
    )
    checkmate::reportAssertions(coll)
  }

  output_format_input <- output_format[1]

  output_format_input <- if (output_format_input == "html") {
    "--output-format=debug"
  } else {
    paste0("--format-", output_format)
  }

  return(output_format_input)
}

assert_and_assign_type <- function(object_type) {
  possible_choices <- c("all", "node", "way", "relation", "changeset")
  checkmate::assert_names(object_type, subset.of = possible_choices)

  if ("all" %in% object_type) {
    object_type <- c(object_type, possible_choices[-1])

    all_index <- grep("^all$", object_type)
    object_type <- object_type[-all_index]

    object_type <- unique(object_type)
  }

  object_type_input <- paste0("--object-type=", object_type)

  return(object_type_input)
}

write_content_to_file <- function(content, output_format) {
  output_format <- output_format[1]

  fileext <- paste0(".", output_format)
  tmpfile <- tempfile("osm_content", fileext = fileext)

  if (output_format == "html") {
    content <- gsub("\r|\n|\r\n", "<br>", content)
  }

  cat(content, file = tmpfile)

  return(normalizePath(tmpfile))
}
