#' Process the pathogenic mutation
#'
#' @param pathog.mut A character variable specifying the pathogenic mutation following the HGVS-like syntax for sub, del, ins, delins, or dup.
#' @param reference.genome A BSgenome or DNAStringSet object of reference sequences
#' 
#' @return A list of pathogenic mutation details including chr.id, chr.name, input mutation, standard HGVS syntax, mutation type, start/end positions, reference/alternative alleles
#' 
#' @importFrom GenomeInfoDb commonName
#' @importFrom BSgenome getSeq
#' @importFrom GenomicRanges GRanges
#' @importFrom IRanges IRanges
#' 
#' @examples
#' \dontrun{
#'     processPathogMut()
#' }
#'
#' @noRd
processPathogMut <- function(pathog.mut, reference.genome){
    
    parsed.mut <- parseHGVS( pathog.mut )
    seq.id <- parsed.mut$seq.id
    ref.genome.class <- class(reference.genome)
    
    if( ref.genome.class=="BSgenome" ){
        
        ref.species <- GenomeInfoDb::commonName(reference.genome) 
        if(ref.species=="Human"){
            primary.chr.names <- paste0("chr", c(1:22,"X","Y"))
            genome.chromosomes <- "human genome (chromosomes 1-22,X,Y)" 
        }else if(ref.species=="Mouse"){
            primary.chr.names <- paste0("chr", c(1:19,"X","Y"))
            genome.chromosomes <- "mouse genome (chromosomes 1-19,X,Y)" 
        }
        
        chr.id <- sub("^CHR", "", toupper(seq.id))
        chr.name <- paste0("chr", chr.id)
        if( !( chr.name %in% primary.chr.names ) ){
            stop("The chromosome identifier in pathogenic mutation is not included in the primary assembly of ", genome.chromosomes, ".")
        }
        
    }else if( ref.genome.class=="DNAStringSet" ){
        
        chr.id <- seq.id
        chr.name <- seq.id
        if( !( chr.name %in% names(reference.genome) ) ){
            stop("The sequence identifier in pathogenic mutation is not included in the reference fasta file.")
        }
        
    }
    
    mut.start <- parsed.mut$start
    mut.end <- parsed.mut$end
    mut.ref <- parsed.mut$ref
    mut.alt <- parsed.mut$alt

    if(parsed.mut$type=="snv"){
        
        mut.type <- "SNV"
        true.ref.allele <- as.character( BSgenome::getSeq(reference.genome, GenomicRanges::GRanges(chr.name, IRanges::IRanges(mut.start, mut.end))) )
        if( mut.ref != true.ref.allele ){
            stop("The reference nucleotide base in pathogenic mutation does not match the reference sequence.")
        }
        mut.hgvs.coord <- paste0(mut.start, mut.ref, ">", mut.alt)
        
    }else if(parsed.mut$type=="del"){
        
        mut.type <- "DEL"
        true.ref.del <- as.character( BSgenome::getSeq(reference.genome, GenomicRanges::GRanges(chr.name, IRanges::IRanges(mut.start, mut.end))) )
        if( grepl( "[^ATCG]", true.ref.del ) ){
            stop("The reference sequence of pathogenic mutation locus contains unkown bases.")
        }
        if( true.ref.del!=mut.ref & !is.na(mut.ref) ){
            stop("The deletion sequence in pathogenic mutation does not match the reference sequence.")
        }
        mut.ref <- nchar(true.ref.del)
        if(mut.start==mut.end){
            mut.hgvs.coord <- paste0(mut.start, "del")
        }else{
            mut.hgvs.coord <- paste0(mut.start, "_", mut.end, "del")
        }
        
    }else if(parsed.mut$type=="ins"){
        
        mut.type <- "INS"
        mut.hgvs.coord <- paste0(mut.start, "_", mut.end, "ins", mut.alt)
        
    }else if(parsed.mut$type=="delins"){
        
        mut.type <- "DELINS"
        true.ref.del <- as.character( BSgenome::getSeq(reference.genome, GenomicRanges::GRanges(chr.name, IRanges::IRanges(mut.start, mut.end-1))) )
        if( grepl( "[^ATCG]", true.ref.del ) ){
            stop("The reference sequence of pathogenic mutation locus contains unkown bases.")
        }
        mut.ref <- nchar(true.ref.del)
        if(mut.start==(mut.end-1)){
            mut.hgvs.coord <- paste0(mut.start, "delins", mut.alt)
        }else{
            mut.hgvs.coord <- paste0(mut.start, "_", mut.end-1, "delins", mut.alt)
        }
        
    }else if(parsed.mut$type=="dup"){
        
        mut.type <- "INS"
        true.ref.dup <- as.character( BSgenome::getSeq(reference.genome, GenomicRanges::GRanges(chr.name, IRanges::IRanges(mut.start, mut.end))) )
        if( grepl( "[^ATCG]", true.ref.dup ) ){
            stop("The reference sequence of pathogenic mutation locus contains unkown bases.")
        }
        if( is.na(mut.alt) ){
            mut.alt <- true.ref.dup
        }else{
            if( mut.alt != true.ref.dup ){
                stop("The duplication sequence in pathogenic mutation does not match the reference sequence.")
            }
        }
        mut.end <- mut.start
        mut.start <- mut.start - 1
        mut.hgvs.coord <- paste0(mut.start, "_", mut.end, "ins", mut.alt)
        
    }
    
    if( mut.type=="DEL" ){
        left.aln.mut <- leftAlignIndel(
            reference.genome,
            chr.name,
            mut.type,
            mut.start, 
            mut.end, 
            mut.ref,
            mut.alt
        )
        mut.start <- left.aln.mut$mut.start
        mut.end <- left.aln.mut$mut.end
        mut.ref <- left.aln.mut$mut.ref
        mut.alt <- left.aln.mut$mut.alt
        if(mut.start==mut.end){
            mut.hgvs.coord <- paste0( mut.start, "del")
        }else{
            mut.hgvs.coord <- paste0( mut.start, "_", mut.end, "del")
        }
    }else if(mut.type=="INS"){
        left.aln.mut <- leftAlignIndel(
            reference.genome,
            chr.name,
            mut.type,
            mut.start, 
            mut.end, 
            mut.ref,
            mut.alt
        )
        mut.start <- left.aln.mut$mut.start
        mut.end <- left.aln.mut$mut.end
        mut.ref <- left.aln.mut$mut.ref
        mut.alt <- left.aln.mut$mut.alt
        mut.hgvs.coord <- paste0( mut.start, "_", mut.end, "ins", mut.alt)
    }
    
    if( ref.genome.class=="BSgenome" ){
        mut.hgvs <- paste0( chr.name, ":g.", mut.hgvs.coord )
    }else if( ref.genome.class=="DNAStringSet" ){
        mut.hgvs <- paste0( chr.name, ":", mut.hgvs.coord )
    }
    
    pathog.mut.obj <- list(
        chr.id = chr.id, 
        chr.name = chr.name, 
        mut.raw = pathog.mut, 
        mut.hgvs = mut.hgvs,
        mut.type = mut.type, 
        mut.start = mut.start,
        mut.end = mut.end, 
        mut.ref = mut.ref, 
        mut.alt = mut.alt
    )
    
    return(pathog.mut.obj)
}



