#' Get the information of a gene from an Ensembl-based annotation database
#'
#' @param gene.name The gene name
#' @param chr.id The chromosome ID of the gene
#' @param ensdb An EnsDb object of Ensembl-based annotation database
#' 
#' @return A list of gene annotation including gene name, Ensembl Gene ID, chromosome id, strand, list of protein-coding transcripts, the longest transcript
#' 
#' @importFrom ensembldb genes
#' @importFrom ensembldb transcripts
#' @importFrom ensembldb cdsBy
#' @importFrom IRanges ranges
#' @importFrom BiocGenerics width
#' @importFrom BiocGenerics strand
#' 
#' @examples
#' \dontrun{
#'     getGeneInfo()
#' }
#'
#' @noRd
getGeneInfo <- function(gene.name, chr.id, ensdb){
    gene.id <- NA
    gene.strand <- NA
    tx.list <- NA
    tx.id <- NA
    
    gene.gr <- ensembldb::genes(ensdb, filter = ~ (gene_name==gene.name & seq_name==chr.id & gene_biotype=="protein_coding") )
    if( length(gene.gr)>=1 ){
        gene.gr <- gene.gr[1]
        gene.id <- gene.gr$gene_id
        gene.strand <- as.character(BiocGenerics::strand(gene.gr))
        tx.gr <- ensembldb::transcripts(ensdb, filter = ~ (gene_id==gene.id & tx_biotype=="protein_coding") )
        tx.list <- names(tx.gr)
        cds.gr <- ensembldb::cdsBy(ensdb, filter = ~ (tx_id==tx.list) )
        cds.length <- sapply(cds.gr, function(x){sum(BiocGenerics::width(IRanges::ranges(x)))}  )
        tx.id <- names(cds.length[which(cds.length==max(cds.length))])[1]
        
    }
    
    gene.obj <- list(
        gene.name = gene.name, 
        gene.id = gene.id, 
        chr.id = chr.id, 
        strand = gene.strand, 
        tx.list = tx.list, 
        longest.tx = tx.id
    )
    
    return( gene.obj )
}


