#' Pairwise alignment with left- or right-normalization for indels
#'
#' @param ref The reference sequence 
#' @param reads The read sequence
#' @param gapShifting Normalize indels to left or right
#' @param alignment.type Type of alignment, please see details of parameter type in function pwalign::pairwiseAlignment()
#' @param match The scoring for a nucleotide match in the alignment
#' @param mismatch The scoring for a nucleotide mismatch in the alignment
#' @param gapOpening The cost for opening a gap in the alignment
#' @param gapExtension The incremental cost incurred along the length of the gap in the alignment
#' 
#' @return A pair of aligned sequences
#' 
#' @importFrom pwalign nucleotideSubstitutionMatrix
#' @importFrom pwalign pairwiseAlignment
#' @importFrom pwalign alignedPattern
#' @importFrom pwalign alignedSubject
#' @importFrom IRanges reverse
#'
#' @examples
#' \dontrun{
#'     gapShiftingPairwiseAlignment()
#' }
#'
#' @noRd
gapShiftingPairwiseAlignment <- function(
		ref, 
		reads, 
		gapShifting = "left",
		alignment.type = "global",
		match = 1,
		mismatch = 4,
		gapOpening = 6, 
		gapExtension = 1 ){
    
	submat <- pwalign::nucleotideSubstitutionMatrix( 
    		match = match, 
    		mismatch = mismatch, 
    		baseOnly = FALSE, 
    		type = "DNA" 
    )
	
	if( gapShifting == "left" ){
		aln.pw <- pwalign::pairwiseAlignment(
    			pattern = IRanges::reverse(ref), 
    			subject = IRanges::reverse(reads), 
    			type = alignment.type,
    			substitutionMatrix = submat,
    			gapOpening = gapOpening, 
    			gapExtension = gapExtension 
		)
		aln.sequences <- IRanges::reverse( as.character( c(pwalign::alignedPattern(aln.pw), pwalign::alignedSubject(aln.pw)) ) ) 
	}else if( gapShifting == "right" ){
		aln.pw <- pwalign::pairwiseAlignment(
    			pattern = ref, 
    			subject = reads, 
    			type = alignment.type,
    			substitutionMatrix = submat,
    			gapOpening = gapOpening, 
    			gapExtension = gapExtension 
		)
		aln.sequences <- as.character( c(pwalign::alignedPattern(aln.pw), pwalign::alignedSubject(aln.pw)) ) 
	}
	
	return(aln.sequences)
}
