#' rSPDE linear mixed effects models
#'
#' Fitting linear mixed effects model with latent Whittle-Matern models.
#'
#' @param formula Formula object describing the relation between the response 
#' variables and the fixed effects. If the response variable is a matrix, each 
#' column of the matrix will be treated as a replicate.
#' @param loc A vector with the names of the columns in `data` that contain the 
#' observation locations, or a `matrix` or a `data.frame` containing the 
#' observation locations. If the model is of class `metric_graph`, the locations 
#' must be either a `matrix` or a `data.frame` with two columns, or a character 
#' vector with the names of the two columns. The first column being the number of 
#' the edge, and the second column being the normalized position on the edge. 
#' If the model is a 2d model, `loc` must be either a `matrix` or `data.frame` 
#' with two columns or a character vector with the name of the two columns that 
#' contain the location, the first entry corresponding to the `x` entry and the 
#' second corresponding to the `y` entry.
#' @param loc_time For spatio-temporal models, the name of the column in `data` that
#' is the time variable, or a `matrix` or `vector` containing the observation time
#' points. 
#' @param data A `data.frame` containing the data to be used.
#' @param model Object generated by `matern.operators()`, `spde.matern.operators()`
#' or `spacetime.operators()`. If `NULL`, simple linear regression will be performed.
#' @param repl Vector indicating the replicate of each observation. 
#' If `NULL` it will assume there is only one replicate. If the model is generated from graphs from
#'  `metric_graph` class and `use_data_from_graph` is `TRUE`, `repl` needs to be the name of the 
#' column inside the metric graph data that contains the replicate. If `NULL` it will assume there is only
#' one replicate.
#' @param which_repl Which replicates to use? If `NULL` all replicates will be used.
#' @param optim_method The method to be used with `optim` function.
#' @param possible_methods The optimization methods to try if the model fitting fails.
#' @param use_data_from_graph Logical. Only for models generated from graphs from 
#' `metric_graph` class. In this case, should the data, the locations and the 
#' replicates be obtained from the graph object?
#' @param rspde_order The order of the rational approximation to be used while 
#' fitting the model. If not given, the order from the model object will be used.
#' Not used for spatio-temporal models.
#' @param mean_correction If TRUE, use mean correction for intrinsic models. Default FALSE. 
#' @param previous_fit An object of class `rspde_lme`. Use the fitted coefficients as starting values.
#' @param fix_coeff If using a previous fit, should all coefficients be fixed at the starting values?
#' @param model_options A list containing additional options to be used in the model. This will take preference over the values extracted from previous_fit. Currently, it is possible to fix parameters during the estimation or change the starting values of the parameters. The general structure of the elements of the list is `fix_parname` and `start_parname`, where `parname` stands for the name of the parameter. If `fix_parname` is not `NULL`, then the model will be fitted with the `parname` being fixed at the value that was passed. If `start_parname` is not `NULL`, the model will be fitted using the value passed as starting value for `parname`. For 'rSPDE' and 'rSPDE1d' models, the possible elements of the list are `fix_sigma_e`, `start_sigma_e`, `fix_nu`, `start_nu`, `fix_sigma`, `start_sigma`, `fix_range`, `start_range`. Alternatively, one can also use the elements `fix_sigma_e`, `start_sigma_e`, `fix_nu`, `start_nu`, `fix_tau`, `start_tau`, `fix_kappa`, `start_kappa`. For 'spacetime' models, the possible elements of the list are `fix_sigma_e`, `start_sigma_e`, `fix_kappa`, `start_kappa`, `fix_sigma`, `start_sigma`, `fix_gamma`, `start_gamma`, `fix_rho`, `start_rho`. For dimension 2, the second coordinate of `rho` has name `rho2` and must be passed as `start_rho2` and `fix_rho2`. For 'rSPDE2d' models, the possible elements of the list are `fix_sigma_e`, `start_sigma_e`, `fix_nu`, `start_nu`, `fix_sigma`, `start_sigma`, `fix_hx`, `start_hx`, `fix_hy`, `start_hy`, `fix_hxy`, `start_hxy`. For nonstationary models, we have two options: the first is to pass the starting values as a vector with name `start_theta` and a vector with the names of the parameters to be fixed with name    `fix_theta`; the second option is to handle the individual parameters, by passing the names `thetan` where `n` is the number of the parameter to pass the starting value or to be fixed. For example, to pass the starting value for `theta3` one simply pass `start_theta3`, and to fix `theta2`, one simply pass `fix_theta2`.
#' @param smoothness_upper_bound Upper bound for `nu`, when `nu` is the smoothness parameter, or for `alpha - d/2` when `alpha` is the smoothness parameter, or for `max\{alpha - d/2, beta - d/2)` for intrinsic models.
#' @param parallel logical. Indicating whether to use optimParallel or not.
#' @param n_cores Number of cores to be used if parallel is true.
#' @param optim_controls Additional controls to be passed to `optim` or `optimParallel`.
# @param improve_gradient Should a more precise estimate of the gradient be obtained? Turning on might increase the overall time. If `TRUE`, the "Richardson" method will be used. See the help of the `grad` function in `numDeriv` package for details.
# @param gradient_args List of controls to be used for the gradient. The list can contain the arguments to be passed to the `method.args` argument in the `numDeriv::grad` function. See the help of the `grad` function in `numDeriv` package for details.
#' @param improve_hessian Should a more precise estimate of the hessian be obtained? 
#' Turning on might increase the overall time.
#' @param hessian_args List of controls to be used if `improve_hessian` is `TRUE`. 
#' The list can contain the arguments to be passed to the `method.args` argument 
#' in the `numDeriv::hessian` function. See the help of the `hessian` function in 
#' `numDeriv` package for details. Observe that it only accepts the "Richardson" 
#' method for now, the method "complex" is not supported.
#' @param alpha `r lifecycle::badge("deprecated")` Use `model_options` instead.
#' @param nu `r lifecycle::badge("deprecated")` Use `model_options` instead.
#' @param beta `r lifecycle::badge("deprecated")` Use `model_options` instead.
#' @param starting_values_latent `r lifecycle::badge("deprecated")` Use `model_options` instead.
#' @param start_sigma_e `r lifecycle::badge("deprecated")` Use `model_options` instead.
#' @param start_alpha `r lifecycle::badge("deprecated")` Use `model_options` instead.
#' @param start_nu `r lifecycle::badge("deprecated")` Use `model_options` instead.
#' @param start_beta `r lifecycle::badge("deprecated")` Use `model_options` instead.
#' @param nu_upper_bound `r lifecycle::badge("deprecated")` Use `smoothness_upper_bound` instead.
#' @return A list containing the fitted model.
#' @rdname rspde_lme
#' @export
rspde_lme <- function(formula, 
                      loc, 
                      loc_time = NULL,
                      data,
                      model = NULL, 
                      repl = NULL,
                      which_repl = NULL,
                      optim_method = "L-BFGS-B",
                      possible_methods = c("L-BFGS-B", "Nelder-Mead"),
                      use_data_from_graph = TRUE,
                      rspde_order = NULL,
                      mean_correction = FALSE,
                      previous_fit = NULL,
                      fix_coeff = FALSE, 
                      model_options = list(),                      
                      smoothness_upper_bound = 4,
                      parallel = FALSE,
                      n_cores = parallel::detectCores() - 1,
                      optim_controls = list(),
                      improve_hessian = FALSE,
                      hessian_args = list(),
                      alpha = lifecycle::deprecated(),
                      nu = lifecycle::deprecated(),
                      beta = lifecycle::deprecated(),
                      starting_values_latent = lifecycle::deprecated(),
                      start_sigma_e = lifecycle::deprecated(),
                      start_alpha = lifecycle::deprecated(),
                      start_nu = lifecycle::deprecated(),
                      start_beta = lifecycle::deprecated(),
                      nu_upper_bound = lifecycle::deprecated()
                      ) {

  available_models <- c("CBrSPDEobj", 
                        "rSPDEobj", 
                        "rSPDEobj1d", 
                        "spacetimeobj", 
                        "CBrSPDEobj2d",
                        "intrinsicCBrSPDEobj")
  
  null_model <- TRUE
  spacetime <- inherits(model, "spacetimeobj")
  anisotropic <- inherits(model, "CBrSPDEobj2d")
  intrinsic <- inherits(model, "intrinsicCBrSPDEobj")
  stationary <- model$stationary
  if(is.null(stationary)) {
    stationary <- TRUE
  }

  if (lifecycle::is_present(starting_values_latent)) {
    lifecycle::deprecate_warn(
      "2.5.0", 
      what = "rspde_lme(starting_values_latent)",
      with = "rspde_lme(model_options)",
      details = "Starting values for latent parameters should now be specified in model_options."
    )
    if (!is.null(starting_values_latent)) {
      model_options$start_theta <- starting_values_latent
    }
  }

  if (lifecycle::is_present(start_sigma_e)) {
    lifecycle::deprecate_warn(
      "2.5.0", 
      what = "rspde_lme(start_sigma_e)",
      with = "rspde_lme(model_options)",
      details = "Starting value for sigma_e should now be specified in model_options."
    )
    if (!is.null(start_sigma_e)) {
      model_options$start_sigma_e <- start_sigma_e
    }
  }

  if (lifecycle::is_present(start_alpha)) {
    lifecycle::deprecate_warn(
      "2.5.0", 
      what = "rspde_lme(start_alpha)",
      with = "rspde_lme(model_options)",
      details = "Starting value for alpha should now be specified in model_options."
    )
    if (!is.null(start_alpha)) {
      model_options$start_alpha <- start_alpha
    }
  }

  if (lifecycle::is_present(start_nu)) {
    lifecycle::deprecate_warn(
      "2.5.0", 
      what = "rspde_lme(start_nu)",
      with = "rspde_lme(model_options)",
      details = "Starting value for nu should now be specified in model_options."
    )
    if (!is.null(start_nu)) {
      model_options$start_nu <- start_nu
    }
  }

  if (lifecycle::is_present(start_beta)) {
    lifecycle::deprecate_warn(
      "2.5.0", 
      what = "rspde_lme(start_beta)",
      with = "rspde_lme(model_options)",
      details = "Starting value for beta should now be specified in model_options."
    )
    if (!is.null(start_beta)) {
      model_options$start_beta <- start_beta
    }
  }

  if (lifecycle::is_present(beta)) {
    lifecycle::deprecate_warn(
      "2.5.0", 
      what = "rspde_lme(beta)",
      with = "rspde_lme(model_options)",
      details = "Fixed value for beta should now be specified in model_options."
    )
    if (!is.null(beta)) {
      model_options$fix_beta <- beta
    }
  }

  if (lifecycle::is_present(alpha)) {
    lifecycle::deprecate_warn(
      "2.5.0", 
      what = "rspde_lme(alpha)",
      with = "rspde_lme(model_options)",
      details = "Fixed value for alpha should now be specified in model_options."
    )
    if (!is.null(alpha)) {
      model_options$fix_alpha <- alpha
    }
  }

  if (lifecycle::is_present(nu)) {
    lifecycle::deprecate_warn(
      "2.5.0", 
      what = "rspde_lme(nu)",
      with = "rspde_lme(model_options)",
      details = "Fixed value for nu should now be specified in model_options."
    )
    if (!is.null(nu)) {
      model_options$fix_nu <- nu
    }
  }

  if (lifecycle::is_present(nu_upper_bound)) {
    lifecycle::deprecate_warn(
      "2.5.0", 
      what = "rspde_lme(nu_upper_bound)",
      with = "rspde_lme(smoothness_upper_bound)",
      details = "Upper bound for nu should now be specified in smoothness_upper_bound ."
    )
    if (!is.null(nu_upper_bound)) {
      smoothness_upper_bound <- nu_upper_bound
    }
  }

  if (!is.null(model)) {
    if (!inherits(model, available_models)) {
      stop(paste0("The model should be an object of class '", paste(available_models, collapse = "', '"), "'."))
    }
    null_model <- FALSE
  }
  
  if(!inherits(model,"spacetimeobj") && !is.null(loc_time)) {
      warning("loc_time should only be supplied for spatio-temporal models. loc_time will be ignored.")
  }

  # Process model_options for nonstationary models
  model_options <- process_model_options(model, model_options)

  # General checks
  parameterization <- general_checks_model_options(model_options, model)
  general_checks_lme(model, model_options)

  model_options <- extract_starting_values(previous_fit, fix_coeff = fix_coeff, model_options = model_options)

  if (is.null(formula)) {
    stop("No formula provided!")
  }

  call_rspde_lme <- match.call()

  likelihood_new <- NULL
  new_likelihood <- NULL

  if (null_model) {
    model <- list(
      has_graph = FALSE,
      stationary = FALSE
    )
  }

  time_data_start <- Sys.time()

  if (missing(data) && (!model$has_graph)) {
    data <- environment(formula)
  } else if (model$has_graph) {
    if (use_data_from_graph) {
      if (is.null(model$graph$.__enclos_env__$private$data)) {
        stop("The graph has no data! Either add data to the graph, or add the data manually and set 'use_data_from_graph' to FALSE.")
      }
      data <- model$graph$.__enclos_env__$private$data
      if(!is.null(repl)){
        if(!(repl%in%names(model$graph$.__enclos_env__$private$data))){
          stop("Since 'use_data_from_graph' is TRUE, 'repl' must be the name of a column inside the graph!")
        }
        repl <- model$graph$.__enclos_env__$private$data[[repl]]
      }
      if (missing(loc)) {
        # Don't do anything, we will replace loc anyway
      }
      loc <- cbind(
        model$graph$.__enclos_env__$private$data[[".edge_number"]],
        model$graph$.__enclos_env__$private$data[[".distance_on_edge"]]
      )
    }
  }

  y_term <- stats::terms(formula)[[2]]

  y_resp <- eval(y_term, envir = data, enclos = parent.frame())
  y_resp <- as.numeric(y_resp)

  if (model$has_graph  && use_data_from_graph) {
    idx_Y <- !is.na(y_resp)
    y_resp <- y_resp[idx_Y]
    data <- lapply(data, function(x) x[idx_Y])
    loc <- loc[idx_Y, , drop = FALSE]
  }

  cov_term <- stats::delete.response(terms(formula))

  X_cov <- stats::model.matrix(cov_term, data)

  if (all(dim(X_cov) == c(0, 1))) {
    names_temp <- colnames(X_cov)
    X_cov <- matrix(1, nrow = length(y_resp))
    colnames(X_cov) <- names_temp
  }

  if (is.null(repl)) {
    repl <- rep(1, length(y_resp))
  }

  if(is.character(repl) && length(repl) == 1) {
      if(repl %in% colnames(data)) {
          repl <- data[[repl]]    
      } else {
        stop(paste("repl provides as a name of a column in data, but the data does not contain the column:", repl, ". The column names in data are:", paste(colnames(data), collapse = ", ")))
      }   
  }

  if (is.null(which_repl)) {
    which_repl <- unique(repl)
  }
  
  which_repl <- unique(which_repl)
  
  if (length(setdiff(which_repl, repl)) > 0) {
    warning("There are elements in 'which_repl' that are not in 'repl'.")
  }

  idx_repl <- (repl %in% which_repl)

  y_resp <- y_resp[idx_repl]

  if (ncol(X_cov) > 0) {
    X_cov <- X_cov[idx_repl, , drop = FALSE]
  } else {
    X_cov <- matrix(ncol = 0, nrow = 0)
  }

  repl <- repl[idx_repl]

  time_data_end <- Sys.time()

  time_data <- time_data_end - time_data_start


  ### To add new models in rspde_lme the main things to check are:
  ### Does the new model use the spde/matern parameterization? if so, update the parameterization as below
  ### Does it use a rational approximation, if so, use rspde_order as below
  ### extract_starting_values should be robust in general and not be updated (it only differentiates between nonstationary and other models, because nonstationary models have parameters without name)
  ### process_model_options might need to be updated depending on the model, if it does not have spde/matern parameterization, it needs to be excluded
  ### general_checks_lme might need to be updated
  ### get_model_starting_values this is one that most likely needs to be updated as it handles the names of the parameters as well
  ### update_starting_values might need to be updated if some of the parameters need a more special transformation
  ### It is possible get_starting_values_lme needs to be updated, depending on the new parameters
  ### get_aux_likelihood_function must be updated by including the new model's auxiliary likelihood function
  ### determine_estimate_params should be robust and most likely does not need to be updated
  ### extract_model_update_args might need to be updated by including new parameter names
  ### get_aux_lik_fun_args probably wont need to be updated, but if a novel argument appears in the auxiliary likelihood function, then it must be included there
  ### create_likelihood is designed to be robust and not need any updates
  ### Will need to update the exported objects to the parallel cores
  ### Will not need to change anything related to the model fitting

  if (!null_model) {
    time_build_likelihood_start <- Sys.time()

    # General updates for the different models

    if (!inherits(model, "spacetimeobj") && !inherits(model, "CBrSPDEobj2d") && !inherits(model, "intrinsicCBrSPDEobj")) {
        model <- update(model, parameterization = parameterization)    
    }

    if (!is.null(rspde_order) && !is.null(model)) {
        if (inherits(model, "CBrSPDEobj") || inherits(model, "rSPDEobj") || inherits(model, "rSPDEobj1d")) {
          model <- update(model, m = rspde_order)      
        }
    } else if (!is.null(model)) {
      rspde_order <- model$m
    } else {
      rspde_order <- NULL
    }

    # Getting starting values

    start_values <- get_model_starting_values(model = model, model_options = model_options, y_resp = y_resp, parameterization = parameterization) 

    if (is.data.frame(loc) || is.matrix(loc)) {
      loc_df <- loc
    } else if (is.character(loc)) {
      if (!model$has_graph) {
        dim <- model$d
        if(dim > 1) {
            if(spacetime) {
                if ( (length(loc) != (dim+1) && (model$mesh_space$manifold == "S2")) || 
                     (length(loc) != dim && !(model$mesh_space$manifold == "S2"))){
                    stop("If 'loc' is a character vector, it must have the same length as the dimension (unless model comes from a metric graph).")
                }    
            } else {
                if ( (length(loc) != (dim+1) && (model$mesh$manifold == "S2")) || 
                     (length(loc) != dim && !(model$mesh$manifold == "S2"))){
                    stop("If 'loc' is a character vector, it must have the same length as the dimension (unless model comes from a metric graph).")
                }        
            }
            
        } else {
            if ( length(loc) != dim){
                stop("If 'loc' is a character vector, it must have the same length as the dimension (unless model comes from a metric graph).")
            }
        }
        
        if (dim == 1) {
          loc_df <- matrix(data[[loc[1]]], ncol = 1)
        } else {
            if(spacetime) {
                if(model$mesh_space$manifold == "S2"){
                    loc_df <- cbind(
                        as.vector(data[[loc[1]]]),
                        as.vector(data[[loc[2]]]),
                        as.vector(data[[loc[3]]])
                    )          
                } else {
                    loc_df <- cbind(
                        as.vector(data[[loc[1]]]),
                        as.vector(data[[loc[2]]])
                    )
                }
            } else {
                if(model$mesh$manifold == "S2"){
                    loc_df <- cbind(
                        as.vector(data[[loc[1]]]),
                        as.vector(data[[loc[2]]]),
                        as.vector(data[[loc[3]]])
                    )          
                } else {
                    loc_df <- cbind(
                        as.vector(data[[loc[1]]]),
                        as.vector(data[[loc[2]]])
                    )        
                }
            }
        } 
    } else {
        if (length(loc) != 2) {
            stop("For a metric graph, 'loc' must have length two.")
        }
        loc_df <- cbind(
          as.vector(data[[loc[1]]]),
          as.vector(data[[loc[2]]])
        )
      }
    } else {
      stop("loc must be either a matrix, a data.frame or a character vector with the names of the columns of the observation locations.")
    }
    
    if(spacetime) {
        if(is.null(loc_time)) {
            stop("loc_time should be supplied for spatio-temporal models.")
        }
        if(is.matrix(loc_time)) {
            if(min(dim(loc_time)) > 1) {
                stop("loc_time should be one-dimensional.")
            }
            time_df <- loc_time
        } else if(is.character(loc_time)) {
            time_df <- data[[loc_time]]
        } else if(is.vector(loc_time)) {
            time_df <- as.matrix(loc_time)
        } else {
            stop("loc_time should be a matrix, vector or character vector.")
        }
    }


    repl_val <- unique(repl)
    A_list <- list()
    
    loc_df <- loc_df[idx_repl, , drop = FALSE]
    if(spacetime) {
        time_df <- time_df[idx_repl]    
    }
    
    if (!is.null(model$make_A) && !spacetime) {
      for (j in repl_val) {
        ind_tmp <- (repl %in% j)
        y_tmp <- y_resp[ind_tmp]
        na_obs <- is.na(y_tmp)
        A_list[[as.character(j)]] <- model$make_A(loc_df[ind_tmp, , drop = FALSE])
        A_list[[as.character(j)]] <- A_list[[as.character(j)]][!na_obs, , drop = FALSE]

        if (inherits(model, "CBrSPDEobj")) {
          alpha <- NULL
          if(!is.null(model_options$fix_alpha)){
            alpha <- model_options$fix_alpha
          }
          if(!is.null(model_options$fix_nu)){
            alpha <- model_options$fix_nu + model$d / 2
          }
          if (!is.null(alpha)) {
            if (alpha %% 1 != 0) {
              A_list[[as.character(j)]] <- kronecker(matrix(1, 1, model$m + 1), A_list[[as.character(j)]])
            }
          } else {
            A_list[[as.character(j)]] <- kronecker(matrix(1, 1, model$m + 1), A_list[[as.character(j)]])
          }
        }
      }
    } else if(spacetime) {
        for (j in repl_val) {
            ind_tmp <- (repl %in% j)
            y_tmp <- y_resp[ind_tmp]
            na_obs <- is.na(y_tmp)
            A_list[[as.character(j)]] <- model$make_A(loc = loc_df[ind_tmp, , drop = FALSE], 
                                                      time = time_df[ind_tmp])
            A_list[[as.character(j)]] <- A_list[[as.character(j)]][!na_obs, , drop = FALSE]
        }
    } else if (!inherits(model, "rSPDEobj1d")){
        stop("When creating the model object using matern.operators() or spde.matern.operators(), you should either supply a graph, or a mesh, or mesh_loc (this last one only works for dimension 1).")
    }  

    model_tmp <- model
    model_tmp$graph <- NULL
    if(inherits(model, "CBrSPDEobj") || inherits(model, "rSPDEobj")) {
        model_tmp$mesh <- NULL
        model_tmp$make_A <- NULL    
    }

    like_aux <- create_likelihood(model = model, model_options = model_options, y_resp = y_resp, X_cov = X_cov, A_list = A_list, 
                                    repl = repl, start_values = start_values, mean_correction = mean_correction, 
                                    smoothness_upper_bound = smoothness_upper_bound,
                                    loc_df = loc_df)

    likelihood <- like_aux$likelihood
    estimate_pars <- like_aux$estimate_params
    n_coeff_nonfixed <- like_aux$n_coeff_nonfixed

    if (ncol(X_cov) > 0 && !is.null(model)) {
      names_tmp <- colnames(X_cov)
      data_tmp <- cbind(y_resp, X_cov)
      data_tmp <- na.omit(data_tmp)
      temp_coeff <- lm(data_tmp[, 1] ~ data_tmp[, -1] - 1)$coeff
      names(temp_coeff) <- names_tmp
      start_values_aux <- start_values[estimate_pars]
      start_values_aux <- c(start_values_aux, temp_coeff)
      rm(data_tmp)
    } else{
      start_values_aux <- start_values[estimate_pars]
    }

    time_build_likelihood_end <- Sys.time()

    time_build_likelihood <- time_build_likelihood_end - time_build_likelihood_start

    hessian <- TRUE

    if (improve_hessian) {
      hessian <- FALSE
    }

    time_par <- NULL

    likelihood_new <- function(theta) {
      l_tmp <- tryCatch(likelihood(theta),
        error = function(e) {
          return(NULL)
        }
      )
      if (is.null(l_tmp)) {
        return(10^100)
      }
      return(l_tmp)
    }

    if(length(start_values_aux) > 0){

      if (parallel) {
        start_par <- Sys.time()
        n_cores_lim <- Sys.getenv("_R_CHECK_LIMIT_CORES_", "")
  
        if (nzchar(n_cores_lim) && n_cores_lim == "TRUE") {
          n_cores <- 2L
        }
        cl <- parallel::makeCluster(n_cores)
        parallel::setDefaultCluster(cl = cl)
  
        parallel::clusterExport(cl, "y_resp", envir = environment())
        parallel::clusterExport(cl, "model_tmp", envir = environment())
        parallel::clusterExport(cl, "A_list", envir = environment())
        parallel::clusterExport(cl, "X_cov", envir = environment())
        if (exists("loc_df")) {
          parallel::clusterExport(cl, "loc_df", envir = environment())
        }
        if (exists("mean_correction")) {
          parallel::clusterExport(cl, "mean_correction", envir = environment())
        }
        if (exists("repl")) {
          parallel::clusterExport(cl, "repl", envir = environment())
        }
        
        # Export auxiliary likelihood functions from rSPDE namespace
        parallel::clusterExport(cl, "aux_lme_CBrSPDE.matern.loglike",
          envir = as.environment(asNamespace("rSPDE"))
        )
        parallel::clusterExport(cl, "aux_lme_CBrSPDE.matern2d.loglike",
          envir = as.environment(asNamespace("rSPDE"))
        )
        parallel::clusterExport(cl, "aux_lme_rSPDE.matern.loglike",
          envir = as.environment(asNamespace("rSPDE"))
        )
        parallel::clusterExport(cl, "aux_lme_rSPDE.matern.rational.loglike",
          envir = as.environment(asNamespace("rSPDE"))
        )
        parallel::clusterExport(cl, "aux_lme_spacetime.loglike",
          envir = as.environment(asNamespace("rSPDE"))
        )
        parallel::clusterExport(cl, "aux_lme_intrinsic.loglike",
          envir = as.environment(asNamespace("rSPDE"))
        )
        
        # Export helper functions
        parallel::clusterExport(cl, "get_aux_likelihood_function",
          envir = environment()
        )
        parallel::clusterExport(cl, "determine_estimate_params",
          envir = environment()
        )
        parallel::clusterExport(cl, "extract_model_update_args",
          envir = environment()
        )
        parallel::clusterExport(cl, "get_aux_lik_fun_args",
          envir = environment()
        )
        parallel::clusterExport(cl, "create_likelihood",
          envir = environment()
        )
        
        # If theta2beta is used in the code
        if (inherits(model_tmp, "intrinsicCBrSPDEobj")) {
          parallel::clusterExport(cl, "theta2beta",
            envir = as.environment(asNamespace("rSPDE"))
          )
        }
        
        # Export model_options 
        if (exists("model_options")) {
          parallel::clusterExport(cl, "model_options", envir = environment())
        }
        
        # Export start_values 
        if (exists("start_values")) {
          parallel::clusterExport(cl, "start_values", envir = environment())
        }
        
        # Export smoothness_upper_bound
        if (exists("smoothness_upper_bound")) {
          parallel::clusterExport(cl, "smoothness_upper_bound", envir = environment())
        }
  
        end_par <- Sys.time()
        time_par <- end_par - start_par
  
        start_fit <- Sys.time()
        res <- optimParallel::optimParallel(start_values_aux,
          likelihood_new,
          method = optim_method,
          control = optim_controls,
          hessian = hessian,
          parallel = list(
            forward = FALSE, cl = cl,
            loginfo = FALSE
          )
        )
        end_fit <- Sys.time()
        time_fit <- end_fit - start_fit
        parallel::stopCluster(cl)
  
        time_hessian <- NULL
  
        if (!is.na(res[1])) {
          if (!improve_hessian) {
            observed_fisher <- res$hessian
          } else {
            if (!is.list(hessian_args)) {
              stop("hessian_controls must be a list")
            }
  
            start_hessian <- Sys.time()
            observed_fisher <- numDeriv::hessian(likelihood_new, res$par,
              method.args = hessian_args
            )
            end_hessian <- Sys.time()
            time_hessian <- end_hessian - start_hessian
          }
          if(nrow(observed_fisher) > 0) {
              eig_hes <- eigen(observed_fisher)$value
              cond_pos_hes <- (min(eig_hes) > 1e-15)
          } else{
            eig_hes <- 1
            cond_pos_hes <- TRUE
          }
  
        } else {
          stop("Could not fit the model. Please, try another method with 'parallel' set to FALSE.")
        }
        if (min(eig_hes) < 1e-15) {
          warning("The optimization failed to provide a numerically positive-definite Hessian. You can try to obtain a positive-definite Hessian by setting 'improve_hessian' to TRUE or by setting 'parallel' to FALSE, which allows other optimization methods to be used.")
        }
      } else {
      
        start_fit <- Sys.time()
        res <- withCallingHandlers(
          tryCatch(optim(start_values_aux,
            likelihood_new,
            method = optim_method,
            control = optim_controls,
            hessian = hessian
          ), error = function(e) {
            return(NA)
          }),
          warning = function(w) {
            invokeRestart("muffleWarning")
          }
        )
        end_fit <- Sys.time()
        time_fit <- end_fit - start_fit
  
        cond_pos_hes <- FALSE
        time_hessian <- NULL
  
        if (!is.na(res[1])) {
          if (!improve_hessian) {
            observed_fisher <- res$hessian
          } else {
            if (!is.list(hessian_args)) {
              stop("hessian_controls must be a list")
            }
  
            start_hessian <- Sys.time()
            observed_fisher <- numDeriv::hessian(likelihood_new, res$par,
              method.args = hessian_args
            )
            end_hessian <- Sys.time()
            time_hessian <- end_hessian - start_hessian
          }
          if(nrow(observed_fisher) > 0) {
            eig_hes <- eigen(observed_fisher)$value
            cond_pos_hes <- (min(eig_hes) > 1e-15)
          } else{
            eig_hes <- 1
            cond_pos_hes <- TRUE
          }
        }
  
        problem_optim <- list()
  
        if (is.na(res[1]) || !cond_pos_hes) {
          problem_optim[[optim_method]] <- list()
          if (is.na(res[1])) {
            problem_optim[[optim_method]][["lik"]] <- NA
            problem_optim[[optim_method]][["res"]] <- res
            problem_optim[[optim_method]][["hess"]] <- NA
            problem_optim[[optim_method]][["time_hessian"]] <- NA
            problem_optim[[optim_method]][["time_fit"]] <- NA
          } else {
            problem_optim[[optim_method]][["lik"]] <- -res$value
            problem_optim[[optim_method]][["res"]] <- res
            problem_optim[[optim_method]][["hess"]] <- observed_fisher
            problem_optim[[optim_method]][["time_hessian"]] <- time_hessian
            problem_optim[[optim_method]][["time_fit"]] <- time_fit
          }
        }
  
        ok_optim <- FALSE
        orig_optim_method <- optim_method
        test_other_optim <- (is.na(res[1]) || !cond_pos_hes)
  
        if (test_other_optim) {
          tmp_method <- optim_method
          while (length(possible_methods) > 1) {
            possible_methods <- setdiff(possible_methods, tmp_method)
            new_method <- possible_methods[1]
            time_fit <- NULL
            start_fit <- Sys.time()
            res <- withCallingHandlers(
              tryCatch(optim(start_values_aux,
                likelihood_new,
                method = new_method,
                control = optim_controls,
                hessian = hessian
              ), error = function(e) {
                return(NA)
              }),
              warning = function(w) {
                invokeRestart("muffleWarning")
              }
            )
            end_fit <- Sys.time()
            time_fit <- end_fit - start_fit
            tmp_method <- new_method
            cond_pos_hes <- FALSE
            if (is.na(res[1])) {
              problem_optim[[tmp_method]][["lik"]] <- NA
              problem_optim[[tmp_method]][["res"]] <- res
              problem_optim[[tmp_method]][["hess"]] <- NA
              problem_optim[[tmp_method]][["time_hessian"]] <- NA
              problem_optim[[tmp_method]][["time_fit"]] <- NA
            } else {
              if (!improve_hessian) {
                observed_fisher <- res$hessian
              } else {
                if (!is.list(hessian_args)) {
                  stop("hessian_controls must be a list")
                }
  
                start_hessian <- Sys.time()
                observed_fisher <- numDeriv::hessian(likelihood_new, res$par,
                  method.args = hessian_args
                )
                end_hessian <- Sys.time()
                time_hessian <- end_hessian - start_hessian
              }
              if(nrow(observed_fisher) > 0) {
                eig_hes <- eigen(observed_fisher)$value
                cond_pos_hes <- (min(eig_hes) > 1e-15)
              } else{
                eig_hes <- 1
                cond_pos_hes <- TRUE
              }
  
              problem_optim[[tmp_method]][["lik"]] <- -res$value
              problem_optim[[tmp_method]][["res"]] <- res
              problem_optim[[tmp_method]][["hess"]] <- observed_fisher
              problem_optim[[tmp_method]][["time_hessian"]] <- time_hessian
              problem_optim[[tmp_method]][["time_fit"]] <- time_fit
            }
  
            cond_ok <- ((!is.na(res[1])) && cond_pos_hes)
            if (cond_ok) {
              optim_method <- new_method
              ok_optim <- TRUE
              break
            }
          }
        }
  
        if (test_other_optim) {
          lik_val <- lapply(problem_optim, function(dat) {
            dat[["lik"]]
          })
          if (all(is.na(lik_val))) {
            stop("The model could not be fitted. All optimizations method failed.")
          } else if (ok_optim) {
            warning(paste("optim method", orig_optim_method, "failed to provide a positive-definite Hessian. Another optimization method was used."))
            rm(problem_optim)
          } else {
            max_method <- which.max(lik_val)
            res <- problem_optim[[max_method]][["res"]]
            observed_fisher <- problem_optim[[max_method]][["hess"]]
            time_hessian <- problem_optim[[max_method]][["time_hessian"]]
            time_fit <- problem_optim[[max_method]][["time_fit"]]
            warning("All optimization methods failed to provide a numerically positive-definite Hessian. The optimization method with largest likelihood was chosen. You can try to obtain a positive-definite Hessian by setting 'improve_hessian' to TRUE.")
          }
        }
      }
    loglik <- -res$value
    } else {
      res <- list(counts = 0)
      observed_fisher <- NULL
      time_hessian <- NULL
      time_fit <- as.difftime(0, units="secs") 
      loglik <- - likelihood_new(start_values_aux)
    }

    coeff_results <- process_model_results(res = res, observed_fisher = observed_fisher, 
                                                      start_values = start_values, estimate_params = estimate_pars,
                                                      model = model,
                                                      model_options = model_options, X_cov = X_cov, n_coeff_nonfixed = n_coeff_nonfixed)

    
    coeff_meas <- coeff_results$coeff_meas
    coeff_random <- coeff_results$coeff_random
    coeff_fixed <- coeff_results$coeff_fixed      
    std_meas <- coeff_results$std_meas
    std_random <- coeff_results$std_random
    std_fixed <- coeff_results$std_fixed
    observed_fisher <- coeff_results$observed_fisher

    new_likelihood <- NULL

    start_time_alt_par <- Sys.time()
    coeff_alt_par_result <- convert_parameterization_matern_spde(model, parameterization, coeff_random, model_options, observed_fisher, estimate_pars, std_random)
    time_alt_par <- Sys.time() - start_time_alt_par

  } else { # If model is NULL
    coeff_random <- NULL
    time_matern_par <- NULL
    std_random <- NULL
    time_build_likelihood <- NULL
    start_values <- NULL
    matern_coeff <- NULL
    time_fit <- NULL
    time_hessian <- NULL
    time_par <- NULL
    A_list <- NULL

    if (ncol(X_cov) == 0) {
      stop("The model does not have either random nor fixed effects.")
    }

    names_tmp <- colnames(X_cov)
    data_tmp <- cbind(y_resp, X_cov)
    data_tmp <- na.omit(data_tmp)
    res <- lm(data_tmp[, 1] ~ data_tmp[, -1] - 1)
    coeff_fixed <- res$coeff
    names(coeff_fixed) <- names_tmp
    sm_temp <- summary(res)
    std_fixed <- sm_temp$coefficients
    rownames(std_fixed) <- names_tmp
    coeff_meas <- sm_temp$sigma
    names(coeff_meas) <- "std. dev"
    std_meas <- NULL
    loglik <- logLik(res)[[1]]
  }

  if (is.null(coeff_fixed) && is.null(coeff_random)) {
    stop("The model does not have either random nor fixed effects.")
  } 

  object <- list()
  object$coeff <- list(
    measurement_error = coeff_meas,
    fixed_effects = coeff_fixed, 
    random_effects = coeff_random
  )

  object$std_errors <- list(
    std_meas = std_meas,
    std_fixed = std_fixed, std_random = std_random
  )
  object$call <- call_rspde_lme
  object$terms <- list(fixed_effects = X_cov)
  object$response_data <- list(y = y_resp)
  object$formula <- formula
  object$alt_par_coeff <- coeff_alt_par_result
  object$estimation_method <- optim_method
  object$parameterization_latent <- parameterization
  object$repl <- repl
  object$idx_repl <- idx_repl
  object$optim_controls <- optim_controls
  object$latent_model <- model
  object$nobs <- sum(idx_repl)
  object$null_model <- null_model
  object$start_values <- start_values
  object$loglik <- loglik
  object$niter <- res$counts
  object$response_var <- y_term
  object$covariates <- cov_term
  object$fitting_time <- time_fit
  object$rspde_order <- rspde_order
  object$time_alt_par <- time_alt_par
  object$improve_hessian <- improve_hessian
  object$time_hessian <- time_hessian
  object$parallel <- parallel
  object$time_par <- time_par
  object$time_data <- time_data
  object$optim_method <- optim_method
  object$time_likelihood <- time_build_likelihood
  object$A_list <- A_list
  object$has_graph <- model$has_graph
  object$which_repl <- which_repl
  object$stationary <- model$stationary
  
  object$start_val_lik <- start_values_aux

  object$df.residual <- object$nobs - (1 + length(object$coeff$fixed_effects) + length(object$coeff$random_effects))
  object$lik_fun <- likelihood_new
  object$mle_par_orig <- res$par
  object$loc <- loc_df
  object$spacetime <- spacetime
  object$anisotropic <- anisotropic
  object$intrinsic <- intrinsic
  object$mean_correction <- mean_correction
  if(spacetime) { 
      object$time <- time_df
  }

  if (ncol(X_cov) > 0) {
    object$model_matrix <- cbind(y_resp, X_cov)
  } else {
    object$model_matrix <- y_resp
  }

  class(object) <- "rspde_lme"
  return(object)
}


#' @name print.rspde_lme
#' @title Print Method for \code{rspde_lme} Objects
#' @description Provides a brief description of results related to mixed effects with Whittle-Matern latent models.
#' @param x object of class "rspde_lme" containing results from the fitted model.
#' @param ... further arguments passed to or from other methods.
#' @return Called for its side effects.
#' @noRd
#' @method print rspde_lme
#' @export

print.rspde_lme <- function(x, ...) {
  #
    if (!is.null(x$latent_model)) {
        if(x$intrinsic) {
            call_name <- "Latent model - Intrinsic Whittle-Matern"
        } else if (x$latent_model$stationary && !x$spacetime && !x$anisotropic) {
            call_name <- "Latent model - Whittle-Matern"
        } else if (x$anisotropic) {
            call_name <- "Latent model - Anisotropic Whittle-Matern"
        } else if (x$spacetime) {
            call_name <- "Latent model - Spatio-temporal"
        } else {
        call_name <- "Latent model - Generalized Whittle-Matern"
        }
    } else {
        call_name <- "Linear regression model"
    }

    coeff_fixed <- x$coeff$fixed_effects
    coeff_random <- x$coeff$random_effects

    cat("\n")
    cat(call_name)
    if(x$spacetime) {
        cat(" with alpha = ", x$latent_model$alpha, ", beta = ", x$latent_model$beta)
    }
    cat("\n\n")
    cat("Call:\n", paste(deparse(x$call), sep = "\n", collapse = "\n"),
        "\n\n", sep = "")
    cat(paste0("Fixed effects:", "\n"))
    if (!is.null(coeff_fixed)) {
        print(coeff_fixed)
    } else {
        message("No fixed effects")
    }
    cat("\n")
    cat(paste0("Random effects:", "\n"))
    if (!is.null(coeff_random)) {
        print(coeff_random)
        if (x$stationary && !x$spacetime && !x$anisotropic && !x$intrinsic) {
            if (x$parameterization_latent == "spde") {
                cat(paste0("\n", "Random effects (Matern parameterization):", "\n"))
            } else {
                cat(paste0("\n", "Random effects (SPDE parameterization):", "\n"))
            }
            print(x$alt_par_coeff$coeff)
        }
    } else {
        message("No random effects")
    }
    cat("\n")
    cat(paste0("Measurement error:", "\n"))
    print(x$coeff$measurement_error)
}


#' @name summary.rspde_lme
#' @title Summary Method for \code{rspde_lme} Objects.
#' @description Function providing a summary of results related to mixed effects regression models with Whittle-Matern latent models.
#' @param object an object of class "rspde_lme" containing results from the fitted model.
#' @param all_times Show all computed times.
#' @param ... not used.
#' @return An object of class \code{summary_rspde_lme} containing several
#' informations of a *rspde_lme* object.
#' @method summary rspde_lme
#' @export
summary.rspde_lme <- function(object, all_times = FALSE, ...) {
  ans <- list()

  nfixed <- length(object$coeff$fixed_effects)
  nrandom <- length(object$coeff$random_effects)
  model_type <- !object$null_model
  if (model_type) {
    if(object$intrinsic) {
        call_name <- "Latent model - Intrinsic Whittle-Matern"
    } else if (object$latent_model$stationary && !object$spacetime && !object$anisotropic) {
      call_name <- "Latent model - Whittle-Matern"
    } else if(object$anisotropic) {
        call_name <- "Latent model - Anisotropic Whittle-Matern"
    } else if(object$spacetime) {
        call_name <- "Latent model - Spatio-temporal"
    } else {
      call_name <- "Latent model - Generalized Whittle-Matern"
    }
  } else {
    call_name <- "Linear regression model"
  }

  coeff_fixed <- object$coeff$fixed_effects
  coeff_random <- object$coeff$random_effects 
  coeff_meas <- object$coeff$measurement_error

  SEr_fixed <- object$std_errors$std_fixed
  SEr_random <- object$std_errors$std_random
  SEr_meas <- object$std_errors$std_meas

  if (object$stationary && !object$spacetime && !object$anisotropic && !object$intrinsic) {
    coeff <- c(coeff_fixed, coeff_random, object$alt_par_coeff$coeff, coeff_meas)
    SEr <- c(SEr_fixed, SEr_random, object$alt_par_coeff$std_random, SEr_meas)
  } else {
    coeff <- c(coeff_fixed, coeff_random, coeff_meas)
    SEr <- c(SEr_fixed, SEr_random, SEr_meas)
  }

  if (model_type) {
    tab <- cbind(coeff, SEr, coeff / SEr, 2 * stats::pnorm(-abs(coeff / SEr)))
    colnames(tab) <- c("Estimate", "Std.error", "z-value", "Pr(>|z|)")
    rownames(tab) <- names(coeff)
    if (object$stationary && !object$spacetime && !object$anisotropic && !object$intrinsic) {
      tab <- list(
        fixed_effects = tab[seq.int(length.out = nfixed), , drop = FALSE], 
        random_effects = tab[seq.int(length.out = nrandom) + nfixed, , drop = FALSE],
        random_effects_alt_par = tab[seq.int(length.out = nrandom) + nrandom + nfixed, , drop = FALSE],
        meas_error = tab[seq.int(length.out = 1) + nfixed + 2 * nrandom, , drop = FALSE]
      )
    } else {
      tab <- list(
        fixed_effects = tab[seq.int(length.out = nfixed), , drop = FALSE], 
        random_effects = tab[seq.int(length.out = nrandom) + nfixed, , drop = FALSE],
        meas_error = tab[seq.int(length.out = 1) + nfixed + nrandom, , drop = FALSE]
      )
    }
  } else {
    tab <- list(fixed_effects = SEr_fixed, coeff_meas = coeff_meas)
  }

  ans$coefficients <- tab

  ans$model_type <- model_type

  ans$call_name <- call_name

  ans$call <- object$call

  ans$loglik <- object$loglik

  ans$optim_method <- object$optim_method

  ans$niter <- object$niter

  if(object$spacetime) {
      ans$alpha <- object$latent_model$alpha
      ans$beta <- object$latent_model$beta
  }
  
  ans$spacetime <- object$spacetime
  ans$anisotropic <- object$anisotropic
  ans$intrinsic <- object$intrinsic
  ans$all_times <- all_times
  ans$fitting_time <- object$fitting_time
  ans$improve_hessian <- object$improve_hessian
  ans$time_hessian <- object$time_hessian
  ans$parallel <- object$parallel
  ans$parameterization_latent <- object$parameterization_latent
  ans$time_par <- object$time_par
  ans$time_data <- object$time_data
  ans$time_alt_par <- object$time_alt_par
  ans$time_likelihood <- object$time_likelihood
  class(ans) <- "summary_rspde_lme"
  ans
}

#' @name print.summary_rspde_lme
#' @title Print Method for \code{summary_rspde_lme} Objects
#' @description Provides a brief description of results related to mixed effects regression models with Whittle-Matern latent models.
#' @param x object of class "summary_rspde_lme" containing results of summary method applied to a fitted model.
#' @param ... further arguments passed to or from other methods.
#' @return Called for its side effects.
#' @noRd
#' @method print summary_rspde_lme
#' @export
print.summary_rspde_lme <- function(x, ...) {
  tab <- x$coefficients

  #
  digits <- max(3, getOption("digits") - 3)
  #

  call_name <- x$call_name

  cat("\n")
  cat(call_name)
  
  if(x$spacetime) { 
      cat(" with alpha = ", x$alpha, ", beta = ", x$beta)
  }  
  
  cat("\n\n")
  cat("Call:\n", paste(deparse(x$call), sep = "\n", collapse = "\n"),
    "\n",
    sep = ""
  )


  #
  model_type <- tolower(x$model_type)
  #
  if (model_type) {
    if (NROW(tab$fixed_effects)) {
      cat(paste0("\nFixed effects:\n"))
      stats::printCoefmat(tab[["fixed_effects"]], digits = digits, signif.legend = FALSE)
    } else {
      message("\nNo fixed effects. \n")
    }
    #
    if (NROW(tab$random_effects)) {
      cat(paste0("\nRandom effects:\n"))
      stats::printCoefmat(tab[["random_effects"]][, 1:3, drop = FALSE], digits = digits, signif.legend = FALSE)
    } else {
      cat(paste0("\nRandom effects:\n"))
      message("No random effects. \n")
    }
    if (NROW(tab$random_effects_alt_par)) {
      if (x$parameterization_latent == "spde") {
        cat(paste0("\nRandom effects (Matern parameterization):\n"))
      } else {
        cat(paste0("\nRandom effects (SPDE parameterization):\n"))
      }
      stats::printCoefmat(tab[["random_effects_alt_par"]][, 1:3], digits = digits, signif.legend = FALSE)
    }
    #
    cat(paste0("\nMeasurement error:\n"))
    stats::printCoefmat(tab[["meas_error"]][1, 1:3, drop = FALSE], digits = digits, signif.legend = FALSE)
  } else {
    cat(paste0("\nFixed effects:\n"))
    stats::printCoefmat(tab[["fixed_effects"]], digits = digits, signif.legend = FALSE)

    cat(paste0("\nRandom effects:\n"))
    message("No random effects. \n")
    cat(paste0("\nMeasurement error:\n"))
    print(tab$coeff_meas)
  }
  #
  if (getOption("show.signif.stars")) {
    cat("---\nSignif. codes: ", "0 '***' 0.001 '**' 0.01 '*' 0.05 '.' 0.1 ' ' 1", "\n\n")
  }
  #

  cat("Log-Likelihood: ", x$loglik, "\n")
  if (model_type) {
    cat(paste0("Number of function calls by 'optim' = ", x$niter[1], "\n"))
    cat(paste0("Optimization method used in 'optim' = ", x$optim_method, "\n"))
    cat(paste0("\nTime used to:"))
    if (x$all_times) {
      cat("\t prepare the data = ", paste(trunc(x$time_data[[1]] * 10^5) / 10^5, attr(x$time_data, "units"), "\n"))
      cat("\t build the likelihood = ", paste(trunc(x$time_likelihood[[1]] * 10^5) / 10^5, attr(x$time_likelihood, "units"), "\n"))
      cat("\t compute alternative parameterization = ", paste(trunc(x$time_alt_par[[1]] * 10^5) / 10^5, attr(x$time_likelihood, "units"), "\n"))
    }
    cat("\t fit the model = ", paste(trunc(x$fitting_time[[1]] * 10^5) / 10^5, attr(x$fitting_time, "units"), "\n"))
    if (x$improve_hessian) {
      cat(paste0("\t compute the Hessian = ", paste(trunc(x$time_hessian[[1]] * 10^5) / 10^5, attr(x$time_hessian, "units"), "\n")))
    }
    if (x$parallel) {
      cat(paste0("\t set up the parallelization = ", paste(trunc(x$time_par[[1]] * 10^5) / 10^5, attr(x$time_par, "units"), "\n")))
    }
  }
}



#' @name predict.rspde_lme
#' @title Prediction of a mixed effects regression model on a metric graph.
#' @param object The fitted object with the `rspde_lme()` function
#' @param newdata A `data.frame` or a `list` containing the covariates, the edge 
#' number and the distance on edge for the locations to obtain the prediction.
#' @param loc Prediction locations. Can either be a `data.frame`, a `matrix` or 
#' a character vector, that contains the names of the columns of the coordinates 
#' of the locations. For models using `metric_graph` objects, plase use 
#' `edge_number` and `distance_on_edge` instead.
#' @param time Prediction times for spatio-temporal models. 
#' @param mesh Obtain predictions for mesh nodes? The graph must have a mesh, and 
#' either `only_latent` is set to TRUE or the model does not have covariates.
# @param repl Which replicates to obtain the prediction. If `NULL` predictions will be obtained for all replicates. Default is `NULL`.
#' @param which_repl Which replicates to use? If `NULL` all replicates will be used.
#' @param compute_variances Set to also TRUE to compute the kriging variances.
#' @param posterior_samples If `TRUE`, posterior samples will be returned.
#' @param n_samples Number of samples to be returned. Will only be used if 
#' `sampling` is `TRUE`.
#' @param sample_latent Do posterior samples only for the random effects?
#' @param return_as_list Should the means of the predictions and the posterior 
#' samples be returned as a list, with each replicate being an element?
#' @param return_original_order Should the results be return in the original 
#' (input) order or in the order inside the graph?
#' @param ... Additional arguments. Expert use only. 
#' @param data `r lifecycle::badge("deprecated")` Use `newdata` instead.
#' @return A list with elements `mean`, which contains the means of the
#' predictions, `fe_mean`, which is the prediction for the fixed effects, 
#' `re_mean`, which is the prediction for the random effects, `variance` (if 
#' `compute_variance` is `TRUE`), which contains the variances of the predictions, 
#' `samples` (if `posterior_samples` is `TRUE`), which contains the posterior samples.
#' @export
#' @method predict rspde_lme

predict.rspde_lme <- function(object, 
                              newdata = NULL, 
                              loc = NULL, 
                              time = NULL, 
                              mesh = FALSE, 
                              which_repl = NULL, 
                              compute_variances = FALSE, 
                              posterior_samples = FALSE,
                              n_samples = 100, 
                              sample_latent = FALSE, 
                              return_as_list = FALSE, 
                              return_original_order = TRUE,
                              ...,
                              data = lifecycle::deprecated()) {
  if (lifecycle::is_present(data)) {
    if (is.null(newdata)) {
      lifecycle::deprecate_warn("2.3.3", "predict(data)", "predict(newdata)",
        details = c("`data` was provided but not `newdata`. Setting `newdata <- data`.")
      )
      newdata <- data
    } else {
      lifecycle::deprecate_warn("2.3.3", "predict(data)", "predict(newdata)",
        details = c("Both `newdata` and `data` were provided. Only `newdata` will be considered.")
      )
    }
    data <- NULL
  }

  data <- newdata

  if (is.null(data)) {
    if (!mesh) {
      stop("If 'mesh' is false, you should supply data!")
    }
  }

  tmp_args <- list(...)

  distance_on_edge <- tmp_args[["distance_on_edge"]]
  edge_number <- tmp_args[["edge_number"]]
  normalized <- tmp_args[["normalized"]]

  if(is.null(distance_on_edge)){
    distance_on_edge <- "distance_on_edge"
  }
  if(is.null(edge_number)){
    edge_number <- "edge_number"
  }
  if(is.null(normalized)){
    normalized <- FALSE
  }

  out <- list()

  coeff_fixed <- object$coeff$fixed_effects
  coeff_random <- object$coeff$random_effects
  coeff_meas <- object$coeff$measurement_error

  if (object$has_graph) {
    loc <- cbind(data[[edge_number]], data[[distance_on_edge]])
  } else if (is.character(loc)) {
    loc <- data[loc]
  }

  loc <- as.matrix(loc)
  
  if(object$spacetime) { 
    if(is.character(time)) {
        time <- data[time]
    }    
    time <- as.matrix(time)
  }

  X_cov_initial <- as.matrix(object$model_matrix)[, -1, drop = FALSE]
  if (ncol(X_cov_initial) > 0) {
    if (mesh) {
      stop("In the presence of covariates, you should provide the data, including the covariates at the prediction locations. If you only want predictions for the latent model, set 'only_latent' to TRUE.")
    }
  }

  X_cov_pred <- stats::model.matrix(object$covariates, data)
  if (nrow(X_cov_pred) > 0 && nrow(X_cov_pred) != nrow(as.matrix(loc))) {
    stop("Covariates not found in data.")
  }

  if (sum(duplicated(loc)) > 0 && !object$spacetime) {
    warning("There are duplicated locations for prediction, we will try to process the data to extract the unique locations,
    along with the corresponding covariates.")
    if (nrow(X_cov_pred) == nrow(loc)) {
      data_tmp <- cbind(loc, X_cov_pred)
    }
    data_tmp <- unique(data_tmp)
    if (sum(duplicated(cbind(data_tmp[, 1:ncol(loc)]))) > 0) {
      stop("Data processing failed, please provide a data with unique locations.")
    }
  }


  if(!inherits(object$latent_model, "rSPDEobj1d")) {
      if (!mesh) {
          n_prd <- length(loc[, 1])
          # Convert data to normalized
          if (object$has_graph && !normalized) {
              loc[, 2] <- loc[, 2] / object$graph$edge_lengths[loc[, 1]]
          }
          if(inherits(object$latent_model, "spacetimeobj")) {
              if(length(time) != n_prd) {
                  stop("loc and time should have the same length.")
              }
              Aprd <- object$latent_model$make_A(loc = loc, time = time)    
          } else {
              Aprd <- object$latent_model$make_A(loc)
          }
          
      } else {
          Aprd <- Matrix::Diagonal(nrow(object$latent_model$C))
          if(inherits(object$latent_model, "intrinsicCBrSPDEobj")) {
              # Extract alpha or beta from random effects
              alpha_param <- grep("^alpha", names(object$coeff$random_effects), value = TRUE)
              beta_param <- grep("^beta", names(object$coeff$random_effects), value = TRUE)
              
              if (length(alpha_param) > 0) {
                alpha <- object$coeff$random_effects[[alpha_param]]
              } else {
                stop("alpha parameter not found in random effects")
              }
              
              if (length(beta_param) > 0) {
                beta <- object$coeff$random_effects[[beta_param]]
              } else {
                stop("beta parameter not found in random effects")
              }
              
              # Apply kronecker product only if alpha or beta is non-integer
              if (alpha %% 1 != 0 || beta %% 1 != 0) {
                Aprd <- kronecker(matrix(1, 1, object$m), Aprd)
              }
          } 
      }    
  } else {
      n_prd <- length(loc[, 1])
  }
  
  ##
  repl_vec <- object$repl
  if (is.null(which_repl)) {
    u_repl <- unique(repl_vec)
  } else {
    u_repl <- unique(which_repl)
  }

  ##

  if (all(dim(X_cov_pred) == c(0, 1))) {
    X_cov_pred <- matrix(1, nrow = nrow(loc), ncol = 1)
  }
  if (ncol(X_cov_pred) > 0) {
    mu_prd <- X_cov_pred %*% coeff_fixed
  } else {
    mu_prd <- matrix(0, nrow = nrow(loc), ncol = 1)
  }

  model_matrix_fit <- object$model_matrix

  model_matrix_fit <- as.matrix(model_matrix_fit)

  if (ncol(model_matrix_fit) > 1) {
    X_cov_fit <- model_matrix_fit[, 2:ncol(model_matrix_fit), drop = FALSE]
    mu <- X_cov_fit %*% coeff_fixed
  } else {
    mu <- 0
  }

  Y <- model_matrix_fit[, 1] - mu

  model_type <- object$latent_model

  sigma.e <- coeff_meas[[1]]
  sigma_e <- sigma.e

  ## construct Q
  # Get parameters from object$coeff$random_effects or object$alt_par_coeff$coeff
  # based on model type and parameterization
  
  # Create a list of parameters to pass to update
  update_params <- list()
  
  # Add parameters based on what's available in coeff_random
  for (param_name in names(coeff_random)) {
    param_name_cleaned <- gsub(" \\(fixed\\)$", "", param_name)
    update_params[[param_name_cleaned]] <- coeff_random[[param_name]]
  }
  
  # # If alt_par_coeff exists, also consider those parameters
  # if (!is.null(object$alt_par_coeff) && !is.null(object$alt_par_coeff$coeff)) {
  #   for (param_name in names(object$alt_par_coeff$coeff)) {
  #     if (!(param_name %in% names(update_params))) {
  #       update_params[[param_name]] <- object$alt_par_coeff$coeff[[param_name]]
  #     }
  #   }
  # }
  
  # Update the model with all available parameters
  new_rspde_obj <- do.call(update, c(list(object$latent_model), update_params))
  
  if(object$mean_correction) {
      mu_corr <- new_rspde_obj$mean_correction(full=TRUE)
  } else {
      mu_corr <- 0
  }
  
  idx_obs_full <- as.vector(!is.na(Y))

  if(!inherits(object$latent_model, "rSPDEobj1d") && !inherits(object$latent_model, "spacetimeobj")) {
      Q <- new_rspde_obj$Q
      if(!inherits(object$latent_model, "CBrSPDEobj2d") && !inherits(object$latent_model, "intrinsicCBrSPDEobj")) {
          # Extract alpha or nu from random effects
          alpha_param <- grep("^alpha", names(coeff_random), value = TRUE)
          nu_param <- grep("^nu", names(coeff_random), value = TRUE)
          
          if (length(alpha_param) > 0) {
            alpha <- coeff_random[[alpha_param]]
          } else if (length(nu_param) > 0) {
            nu <- coeff_random[[nu_param]]
            alpha <- nu + object$latent_model$d/2
          } else {
            stop("Neither alpha nor nu parameter found in random effects")
          }
          
          # Check if alpha is integer
          if (alpha %% 1 != 0) {
            Aprd <- kronecker(matrix(1, 1, object$rspde_order + 1), Aprd)        
          }
      }
  } else if(inherits(object$latent_model, "spacetimeobj")) {
      Q <- new_rspde_obj$Q
  }
  
  for (repl_y in u_repl) {
    idx_repl <- repl_vec == repl_y

    idx_obs <- idx_obs_full[idx_repl]

    y_repl <- Y[idx_repl]
    y_repl <- y_repl[idx_obs]
    
    
    if(inherits(object$latent_model, "rSPDEobj1d")) {
        loc_repl <- object$loc[idx_repl]
        loc_repl <- loc_repl[idx_obs]
        
        loc_full <- c(loc_repl, loc)
        tmp <- sort(loc_full, index.return = TRUE)
        reo <- tmp$ix
        loc_sort <- tmp$x
        ireo <- 1:length(loc_sort)
        ireo[reo] <- 1:length(loc_sort)
        
        ind.obs <- ireo[1:length(loc_repl)]
        ind.pre <- ireo[(length(loc_repl)+1):length(loc_full)]
        tmp <- matern.rational.ldl(loc = loc_full, 
                                   order = new_rspde_obj$m, 
                                   nu = new_rspde_obj$nu, 
                                   kappa = new_rspde_obj$kappa, 
                                   sigma = new_rspde_obj$sigma, 
                                   type_rational = new_rspde_obj$type_rational_approx, 
                                   type_interp =  new_rspde_obj$type_interp)    
        A_repl <- tmp$A[ind.obs, ] 
        Aprd <- tmp$A[ind.pre, ] 
        Q <- t(tmp$L)%*%tmp$D%*%tmp$L
    } else {
        A_repl <- object$A_list[[repl_y]]
    }
    
    if(object$mean_correction) {
        y_repl <- y_repl - A_repl %*% mu_corr    
    }
    
    Q_xgiveny <- t(A_repl) %*% A_repl / sigma_e^2 + Q
    mu_krig <- solve(Q_xgiveny, as.vector(t(A_repl) %*% y_repl / sigma_e^2))

    mu_krig <- Aprd %*% mu_krig
    
    mu_re <- mu_krig
    mu_fe <- mu_prd
    mu_krig <- mu_prd + mu_krig
    
    if(object$mean_correction) {
        mu_krig <- mu_krig + Aprd %*% mu_corr
        mu_re <- mu_re + Aprd %*% mu_corr
    }

    mean_tmp <- as.vector(mu_krig)
    mean_fe_tmp <- as.vector(mu_fe)
    mean_re_tmp <- as.vector(mu_re)

    if (!return_as_list) {
      out$mean <- c(out$mean, mean_tmp)
      out$repl <- c(out$repl, rep(repl_y, n_prd))
      out$fe_mean <- c(out$fe_mean, mean_fe_tmp)
      out$re_mean <- c(out$re_mean, mean_re_tmp)
    } else {
      out$mean[[repl_y]] <- mean_tmp
      out$fe_mean[[repl_y]] <- mean_fe_tmp
      out$re_mean[[repl_y]] <- mean_re_tmp
    }

    if (compute_variances) {
      post_cov <- Aprd %*% solve(Q_xgiveny, t(Aprd))
      var_tmp <- pmax(diag(post_cov), 0)

      if (!return_as_list) {
        out$variance <- rep(var_tmp, length(u_repl))
      } else {
        for (repl_y in u_repl) {
          out$variance[[repl_y]] <- var_tmp
        }
      }
    }

    if (posterior_samples) {
      mean_tmp <- as.vector(mu_krig)
      if(0) { #Old coviariance-based sampling code
          post_cov <- Aprd %*% solve(Q_xgiveny, t(Aprd))
          Z <- rnorm(dim(post_cov)[1] * n_samples)
          dim(Z) <- c(dim(post_cov)[1], n_samples)
          LQ <- tryCatch(chol(post_cov), error = function(e) {
              chol(post_cov + 1e-8 * Matrix::Diagonal(nrow(post_cov)))
          })
          X <- LQ %*% Z    
      } else {
          Z <- rnorm(dim(Q_xgiveny)[1] * n_samples)
          dim(Z) <- c(dim(Q_xgiveny)[1], n_samples)
          LQ <- tryCatch(chol(Q_xgiveny), error = function(e) {
              chol(Q_xgiveny + 1e-8 * Matrix::Diagonal(nrow(Q_xgiveny)))
          })
          X <- as.matrix( Aprd %*% solve(LQ, Z)) 
      }
      
      X <- X + mean_tmp
      if (!sample_latent) {
        X <- X + matrix(rnorm(n_samples * length(mean_tmp), sd = sigma.e), nrow = length(mean_tmp))
      } else {
        X <- X - as.vector(mu_prd)
      }

      if (!return_as_list) {
        out$samples <- rbind(out$samples, X)
      } else {
        out$samples[[repl_y]] <- X
      }
    }
  }

  return(out)
}


#' @name logLik.rspde_lme
#' @title Log-likelihood for \code{rspde_lme} objects
#' @description computes the log-likelihood for a fitted mixed effects model with `rspde_lme`.
#' @param x Object of class `rspde_lme` containing results from the fitted model.
#' @param ... Currently not used.
#' @return Log-likelihood value.
#' @noRd
#' @method logLik rspde_lme
#' @export
logLik.rspde_lme <- function(object, ...) {
  ll <- object$loglik
  attr(ll, "df") <- 1 + length(object$coeff$fixed_effects) + length(object$coeff$random_effects)
  return(ll)
}

#' @name nobs.rspde_lme
#' @title Number of observations for \code{rspde_lme} objects
#' @description Gets the number of observations of the fitted object.
#' @param x Object of class `rspde_lme` containing results from the fitted model.
#' @param ... Currently not used.
#' @return The number of observations.
#' @noRd
#' @method nobs rspde_lme
#' @export
nobs.rspde_lme <- function(object, ...) {
  return(object$nobs)
}


#' @name deviance.rspde_lme
#' @title Deviance for \code{rspde_lme} objects
#' @description Gets the deviance for `rspde_lme` fitted object.
#' @param x Object of class `rspde_lme` containing results from the fitted model.
#' @param ... Currently not used.
#' @return Deviance
#' @noRd
#' @method deviance rspde_lme
#' @export
deviance.rspde_lme <- function(object, ...) {
  if (length(object$coeff$random_effects) > 0) {
    return(-2 * object$loglik)
  } else {
    df_temp <- as.data.frame(object$model_matrix)
    colnames(df_temp)[1] <- as.character(object$response_var)

    fit_lm <- stats::lm(object$formula, data = df_temp)
    return(stats::deviance(fit_lm))
  }
}


#' @name glance.rspde_lme
#' @title Glance at an \code{rspde_lme} object
#' @aliases glance glance.rspde_lme
#' @description Glance accepts a \code{rspde_lme} object and returns a
#' [tidyr::tibble()] with exactly one row of model summaries.
#' The summaries are the square root of the estimated variance of the measurement error, residual
#' degrees of freedom, AIC, BIC, log-likelihood,
#' the type of latent model used in the fit and the total number of observations.
#' @param x An \code{rspde_lme} object.
#' @param ... Currently not used.
#' @return A [tidyr::tibble()] with exactly one row and columns:
#' \itemize{
#'   \item `nobs` Number of observations used.
#'   \item `sigma` the square root of the estimated residual variance
#'   \item `logLik` The log-likelihood of the model.
#'   \item `AIC` Akaike's Information Criterion for the model.
#'   \item `BIC` Bayesian Information Criterion for the model.
#'   \item `deviance` Deviance of the model.
#'   \item `df.residual` Residual degrees of freedom.
#'   \item `model.type` Type of latent model fitted.
#'   }
#' @seealso [augment.rspde_lme]
#' @method glance rspde_lme
#' @export

glance.rspde_lme <- function(x, ...) {
  alpha <- NULL
  if (!is.null(x$alpha)) {
    alpha <- x$alpha
  } else {
    alpha <- x$estimated_alpha
  }
  tidyr::tibble(
    nobs = stats::nobs(x),
    sigma = as.numeric(x$coeff$measurement_error[[1]]),
    logLik = as.numeric(stats::logLik(x)), AIC = stats::AIC(x),
    BIC = stats::BIC(x), deviance = stats::deviance(x),
    df.residual = stats::df.residual(x),
    model = x$latent_model$type,
    alpha = alpha,
    cov_function = x$latent_model$cov_function_name
  )
}


#' @name augment.rspde_lme
#' @title Augment data with information from a `rspde_lme` object
#' @aliases augment augment.rspde_lme
#' @description Augment accepts a model object and a dataset and adds information about each observation in the dataset. It includes
#' predicted values in the `.fitted` column, residuals in the `.resid` column, and standard errors for the fitted values in a `.se.fit` column.
#' It also contains the New columns always begin with a . prefix to avoid overwriting columns in the original dataset.
#' @param x A `rspde_lme` object.
#' @param newdata A `data.frame` or a `list` containing the covariates, the edge
#' number and the distance on edge for the locations to obtain the prediction. If `NULL`, the fitted values will be given for the original locations where the model was fitted.
#' @param loc Prediction locations. Can either be a `data.frame`, a `matrix` or a character vector, that contains the names of the columns of the coordinates of the locations. For models using `metric_graph` objects, plase use `edge_number` and `distance_on_edge` instead.
#' @param mesh Obtain predictions for mesh nodes? The graph must have a mesh, and either `only_latent` is set to TRUE or the model does not have covariates.
# @param repl Which replicates to obtain the prediction. If `NULL` predictions will be obtained for all replicates. Default is `NULL`.
#' @param which_repl  Which replicates to obtain the prediction. If `NULL` predictions
#' will be obtained for all replicates. Default is `NULL`.
#' @param se_fit Logical indicating whether or not a .se.fit column should be added to the augmented output. If TRUE, it only returns a non-NA value if type of prediction is 'link'.
#' @param conf_int Logical indicating whether or not confidence intervals for the fitted variable should be built.
#' @param pred_int Logical indicating whether or not prediction intervals for future observations should be built.
#' @param level Level of confidence and prediction intervals if they are constructed.
#' @param n_samples Number of samples when computing prediction intervals.
#' @param ... Additional arguments. Expert use only. 
#'
#' @return A [tidyr::tibble()] with columns:
#' \itemize{
#'   \item `.fitted` Fitted or predicted value.
#'   \item `.fittedlwrconf` Lower bound of the confidence interval, if conf_int = TRUE
#'   \item `.fitteduprconf` Upper bound of the confidence interval, if conf_int = TRUE
#'   \item `.fittedlwrpred` Lower bound of the prediction interval, if pred_int = TRUE
#'   \item `.fitteduprpred` Upper bound of the prediction interval, if pred_int = TRUE
#'   \item `.fixed` Prediction of the fixed effects.
#'   \item `.random` Prediction of the random effects.
#'   \item `.resid` The ordinary residuals, that is, the difference between observed and fitted values.
#'   \item `.se_fit` Standard errors of fitted values, if se_fit = TRUE.
#'   }
#'
#' @seealso [glance.rspde_lme]
#' @method augment rspde_lme
#' @export
augment.rspde_lme <- function(x, 
                              newdata = NULL, 
                              loc = NULL, 
                              mesh = FALSE, 
                              which_repl = NULL, 
                              se_fit = FALSE, 
                              conf_int = FALSE, 
                              pred_int = FALSE, 
                              level = 0.95, 
                              n_samples = 100, 
                              ...) {
  .resid <- FALSE
  if (is.null(newdata)) {
    .resid <- TRUE
  }


  level <- level[[1]]
  if (!is.numeric(level)) {
    stop("'level' must be numeric!")
  }
  if (level > 1 || level < 0) {
    stop("'level' must be between 0 and 1!")
  }

  tmp_args <- list(...)

  distance_on_edge <- tmp_args[["distance_on_edge"]]
  edge_number <- tmp_args[["edge_number"]]
  normalized <- tmp_args[["normalized"]]

  if(is.null(distance_on_edge)){
    distance_on_edge <- "distance_on_edge"
  }
  if(is.null(edge_number)){
    edge_number <- "edge_number"
  }
  if(is.null(normalized)){
    normalized <- FALSE
  }

  if (pred_int) {
    pred <- stats::predict(x,
      newdata = newdata, loc = loc, mesh = mesh, which_repl = which_repl, compute_variances = TRUE,
      posterior_samples = TRUE, n_samples = n_samples, edge_number = edge_number,
      distance_on_edge = distance_on_edge, normalized = normalized, return_original_order = FALSE, return_as_list = FALSE
    )
  } else if (conf_int || se_fit) {
    pred <- stats::predict(x,
      newdata = newdata, loc = loc, mesh = mesh, which_repl = which_repl, compute_variances = TRUE,
      posterior_samples = FALSE, edge_number = ".edge_number",
      distance_on_edge = ".distance_on_edge", normalized = TRUE, return_original_order = FALSE, return_as_list = FALSE
    )
  } else {
    pred <- stats::predict(x,
      newdata = newdata, loc = loc, mesh = mesh, which_repl = which_repl, compute_variances = FALSE, posterior_samples = FALSE, edge_number = ".edge_number",
      distance_on_edge = ".distance_on_edge", normalized = TRUE, return_original_order = FALSE, return_as_list = FALSE
    )
  }

  if (se_fit || pred_int || conf_int) {
    newdata[[".fitted"]] <- pred$mean
    newdata[[".se_fit"]] <- sqrt(pred$variance)
    newdata[[".fixed"]] <- pred$fe_mean
    newdata[[".random"]] <- pred$re_mean
    if (.resid) {
      newdata[[".resid"]] <- pred$mean - newdata[[as.character(x$response_var)]]
    }
  } else {
    newdata[[".fitted"]] <- pred$mean
    newdata[[".fixed"]] <- pred$fe_mean
    newdata[[".random"]] <- pred$re_mean
    if (.resid) {
      newdata[[".resid"]] <- pred$mean - newdata[[as.character(x$response_var)]]
    }
  }


  if (conf_int) {
    newdata$.fittedlwrconf <- newdata[[".fitted"]] + stats::qnorm((1 - level) / 2) * newdata[[".se_fit"]]
    newdata$.fitteduprconf <- newdata[[".fitted"]] + stats::qnorm((1 + level) / 2) * newdata[[".se_fit"]]
  }

  if (pred_int) {
    list_pred_int <- lapply(1:nrow(pred$samples), function(i) {
      y_sim <- pred$samples[i, ]
      y_sim <- sort(y_sim)
      idx_lwr <- max(1, round(n_samples * (1 - level) / 2))
      idx_upr <- round(n_samples * (1 + level) / 2)
      c(y_sim[idx_lwr], y_sim[idx_upr])
    })
    list_pred_int <- unlist(list_pred_int)
    list_pred_int <- t(matrix(list_pred_int, nrow = 2))
    newdata$.fittedlwrpred <- list_pred_int[, 1]
    newdata$.fitteduprpred <- list_pred_int[, 2]
  }


  newdata
}
