% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clean_genoprob.R
\name{clean_genoprob}
\alias{clean_genoprob}
\alias{clean.calc_genoprob}
\title{Clean genotype probabilities}
\usage{
clean_genoprob(
  object,
  value_threshold = 0.000001,
  column_threshold = 0.01,
  ind = NULL,
  cores = 1,
  ...
)

\method{clean}{calc_genoprob}(
  object,
  value_threshold = 0.000001,
  column_threshold = 0.01,
  ind = NULL,
  cores = 1,
  ...
)
}
\arguments{
\item{object}{Genotype probabilities as calculated by
\code{\link[=calc_genoprob]{calc_genoprob()}}.}

\item{value_threshold}{Probabilities below this value will be set
to 0.}

\item{column_threshold}{For genotype columns where the maximum
value is below this threshold, all values will be set to 0.
This must be less than \eqn{1/k} where \eqn{k} is the number of
genotypes.}

\item{ind}{Optional vector of individuals (logical, numeric, or
character). If provided, only the genotype probabilities for
these individuals will be cleaned, though the full set will be
returned.}

\item{cores}{Number of CPU cores to use, for parallel calculations.
(If \code{0}, use \code{\link[parallel:detectCores]{parallel::detectCores()}}.)
Alternatively, this can be links to a set of cluster sockets, as
produced by \code{\link[parallel:makeCluster]{parallel::makeCluster()}}.}

\item{...}{Ignored at this point.}
}
\value{
A cleaned version of the input genotype probabilities
object, \code{object}.
}
\description{
Clean up genotype probabilities by setting small values to 0 and
for a genotype column where the maximum value is rather small, set
all values in that column to 0.
}
\details{
In cases where a particular genotype is largely absent,
\code{scan1coef()} and \code{fit1()} can give unstable estimates of the
genotype effects. Cleaning up the genotype probabilities by setting
small values to 0 helps to ensure that such effects get set to
\code{NA}.

At each position and for each genotype column, we find the maximum
probability across individuals. If that maximum is <
\code{column_threshold}, all values in that genotype column at that
position are set to 0.

In addition, any genotype probabilities that are < \code{value_threshold}
(generally < \code{column_threshold}) are set to 0.

The probabilities are then re-scaled so that the probabilities for
each individual at each position sum to 1.

If \code{ind} is provided, the function is applied only to the
designated subset of individuals. This may be useful when only a
subset of individuals have been phenotyped, as you may want to zero
out genotype columns where that subset of individuals has only
negligible probability values.
}
\examples{
iron <- read_cross2(system.file("extdata", "iron.zip", package="qtl2"))
\dontshow{iron <- iron[,c("19", "X")] # subset to chr 19 and X}

# calculate genotype probabilities
probs <- calc_genoprob(iron, error_prob=0.002)

# clean the genotype probabilities
# (doesn't really do anything in this case, because there are no small but non-zero values)
probs_clean <- clean(probs)

# clean only the females' genotype probabilities
probs_cleanf <- clean(probs, ind=names(iron$is_female)[iron$is_female])
}
