% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sboot.R
\name{sboot.mb}
\alias{sboot.mb}
\title{Bootstrap with residual moving blocks for individual SVAR models}
\usage{
sboot.mb(
  x,
  b.length = 1,
  n.ahead = 20,
  n.boot = 500,
  n.cores = 1,
  fix_beta = TRUE,
  deltas = cumprod((100:0)/100),
  normf = NULL
)
}
\arguments{
\item{x}{VAR object of class '\code{id}' or '\code{varx}' or any other 
that can be \link[=as.varx]{coerced} to '\code{varx}', e.g. '\code{svars}'.
If a bias term \code{x$PSI_bc} is available for coefficient matrix \eqn{A} (such as in '\code{sboot2}'), 
the bias-corrected second-step of the bootstrap-after-bootstrap procedure by Kilian (1998) is performed.}

\item{b.length}{Integer. Length \eqn{b_{(t)}} of each residual time series block, 
which is often set to \eqn{T/10}.}

\item{n.ahead}{Integer. Number of periods to consider after the initial impulse, i.e. the horizon of the IRF.}

\item{n.boot}{Integer. Number of bootstrap iterations.}

\item{n.cores}{Integer. Number of allocated processor cores.}

\item{fix_beta}{Logical. If \code{TRUE} (the default), the cointegrating vectors \eqn{\beta} 
are fixed over all bootstrap iterations. Ignored in case of rank-unrestricted VAR. 
Use this for VECM with known \eqn{\beta}, too. Note that \eqn{\beta} is fixed in \code{vars:::.bootsvec}, 
but not in \code{vars:::.bootirfsvec} or \code{vars:::.bootirfvec2var}.}

\item{deltas}{Vector. Numeric weights \eqn{\delta_j} that are successively 
multiplied to the bias estimate \eqn{\hat{\Psi}} for a stationary correction. 
The default weights \code{deltas = cumprod((100:0)/100)} correspond
to the iterative correction procedure of Step 1b in Kilian (1998). 
Choosing \code{deltas = NULL} deactivates the bootstrap-after-bootstrap procedure.}

\item{normf}{Function. A given function that normalizes the \eqn{K \times S} input-matrix 
into an output matrix of same dimension. See the example in \code{\link{id.iv}} 
for the normalization of Jentsch and Lunsford (2021) 
that fixes the size of the impact response in the IRF.}
}
\value{
A list of class '\code{sboot2}' with elements:
\item{true}{Point estimate of impulse response functions.}
\item{bootstrap}{List of length '\code{n.boot}' holding bootstrap impulse response functions.}
\item{A}{List for the VAR coefficients containing 
  the matrix of point estimates '\code{par}' and 
  the array of bootstrap results '\code{sim}'.}
\item{B}{List for the structural impact matrix containing 
  the matrix of point estimates '\code{par}' and 
  the array of bootstrap results '\code{sim}'.}
\item{PSI_bc}{Matrix of the estimated bias term \eqn{\hat{\Psi}} 
  for the VAR coefficients \eqn{\hat{A}} according to Kilian (1998).}
\item{varx}{Input VAR object of class '\code{varx}' 
  that has been subjected to the first-step bias-correction.}
\item{nboot}{Number of correct bootstrap iterations.}
\item{b_length}{Length of each block.}
\item{design}{Character indicating that the recursive design bootstrap has been performed.}
\item{method}{Used bootstrap method.}
\item{stars}{Matrix of (\eqn{T \times }\code{n.boot}) integers containing 
  the \eqn{T} resampling draws from each bootstrap iteration.}
}
\description{
Calculates confidence bands for impulse response functions via recursive-design bootstrap.
}
\examples{
\donttest{
# select minimal or full example #
is_min = TRUE
n.boot = ifelse(is_min, 5, 500)

# use 'b.length=1' to conduct basic "vars" bootstraps #
set.seed(23211)
data("Canada")
R.vars = vars::VAR(Canada, p=2, type="const")
R.svar = svars::id.chol(R.vars)
R.boot = sboot.mb(R.svar, b.length=1, n.boot=n.boot, n.ahead=30, n.cores=1)
summary(R.boot, idx_par="A", level=0.9)  # VAR coefficients with 90\%-confidence intervals 
plot(R.boot, lowerq = c(0.05, 0.1, 0.16), upperq = c(0.95, 0.9, 0.84))

# second step of bootstrap-after-bootstrap #
R.bab = sboot.mb(R.boot, b.length=1, n.boot=n.boot, n.ahead=30, n.cores=1)
summary(R.bab, idx_par="A", level=0.9)  # VAR coefficients with 90\%-confidence intervals 
plot(R.bab, lowerq = c(0.05, 0.1, 0.16), upperq = c(0.95, 0.9, 0.84))

# conduct bootstraps for Blanchard-Quah type SVAR from "vars" #
set.seed(23211)
data("Canada")
R.vars = vars::VAR(Canada, p=2, type="const")
R.svar = vars::BQ(R.vars)
R.boot = sboot.mb(R.svar, b.length=1, n.boot=n.boot, n.ahead=30, n.cores=1)
summary(R.boot, idx_par="B", level=0.9)  # impact matrix with 90\%-confidence intervals 
plot(R.boot, lowerq = c(0.05, 0.1), upperq = c(0.95, 0.9), cumulative=2:3) 
# impulse responses of the second and third variable are accumulated

# set 'args_id' to CvM defaults of "svars" bootstraps #
set.seed(23211)
data("USA")
R.vars = vars::VAR(USA, lag.max=10, ic="AIC")
R.cob  = copula::indepTestSim(R.vars$obs, R.vars$K, verbose=FALSE)
R.svar = svars::id.cvm(R.vars, dd=R.cob)

R.varx = as.varx(R.svar, dd=R.cob, itermax=300, steptol=200, iter2=50)
R.boot = sboot.mb(R.varx, b.length=15, n.boot=n.boot, n.ahead=30, n.cores=1)
plot(R.boot, lowerq = c(0.05, 0.1, 0.16), upperq = c(0.95, 0.9, 0.84))
}

}
\references{
Breitung, J., Brueggemann R., and Luetkepohl, H. (2004): 
  "Structural Vector Autoregressive Modeling and Impulse Responses", 
  in \emph{Applied Time Series Econometrics}, 
  ed. by H. Luetkepohl and M. Kraetzig, 
  Cambridge University Press, Cambridge.

Brueggemann R., Jentsch, C., and Trenkler, C. (2016): 
  "Inference in VARs with Conditional Heteroskedasticity of Unknown Form", 
  \emph{Journal of Econometrics}, 191, pp. 69-85.

Jentsch, C., and Lunsford, K. G. (2021): 
  "Asymptotically Valid Bootstrap Inference for Proxy SVARs",
  \emph{Journal of Business and Economic Statistics}, 40, pp. 1876-1891.

Kilian, L. (1998): 
  "Small-Sample Confidence Intervals for Impulse Response Functions",
  \emph{Review of Economics and Statistics}, 80, pp. 218-230.

Luetkepohl, H. (2005): 
  \emph{New Introduction to Multiple Time Series Analysis}, 
  Springer, 2nd ed.
}
\seealso{
\code{\link[svars]{mb.boot}}, \code{\link[vars]{irf}}, 
  and the panel counterpart \code{\link{sboot.pmb}}.
}
