% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/e0_to_lifetab.R
\name{e0_to_lifetab_logit}
\alias{e0_to_lifetab_logit}
\title{Derive Life Tables that Match Life Expectancies,
using a Brass Logit Model}
\usage{
e0_to_lifetab_logit(
  target,
  standard,
  infant = c("constant", "linear", "CD", "AK"),
  child = c("constant", "linear", "CD"),
  closed = c("constant", "linear"),
  open = "constant",
  radix = 1e+05,
  suffix = NULL
)
}
\arguments{
\item{target}{A data frame containing a variable called
\code{"e0"}, and possibly other varibles. See Details.}

\item{standard}{A data frame containing variables
called \code{age} and \code{lx}, and possibly others.
See Details.}

\item{infant, child, closed, open}{Methods used to
calculate life expectancy. See \code{\link[=lifetab]{lifetab()}} for details.}

\item{radix}{Initial population for the
\code{lx} column in the derived life table(s).
Default is \code{100000}.}

\item{suffix}{Optional suffix added to life table columns.}
}
\value{
A \link[tibble:tibble]{tibble}.
}
\description{
Turn life expectancies at birth into full life tables,
using the Brass logit model. The method is simple
and is designed for simulations or for
settings with little or no data on age-specific
mortality rates. In settings where data
on age-specific mortality is available,
other methods might be more appropriate.
}
\section{Method}{


The method implemented by \code{e0_to_lifetab_logit()} is
based on the observation that, if populations A and B
are demographically similar, then, in many cases,

\deqn{\text{logit}(q_x^{\text{B}}) \approx \alpha + \beta \text{logit}(q_x^{\text{A}})}

where \eqn{q_x} is the life table probability of dying
between birth and age \eqn{x}. By definition, \eqn{q_x = 1 - l_x},
where \eqn{l_x} is the standard life table function,
with radix (\eqn{l_0}) equal to 1.

Given (i) target life expectancy,
(ii) a set of \eqn{l_x^{\text{A}}}),
(referred to as a "standard"), and
(iii) a value for \eqn{\beta},
\code{e0_to_lifetab_logit()} finds
a value for \eqn{\alpha} that yields a set of
\eqn{q_x^{\text{B}}}) with the required life expectancy.
If populations A and B
are similar, then \eqn{beta} is likely to close to 1.
}

\section{The \code{target} argument}{


\code{target} is a data frame specifying
life expectancies for each population being modelled,
and, optionally, inputs to the calculations, and
'by' variables.

\code{target} contains the following variables:
\itemize{
\item A variable called \code{"e0"} giving life expectancy at birth.
\item Optionally, a variable called \code{"beta"} with values
for \eqn{\beta}.
Can be an ordinary numeric vector
or an \link[rvec:rvec]{rvec}.
If \code{target} does not include a \code{"beta"} variable,
then \code{e0_to_lifetab_logit()} sets \eqn{\beta} to 1.
\item A variable called \code{"sex"}. The \code{"sex"} variable
must be supplied if the \code{infant}
argument to \code{e0_to_lifetab_logit()} is \code{"CD"} or \code{"AK"},
or if the \code{child} argument is \code{"CD"}.
\item Optionally, 'by' variables. Typical examples
are time, region, and model variant.
}
}

\section{The \code{standard} argument}{


\code{standard} is a data frame specifying
the \eqn{l_x} to be used with each life expectancy
in \code{target}, and, optionally, values for the average age
person-years lived by people who die in each group,
\eqn{_na_x}. Values in \code{standard} are age-specific.

\code{standard} contains the following variables:
\itemize{
\item A variable called \code{"age"}, with labels that
can be parsed by \code{\link[=reformat_age]{reformat_age()}}.
\item A variable called \code{"lx"}.  Cannot be an rvec.
\item Additional variables used to match rows in \code{standard}
to rows in \code{target}.
}
}

\examples{
## create new life tables based on level-1
## 'West' model life tables, but with lower
## life expectancy

library(dplyr, warn.conflicts = FALSE)

target <- data.frame(sex = c("Female", "Male"), 
                     e0 = c(17.5, 15.6))

standard <- west_lifetab |>
    filter(level == 1) |>
    select(sex, age, lx)
    
e0_to_lifetab_logit(target = target,
                    standard = standard,
                    infant = "CD",
                    child = "CD")

## target is an rvec
library(rvec, warn.conflicts = FALSE)
target_rvec <- data.frame(sex = c("Female", "Male"), 
                          e0 = rnorm_rvec(n = 2,
                                          mean = c(17.5, 15.6),
                                          n_draw = 1000))
e0_to_lifetab_logit(target = target_rvec,
                    standard = standard)
}
\references{
Brass W, Coale AJ. 1968. “Methods of analysis and estimation,”
in Brass, W,  Coale AJ, Demeny P, Heisel DF, et al. (eds).
The Demography of Tropical Africa. Princeton NJ:
Princeton University Press, pp. 88–139.

Moultrie TA, Timæus IM. 2013. Introduction to Model Life Tables.
In Moultrie T, Dorrington R, Hill A, Hill K, Timæus I, Zaba B.
(eds). Tools for Demographic Estimation.
Paris: International Union for the Scientific Study of Population.
\href{https://demographicestimation.iussp.org/content/using-models-derive-life-tables-incomplete-data}{online version}.
}
\seealso{
\itemize{
\item \link{tfr_to_asfr_scale} Fertility equivalent of
\code{e0_to_lifetab_logit()}
\item \code{\link[=logit]{logit()}}, \code{\link[=invlogit]{invlogit()}} Logit function
\item \code{\link[=lifeexp]{lifeexp()}} Calculate life expectancy from detailed inputs
}
}
