% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/trendFit.R
\name{ptrend}
\alias{ptrend}
\title{Fit a smooth or linear trend to count survey data.}
\usage{
ptrend(
  formula,
  data = list(),
  family = quasipoisson(),
  nGrid = 500,
  nBoot = 500,
  bootType = "hessian",
  gamModel = TRUE,
  engine = "gam",
  ...
)
}
\arguments{
\item{formula}{A trend formula. This is a GAM formula with an extra term \code{\link[poptrend]{trend}} describing the 
time variable and properties of the trend. All terms except the trend term are treated as covariates. 
Effect of temporal variation in these covariates are not included in the calculation of the trend.}

\item{data}{A data frame containing response variables and covariates.}

\item{family}{The distributional family of the response. The family most use a log-link, it defaults to a quasi-Poisson.}

\item{nGrid}{The number of grid points over which to compute the trend.
If the length of the argument is one, an equally spaced grid over the survey period of length nGrid is set up.
nGrid can also be a vector of length 3, in which case the first element is the number of grid points and the
second and third elements give, respectively, the start and endpoints of the grid.}

\item{nBoot}{The number of bootstrap samples to draw.}

\item{bootType}{Only one method, "hessian", currently implemented. Type "hessian", draws bootstrap samples using the Bayesian
covariance matrix of the parameters (see \code{\link[mgcv]{vcov.gam}}).}

\item{gamModel}{If true, the fit of the underlying gam model is saved in the output. May be set to FALSE to save memory,
but with the side effect that the fit of the gam model cannot be checked.}

\item{engine}{If 'gam', the default, model fitting is done via \code{\link[mgcv]{gam}}. If 'bam', model fitting is done via 
\code{\link[mgcv]{bam}}, which is less memory hungry and can be faster for large data sets.}

\item{...}{Further arguments passed to \code{\link[mgcv]{gam}}.}
}
\value{
An object of class trend.
}
\description{
The function estimates a trend from count survey data.
}
\details{
The function estimates smooth or loglinear population trends, or indexes from simple design count survey data. 
It is essentially a wrapper around a call to \code{\link[mgcv]{gam}}, processing its output using \code{\link[mgcv]{predict.gam}}
to produce a trend estimate.
For smooth trends, cubic regression splines for the temporal variable are set up by the term \code{s(var, k = k, fx = fx , bs = "cr")}
where \code{var} is the first argument to \code{\link[poptrend]{trend}} in the formula.
For loglinear trends, the identity of \code{var} is used, and for index models a factor variable is constructed from \code{var}.

Temporal random effects are set up by converting the temporal variable supplied to \code{\link{trend}} to a factor variable
and adding this factor variable to the data supplied to \code{\link[mgcv]{gam}}.

Bootstrap confidence intervals are computed by drawing normally distributed random variable with means equal to the
estimated coefficients and covariance matrix equal to the Bayesian posterior covariance matrix (see \link[mgcv]{vcov.gam}).
}
\examples{
## Simulate a data set with 15 sites and 25 years
data = simTrend(15, 25)
## Fit a smooth trend with fixed site effects, random time effects,
## and automatic selection of degrees of freedom
trFit = ptrend(count ~ trend(year, tempRE = TRUE, type = "smooth") + site, data = data)
## Check the model fit
checkFit(trFit)
## Plot the trend
plot(trFit)
summary(trFit)
## Check the estimated percent change from year 8 to 25
change(trFit, 8, 25)

## Fit a loglinear trend model with random site effects and random time effects 
## to the same data set.
trLin = ptrend(count ~ trend(year, tempRE = TRUE, type = "loglinear") +
                 s(site, bs = "re"), data = data)
plot(trLin)
summary(trLin)

## Fit an index model with fixed site effects and an (unrelated) continous covariate 
## as a smooth effect.
# Simulate mock covariate unrelated to data.
cov = rnorm(nrow(data))
trInd = ptrend(count ~ trend(year, type = "index") + site + s(cov), data = data)
plot(trInd)
summary(trInd)
}
\author{
Jonas Knape
}
