#' Convert a pooldata object into DIYABC input files.
#' @description Convert a pooldata object into DIYABC data file for pool-seq data. A file containing SNP details is also printed out. Options to generate sub-samples  (e.g., for large number of SNPs) are also available. Note that DIYABC SNP filtering criterion is based on MRC (minimal read count) which may be more stringent than usual MAF-based filtering criterion. It is recommended to parse vcf files and pooldata objects without any MAF criterion or to prefilter pooldata objects with the desired MRC (using option snp.index \code{\link{pooldata.subset}}).
#' @param pooldata A pooldata object containing Pool-Seq information (see \code{\link{vcf2pooldata}} and \code{\link{popsync2pooldata}})
#' @param writing.dir Directory where to create the files  (e.g., set writing.dir=getwd() to copy in the current working directory)
#' @param prefix Prefix used for output file names
#' @param diyabc.mrc MRC to be applied by DIYABC (note that no filtering based on MRC is done by the function)
#' @param subsamplesize Size of the sub-samples. If <=1 (default), all the SNPs are considered in the output
#' @param subsamplingmethod If sub-sampling is activated (argument subsamplesize), define the method used for subsampling that might be either i) "random" (A single data set consisting of randmly chosen SNPs is generated) or ii) "thinning", sub-samples are generated by taking SNPs one every nsub=floor(nsnp/subsamplesize) in the order of the map (a suffix ".subn" is added to each sub-sample files where n varies from 1 to nsub).
#' @return DIYABC data file for pool-seq data
#' @seealso To generate pooldata object, see \code{\link{vcf2pooldata}}, \code{\link{popsync2pooldata}}
#' @examples
#'  make.example.files(writing.dir=tempdir())
#'  pooldata=popsync2pooldata(sync.file=paste0(tempdir(),"/ex.sync.gz"),poolsizes=rep(50,15))
#'  pooldata2diyabc(pooldata=pooldata,writing.dir=tempdir())
#' @export
pooldata2diyabc=function(pooldata,writing.dir=getwd(),prefix="",diyabc.mrc=1,subsamplesize=-1,subsamplingmethod="thinning"){
  if(writing.dir==""){stop("ERROR: Please provide the directory path where to copy the example files  (e.g., set writing.dir=getwd() to copy in the current working directory)")}
  if(!(is.pooldata(pooldata))) {stop("Data are not formatted as a valid pooldata object...")}
  if(pooldata@npools<2){stop("At least 2 pool samples required\n")} #may arrive after pooldata.subset with a single pop
  subsampling=FALSE
  if(subsamplesize>1){
    if(!(subsamplingmethod %in% c("thinning","random"))){stop("subsampling method should either be \"random\" or \"thinning\"")}
    if(subsamplingmethod=="thinning"){
      tmp.n=floor(pooldata@nsnp/subsamplesize)
      cat(tmp.n,"sub-samples of ca.",subsamplesize,"SNPs will be generated by tacking one SNP every",tmp.n,"\n")
    }else{
      cat("One sub-samples of",subsamplesize,"randomly chosen SNPs will be generated\n")
    }
    subsampling=TRUE  
  }
  
  mat.count=matrix(0,pooldata@nsnp,2*pooldata@npools)
  tmp.id=2*(1:pooldata@npools)-1
  mat.count[,tmp.id]=pooldata@refallele.readcount
  mat.count[,(tmp.id+1)]=pooldata@readcoverage - pooldata@refallele.readcount
  outgenofilename      =paste0(writing.dir,"/diyabc")
  outsnpdetfilename    =paste0(writing.dir,"/snpdet") 
  if(nchar(prefix)>0){
    outgenofilename=paste0(writing.dir,"/",prefix,".diyabc")
    outsnpdetfilename=paste0(writing.dir,"/",prefix,".snpdet")
  }

  header1=paste0("<NM=1.0NF> <MRC=",diyabc.mrc,"> Title/comment\n")
  header2=paste("POOL POP_NAME:HAPLOID_SAMPLE_SIZE",paste(paste0(gsub(" ","_",pooldata@poolnames),":",pooldata@poolsizes),collapse=" "),"\n")
  
  if(subsampling){
    if(subsamplingmethod=="thinning"){
      tmp.n=floor(pooldata@nsnp/subsamplesize)
      for(i in 1:tmp.n){
        tmp.outfile=paste0(outgenofilename,".sub",i)
        cat(file=tmp.outfile,header1)
        cat(file=tmp.outfile,header2, append = T)
        tmp.sel=seq(i,pooldata@nsnp,tmp.n)
        write.table(file=tmp.outfile,mat.count[tmp.sel,],quote=F,col.names=F,row.names=F,append = T)
        write.table(file=paste0(outsnpdetfilename,".sub",i),pooldata@snp.info[tmp.sel,],quote=F,col.names=F,row.names=F) 
      }
    }
    if(subsamplingmethod=="random"){
      cat(file=outgenofilename,header1)
      cat(file=outgenofilename,header2, append = T)
      tmp.sel=sort(sample(1:pooldata@nsnp,subsamplesize))
      write.table(file=outgenofilename,mat.count[tmp.sel,],quote=F,col.names=F,row.names=F,append = T)
      write.table(file=outsnpdetfilename,pooldata@snp.info[tmp.sel,],quote=F,col.names=F,row.names=F) 
    }
  }else{
    cat(file=outgenofilename,header1)
    cat(file=outgenofilename,header2, append = T)
    write.table(file=outgenofilename,mat.count,quote=F,col.names=F,row.names=F,append = T) 
    write.table(file=outsnpdetfilename,pooldata@snp.info,quote=F,col.names=F,row.names=F) 
  }
}