% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/as.trelliData.R
\name{trelli_pvalue_filter}
\alias{trelli_pvalue_filter}
\title{Filter a paneled trelliData object by a p-value}
\usage{
trelli_pvalue_filter(
  trelliData,
  p_value_test = "anova",
  p_value_thresh = 0.05,
  comparison = NULL
)
}
\arguments{
\item{trelliData}{A trelliData object with statistics results (statRes). Required.}

\item{p_value_test}{A string to indicate which p_values to plot. Acceptable
entries are "anova" or "gtest". Default is "anova". Unlike the
plotting functions, here p_value_test cannot be null. Required unless
the data is seqData, when this parameter will be ignored.}

\item{p_value_thresh}{A value between 0 and 1 to indicate the p-value threshold 
at which to keep plots. Default is 0.05. Required.}

\item{comparison}{The specific comparison to filter significant values to. Can
be null. See attr(statRes, "comparisons") for the available options. Optional.}
}
\value{
A paneled trelliData object with only plots corresponding to significant
p-values from a statistical test.
}
\description{
This use-case-specific function allows users to filter down their plots to a 
   specified p-value IF statistics data has been included. This function is mostly
   relevant to the MODE application.
}
\examples{
\dontshow{if (requireNamespace("pmartRdata", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\donttest{
library(pmartRdata)

# Transform the data
omicsData <- edata_transform(omicsData = pep_object, data_scale = "log2")

# Group the data by condition
omicsData <- group_designation(omicsData = omicsData, main_effects = c("Phenotype"))

# Apply the IMD ANOVA filter
imdanova_Filt <- imdanova_filter(omicsData = omicsData)
omicsData <- applyFilt(filter_object = imdanova_Filt, omicsData = omicsData,
                       min_nonmiss_anova = 2)

# Normalize my pepData
omicsData <- normalize_global(omicsData, "subset_fn" = "all", "norm_fn" = "median",
                             "apply_norm" = TRUE, "backtransform" = TRUE)

# Implement the IMD ANOVA method and compute all pairwise comparisons 
# (i.e. leave the `comparisons` argument NULL)
statRes <- imd_anova(omicsData = omicsData, test_method = 'combined')

# Generate the trelliData object
trelliData3 <- as.trelliData(statRes = statRes)
trelliData4 <- as.trelliData(omicsData = omicsData, statRes = statRes)

###########################
## MS/NMR OMICS EXAMPLES ##
###########################

# Filter a trelliData object with only statistics results, while not caring about a comparison
trelli_pvalue_filter(trelliData3, p_value_test = "anova", p_value_thresh = 0.1)

# Filter a trelliData object with only statistics results, while caring about a specific comparison
trelli_pvalue_filter(
 trelliData3, p_value_test = "anova", p_value_thresh = 0.1, comparison = "Phenotype3_vs_Phenotype2")

# Filter both a omicsData and statRes object, while not caring about a specific comparison
trelli_pvalue_filter(trelliData4, p_value_test = "anova", p_value_thresh = 0.001)

# Filter both a omicsData and statRes object, while caring about a specific comparison
trelli_pvalue_filter(
 trelliData4, p_value_test = "gtest", p_value_thresh = 0.25, 
 comparison = "Phenotype3_vs_Phenotype2"
)

######################
## RNA-SEQ EXAMPLES ##  
######################

#' # Group data by condition
omicsData_seq <- group_designation(omicsData = rnaseq_object, main_effects = c("Virus"))

# Filter low transcript counts
omicsData_seq <- applyFilt(
 filter_object = total_count_filter(omicsData = omicsData_seq), 
 omicsData = omicsData_seq, min_count = 15
)

# Select a normalization and statistics method (options are 'edgeR', 'DESeq2', and 'voom').
# See ?difexp_seq for more details
statRes_seq <- diffexp_seq(omicsData = omicsData_seq, method = "voom")

# Generate the trelliData object
trelliData_seq3 <- as.trelliData(statRes = statRes_seq)
trelliData_seq4 <- as.trelliData(omicsData = omicsData_seq, statRes = statRes_seq)

# Filter a trelliData seqData object with only statistics results, while not 
# caring about a comparison
trelliData_seq3_filt <- trelli_pvalue_filter(trelliData_seq3, p_value_thresh = 0.05)

# Filter both a omicsData and statRes object, while caring about a specific comparison
trelliData_seq4_filt <- trelli_pvalue_filter(trelliData_seq4, p_value_thresh = 0.05, 
 comparison = "StrainA_vs_StrainB")

}
\dontshow{\}) # examplesIf}
}
\author{
David Degnan, Lisa Bramer
}
