% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/summ_interval.R
\name{summ_interval}
\alias{summ_interval}
\title{Summarize distribution with interval}
\usage{
summ_interval(f, level = 0.95, method = "minwidth", n_grid = 10001)
}
\arguments{
\item{f}{A pdqr-function representing distribution.}

\item{level}{A number between 0 and 1 representing a coverage degree of
interval. Interpretation depends on \code{method} but the bigger is number, the
wider is interval.}

\item{method}{Method of interval computation. Should be on of "minwidth",
"percentile", "sigma".}

\item{n_grid}{Number of grid elements to be used for "minwidth" method (see
Details).}
}
\value{
A \link{region} with one row. That is a data frame with one row
and the following columns:
\itemize{
\item \strong{left} \verb{<dbl>} : Left end of interval.
\item \strong{right} \verb{<dbl>} : Right end of interval.
}

To return a simple numeric vector, call \link[base:unlist]{unlist()} on
\code{summ_interval()}'s output (see Examples).
}
\description{
These functions summarize distribution with one interval based on method of
choice.
}
\details{
Method "minwidth" searches for an interval with total probability of
\code{level} that has minimum width. This is done with grid search: \code{n_grid}
possible intervals with \code{level} total probability are computed and the one
with minimum width is returned (if there are several, the one with the
smallest left end). Left ends of computed set of intervals are created as a
grid from \code{0} to \code{1-level} quantiles with \code{n_grid} number of elements. Right
ends are computed so that intervals have \code{level} total probability.

Method "percentile" returns an interval with edges being \code{0.5*(1-level)} and
\code{1 - 0.5*(1-level)} quantiles. Output has total probability equal to \code{level}.

Method "sigma" computes an interval symmetrically centered at
\link[=summ_mean]{mean} of distribution. Left and right edges are distant from
center by the amount of \link[=summ_sd]{standard deviation} multiplied by
\code{level}'s critical value. Critical value is computed using \link[stats:Normal]{normal distribution} as \code{qnorm(1 - 0.5*(1-level))}, which
corresponds to a way of computing sample confidence interval with known
standard deviation. The final output interval is possibly cut so that not to
be out of \code{f}'s \link[=meta_support]{support}.

\strong{Note} that supported methods correspond to different ways of computing
\link[=summ_center]{distribution's center}. This idea is supported by the fact
that when \code{level} is 0, "minwidth" method returns zero width interval at
distribution's \link[=summ_mode]{global mode}, "percentile" method -
\link[=summ_median]{median}, "sigma" - \link[=summ_mean]{mean}.
}
\examples{
# Type "discrete"
d_dis <- new_d(data.frame(x = 1:6, prob = c(3:1, 0:2) / 9), "discrete")
summ_interval(d_dis, level = 0.5, method = "minwidth")
summ_interval(d_dis, level = 0.5, method = "percentile")
summ_interval(d_dis, level = 0.5, method = "sigma")

## Visual difference between methods
plot(d_dis)
region_draw(summ_interval(d_dis, 0.5, method = "minwidth"), col = "blue")
region_draw(summ_interval(d_dis, 0.5, method = "percentile"), col = "red")
region_draw(summ_interval(d_dis, 0.5, method = "sigma"), col = "green")

# Type "continuous"
d_con <- form_mix(
  list(as_d(dnorm), as_d(dnorm, mean = 5)),
  weights = c(0.25, 0.75)
)
summ_interval(d_con, level = 0.5, method = "minwidth")
summ_interval(d_con, level = 0.5, method = "percentile")
summ_interval(d_con, level = 0.5, method = "sigma")

## Visual difference between methods
plot(d_con)
region_draw(summ_interval(d_con, 0.5, method = "minwidth"), col = "blue")
region_draw(summ_interval(d_con, 0.5, method = "percentile"), col = "red")
region_draw(summ_interval(d_con, 0.5, method = "sigma"), col = "green")

# Output interval is always inside input's support. Formally, next code
# should return interval from `-Inf` to `Inf`, but output is cut to be inside
# support.
summ_interval(d_con, level = 1, method = "sigma")

# To get vector output, use `unlist()`
unlist(summ_interval(d_con))
}
\seealso{
\code{\link[=summ_hdr]{summ_hdr()}} for computing of Highest Density Region, which can
summarize distribution with multiple intervals.

\link[=region]{region_*()} family of functions for working with \code{summ_interval()}
output.

Other summary functions: 
\code{\link{summ_center}()},
\code{\link{summ_classmetric}()},
\code{\link{summ_distance}()},
\code{\link{summ_entropy}()},
\code{\link{summ_hdr}()},
\code{\link{summ_moment}()},
\code{\link{summ_order}()},
\code{\link{summ_prob_true}()},
\code{\link{summ_pval}()},
\code{\link{summ_quantile}()},
\code{\link{summ_roc}()},
\code{\link{summ_separation}()},
\code{\link{summ_spread}()}
}
\concept{summary functions}
