% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/paleopop_simulator.R
\name{paleopop_simulator}
\alias{paleopop_simulator}
\title{Runs a customized population model simulation.}
\usage{
paleopop_simulator(inputs)
}
\arguments{
\item{inputs}{Nested list/object with named elements:
\describe{
  \item{\code{random_seed}}{Number to seed the random number generation for stochasticity.}
  \item{\code{time_steps}}{Number of simulation time steps.}
  \item{\code{years_per_step}}{Number of years per time step.}
  \item{\code{populations}}{Number of populations.}
  \item{\code{initial_abundance}}{Array of initial abundances for each population.}
  \item{\code{transition_rate}}{Rate of transition (or fecundity) between generations.}
  \item{\code{standard_deviation}}{Standard deviation applied to transition rates.}
  \item{\code{compact_decomposition}}{List containing a compact transposed 
  (Cholesky) decomposition \emph{matrix} (t_decomposition_compact_matrix) 
  and a corresponding \emph{map} of population indices 
  (t_decomposition_compact_map), as per 
  \code{\link[poems:SpatialCorrelation]{SpatialCorrelation}} class 
  attributes.}
  \item{\code{carrying_capacity}}{Matrix of carrying capacities (\emph{populations} rows by \emph{time_steps} columns).}
  \item{\code{density_dependence}}{Density dependence type ("competition", "logistic", or "ceiling").}
  \item{\code{growth_rate_max}}{Maximum growth rate (for "competition" or "logistic" density dependence).}
  \item{\code{harvest}}{Boolean for utilizing harvesting.}
  \item{\code{harvest_max}}{Proportion harvested per year (note: annual time scale - not generational).}
  \item{\code{harvest_g}}{The \emph{G} parameter in the harvest function.}
  \item{\code{harvest_z}}{The \emph{Z} parameter in the harvest function.}
  \item{\code{harvest_max_n}}{Maximum density per grid cell.}
  \item{\code{human_density}}{Matrix of human density (fraction) (\emph{populations} rows by \emph{time_steps} columns).}
  \item{\code{dispersal_data}}{List of data frames of non-zero dispersal rates and indices for constructing a compact dispersal matrix, and optional changing rates over time, as per class \code{\link[poems:DispersalGenerator]{DispersalGenerator}} \emph{dispersal_data} attribute.}
  \item{\code{dispersal_target_k}}{Target population carrying capacity threshold for density dependent dispersal.}
  \item{\code{abundance_threshold}}{Abundance threshold (that needs to be exceeded) for each population to persist.}
  \item{\code{occupancy_threshold}}{Threshold for the number of populations occupied (that needs to be exceeded) for all populations to persist.}
  \item{\code{results_selection}}{List of results selection from: "abundance", "ema", "extirpation", "harvested", "occupancy", "human_density".}
}}
}
\value{
Simulation results as a nested list (as selected):
  \describe{
    \item{\code{abundance}}{Matrix of simulation abundances (\emph{populations} rows by \emph{time_steps} columns).}
    \item{\code{ema}}{Matrix of expected minimum abundances (\emph{populations} rows by \emph{time_steps} columns).}
    \item{\code{extirpation}}{Array of extirpation times for each population.}
    \item{\code{harvested}}{Matrix of estimated individuals harvested (\emph{populations} rows by \emph{time_steps} columns).}
    \item{\code{occupancy}}{Array of number of populations occupied at each time-step.}
    \item{\code{human_density}}{Matrix of human densities, (\emph{populations} rows by \emph{time_steps} columns).}
  }
}
\description{
Simulates a population model customized for paleontological time-scales, optimized
for single-generation transitions and large populations, across multiple generations
and returns simulation results. Each generational time-step includes:
\enumerate{
  \item Density dependence calculations
  \item Environmental stochasticity calculations
  \item Generational transition calculations
  \item Harvest calculations
  \item Dispersal calculations
  \item Results collection
}
}
\examples{
library(raster)
library(poems)
# Ring Island example region
coordinates <- data.frame(x = rep(seq(-178.02, -178.06, -0.01), 5),
                          y = rep(seq(19.02, 19.06, 0.01), each = 5))
template_raster <- Region$new(coordinates = coordinates)$region_raster # full extent
sealevel_raster <- template_raster
template_raster[][c(7:9, 12:14, 17:19)] <- NA # make Ring Island
sealevel_raster[][c(7:9, 12:14, 17:18)] <- NA
raster_stack <- raster::stack(x = append(replicate(9, template_raster), sealevel_raster))
region <- PaleoRegion$new(template_raster = raster_stack)

# Model template
model_template <- PaleoPopModel$new(
region = region,
time_steps = 10,
years_per_step = 12, # years per generational time-step
standard_deviation = 0.1,
growth_rate_max = 0.6,
harvest = FALSE,
populations = region$region_cells,
initial_abundance = seq(9000, 0, -1000),
transition_rate = 1.0,
carrying_capacity = rep(1000, 17),
dispersal = (!diag(nrow = 17, ncol = 17))*0.05,
density_dependence = "logistic",
dispersal_target_k = 10,
occupancy_threshold = 1, 
abundance_threshold = 10,
results_selection = c("abundance")
)

# Simulations
results <- paleopop_simulator(model_template) # input as PaleoPopModel object
inputs <- model_template$get_attributes()
paleopop_simulator(inputs) # input as list of attributes

}
