#' Bhapkar's 1979 test for quasi-symmetry.
#'
#' Fits the quasi-symmetry model using WLS.
#' Bhapkar, V. P. (1979). On tests of marginal symmetry and quasi-symmetry in
#' two and three-dimensional contingency tables. Biometrics 35(2), 417-426.
#' @param n the matrix to be analyzed
#' @returns a list containing the chi-square and df.
#' @export
#' @examples
#' Bhapkar_quasi_symmetry(vision_data)
Bhapkar_quasi_symmetry <- function(n) {
  r <- nrow(n)
  N <- sum(n)
  n_bar <- rowSums(n)

  g <- matrix(0.0, nrow = r, ncol = r)
  a <- matrix(0.0, nrow=r, ncol=r)
  for (i in 1: r) {
    for (j in 1: r) {
      a_ij_inverse <- 1.0 / n[i, j] + 1.0 / n[j, i]
      a[i, j] <- 1.0 / a_ij_inverse
      g[i, j] <- log(n[i, j]) - log(n[j, i])
    }
  }

  h <- rep(0.0, r)
  for (i in 1:r) {
    if (i > 1) {
      for (k in 1:(i - 1)) {
        h[i] <- h[i] + g[k, i] * a[k , i]
      }
    }
    if (i == r) {
      next
    }
    for (j in (i + 1):r) {
      h[i] <- h[i] - g[i, j] * a[i, j]
    }
  }

  a_plus <- rowSums(a)
  a_star <- a[1:(r - 1), 1:(r - 1)]
  a_plus <- rowSums(a)
  d_star <- matrix(0.0, nrow=r - 1, ncol=r - 1)
  for (i in 1:(r - 1)) {
    d_star[i, i] <- a_plus[i]
  }

  h_star <- h[1:(r - 1)]
  delta_star <- solve(a_star - d_star, h_star)
  delta <- rep(0.0, r)
  delta[1:(r - 1)] <- delta_star
  # print(delta_star) # this is incorrect in the article. should be
  #   (0.31483924, 0.20872306, 0.09839753)

  s2 <- 0.0
  for (i in 1:(r - 1)) {
    for (j in (i + 1):r) {
      s2 <- s2 + a[i, j] * (g[i, j] - delta[i] + delta[j])^2
    }
  }

  df <- (r - 1) * (r - 2) / 2
  list(chisq=s2, df=df)
}
