% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/harsm.r
\name{harsmfit}
\alias{harsmfit}
\title{Experts only softmax regression under Harville model.}
\usage{
harsmfit(
  y,
  g,
  X,
  wt = NULL,
  eta0 = NULL,
  beta0 = NULL,
  normalize_wt = FALSE,
  reg_wt = NULL,
  reg_zero = 0,
  reg_power = NULL,
  reg_coef_idx = NULL,
  reg_standardize = FALSE,
  method = c("BFGS", "NR", "CG", "NM")
)
}
\arguments{
\item{y}{a vector of the ranked outcomes within each group. Only
the order within a group matters.}

\item{g}{a vector giving the group indices. Need not be integers, but
that is more efficient. Need not be sorted.
Must be the same length as \code{y}.}

\item{X}{a matrix of the independent variables. Must have as many rows
as the length of \code{y}.}

\item{wt}{an optional vector of the observation level weights. These must
be non-negative, otherwise an error is thrown. Note that the weight of
the last ranked outcome within a group is essentially ignored.
Must be the same length as \code{y}.}

\item{eta0}{an optional vector of the consensus odds. These are added to
the fit odds in odds space before the likelihood caclulation. If given,
then when the model is used to predict, similar consensus odds must be
given.
Must be the same length as \code{y}.}

\item{beta0}{an optional vector of the initial estimate of beta for
\sQuote{warm start} of the estimation procedure.
Must be the same length as number of columns in \code{X}.
Should only affect the speed of the computation, not the results.
Defaults to all zeroes.}

\item{normalize_wt}{if \code{TRUE}, we renormalize \code{wt}, if given,
to have mean value 1. Note that the default value has changed
since version 0.1.0 of this package. Moreover, non-normalized
weights can lead to incorrect inference. Use with caution.}

\item{reg_wt}{the multiplicative weight(s) of the regularization terms.
must be non-negative.
May be a scalar or vector, but will be recycled to the length of the \code{reg_coef_idx}.}

\item{reg_zero}{the \sQuote{zero} of the regularization terms. This would
usually be zero if you want to shrink to zero, but in some cases you may
with to shrink to 1 for example.
May be a scalar or vector, but will be recycled to the length of the \code{reg_coef_idx}.
If \code{NULL} is given, defaults to zeroes for beta terms,
and ones for gamma terms in a Henery model fit, all zeroes
for Harville model fits.}

\item{reg_power}{the power of the regularization terms, 2 for ridge
regression, 1 for lasso.}

\item{reg_coef_idx}{the index of the coefficient which the corresponding
regularization term is applied to. For the Harville model, the indices only refer to
the \eqn{\beta} coefficient vector. For the Henery model, the indices refer to the
\eqn{beta} coefficient vector and \eqn{\gamma} coefficient vector
concatenated together.
The other regularization parameters may be recycled as needed, but not the
\code{reg_coef_idx} values.}

\item{reg_standardize}{if true, the \code{reg_wt} are normalized, or
\sQuote{standardized} with respect to the standard deviation of the
corresponding columns of the design matrix. That is, the weight used
is the given weight \emph{times} the standard deviation of the corresponding
independent variable to the corresponding \code{reg_power}.
Only terms associated with the betas are so normalized.}

\item{method}{maximisation method, currently either 
   "NR" (for Newton-Raphson),
   "BFGS" (for Broyden-Fletcher-Goldfarb-Shanno), 
   "BFGSR" (for the BFGS algorithm implemented in \R),
   "BHHH" (for Berndt-Hall-Hall-Hausman), 
   "SANN" (for Simulated ANNealing), 
   "CG" (for Conjugate Gradients), 
   or "NM" (for Nelder-Mead).  
   Lower-case letters (such as "nr" for Newton-Raphson) are allowed.
   The default method is "NR" for unconstrained problems, and "NM" or
   "BFGS" for constrained problems, depending on if the \code{grad}
   argument was provided.  "BHHH" is a good alternative given the
   likelihood is returned observation-wise (see \code{\link[maxLik]{maxBHHH}}).

   Note that stochastic gradient ascent (SGA) is currently not supported
   as this method seems to be rarely used for maximum likelihood estimation.
 }
}
\value{
An object of class \code{harsm}, \code{maxLik}, and \code{linodds}.
}
\description{
An \dQuote{experts only} softmax fitting function for the Harville
model.
}
\details{
Given a number of events, indexed by group, and a vector \eqn{y} of
the ranks of each entry within that group, perform maximum likelihood
estimation under the softmax and proportional probability model.

The user can optionally supply a vector of \eqn{\eta_0}, which are
taken as the fixed, or \sQuote{consensus} odds. The estimation is
then conditional on these fixed odds.

Weighted estimation is supported.

The code relies on the likelihood function of \code{\link{harsmlik}},
and MLE code from \code{\link[maxLik]{maxLik}}.
}
\note{
The fit functions return an object of type \code{\link[maxLik]{maxLik}}
even when regularization penalties are applied; the statistical inference
functions are not valid when regularization is used. The user is warned.
}
\section{Regularization}{
The regularization term is of the form
\deqn{\sum_i w_i |\nu_{c_i} - z_i|^{p_i},}
where \eqn{w_i} are the \code{reg_wt} weights,
\eqn{z_i} are the \code{reg_zero} zeroes,
\eqn{p_i} are the \code{reg_power} powers,
and \eqn{c_i} are the \code{reg_coef_idx} coefficient indices.
Note that the coefficient indices can be repeated so that the
regularization term can include multiple contributions from one
coefficient element. This allows \sQuote{elasticnet} regularization.
The \eqn{\nu} here refer to the regression coefficients \eqn{\beta}
concatenated with the \eqn{\gamma} coefficients in the Henery model.
}

\examples{
nfeat <- 5
set.seed(1234)
g <- ceiling(seq(0.1,1000,by=0.1))
X <- matrix(rnorm(length(g) * nfeat),ncol=nfeat)
beta <- rnorm(nfeat)
eta <- X \%*\% beta
y <- rsm(eta,g)

mod0 <- harsmfit(y=y,g=g,X=X)
summary(mod0)
# now upweight finishers 1-5
modw <- harsmfit(y=y,g=g,X=X,wt=1 + as.numeric(y < 6))
summary(modw)
}
\references{
Harville, D. A. "Assigning probabilities to the outcomes of multi-entry competitions."
Journal of the American Statistical Association 68, no. 342 (1973): 312-316.
\doi{10.1080/01621459.1973.10482425}
}
\seealso{
the friendly interface \code{harsm},
the likelihood function, \code{\link{harsmlik}},
and the expected rank function (the inverse link), \code{\link{erank}}.
}
\author{
Steven E. Pav \email{shabbychef@gmail.com}
}
\keyword{fitting}
