% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sw.R
\name{swSpice}
\alias{swSpice}
\title{Seawater Spiciness}
\usage{
swSpice(
  salinity,
  temperature = NULL,
  pressure = NULL,
  longitude = NULL,
  latitude = NULL,
  eos = getOption("oceEOS", default = "gsw"),
  debug = getOption("oceDebug")
)
}
\arguments{
\item{salinity}{either salinity (PSU) (in which case \code{temperature} and
\code{pressure} must be provided) \emph{or} a \code{ctd} object (in which
case \code{salinity}, \code{temperature} and \code{pressure} are determined
from the object, and must not be provided in the argument list).}

\item{temperature}{\emph{in-situ} temperature (\eqn{^\circ}{deg}C) on the
ITS-90 scale; see \dQuote{Temperature units} in the documentation for
\code{\link[=swRho]{swRho()}}.}

\item{pressure}{Seawater pressure (dbar) (only used if \code{eos} is
\code{"gsw"}); see \dQuote{Details}..}

\item{longitude}{longitude of observation (only used if \code{eos} is
\code{"gsw"}; see \dQuote{Details}).}

\item{latitude}{latitude of observation (only used if \code{eos} is
\code{"gsw"}; see \dQuote{Details}).}

\item{eos}{Character value specifying the equation of state, either
\code{"unesco"} (for the Flament formulation, although this is not actually part
of UNESCO) or \code{"gsw"} for the Gibbs SeaWater formulation.}

\item{debug}{an integer specifying whether debugging information is
to be printed during the processing. This is a general parameter that
is used by many \code{oce} functions. Generally, setting \code{debug=0}
turns off the printing, while higher values suggest that more information
be printed. If one function calls another, it usually reduces the value of
\code{debug} first, so that a user can often obtain deeper debugging
by specifying higher \code{debug} values.}
}
\value{
Flament-formulated spice \eqn{kg/m^3} if \code{eos} is \code{"unesco"}
or surface-referenced GSW spiciness0 \eqn{kg/m^3} if \code{eos} is \code{"gsw"},
the latter provided by \code{\link[gsw:gsw_spiciness0]{gsw::gsw_spiciness0()}}, and hence aimed
at application within the top half-kilometre of the ocean.
}
\description{
Compute seawater "spice", a variable that is in some sense orthogonal to
density in TS space.  Larger spice values correspond to relative warm and
salty water, compared with smaller spice values. Two distinct variants exist.
If \code{eos="unesco"} then Flament's (2002) formulation is used. If \code{eos="gsw"}
then \code{\link[gsw:gsw_spiciness0]{gsw::gsw_spiciness0()}} is used to compute a newer variant that is part
of the Gibbs SeaWater formulation (McDougall and Krzysik, 2015). See the
\dQuote{Examples} section for a graphical illustration of the difference in a
typical coastal case.
}
\details{
If the first argument is a \code{ctd} object, then salinity, temperature and
pressure values are extracted from it, and used for the calculation. For
the \code{eos="gsw"} case, longitude and latitude are also extracted, because
these are required by \code{\link[gsw:gsw_spiciness0]{gsw::gsw_spiciness0()}}.

Roughly speaking, seawater with a high spiciness is relatively warm and
salty compared with less spicy water. Another interpretation is that spice
is a variable measuring distance orthogonal to isopycnal lines on TS
diagrams (if the diagrams are scaled to make the isopycnals run at 45
degrees). Note that pressure, longitude and latitude are all
ignored in the Flament definition.
}
\examples{
# Compare unesco and gsw formulations
library(oce)
data(ctd)
p <- ctd[["pressure"]]
U <- swSpice(ctd, eos = "unesco")
G <- swSpice(ctd, eos = "gsw")
xlim <- range(c(U, G), na.rm = TRUE)
ylim <- rev(range(p))
plot(U, p,
    xlim = xlim, ylim = ylim,
    xlab = "Measure of Spiciness", ylab = "Pressure (dbar)"
)
points(G, p, col = 2)
legend("topleft", col = 1:2, pch = 1, legend = c("unesco", "gsw"))

}
\references{
\enumerate{
\item Flament, P. \dQuote{A State Variable for Characterizing Water Masses and Their
Diffusive Stability: Spiciness.} Progress in Oceanography, Observations of the
1997-98 El Nino along the West Coast of North America, 54, no. 1
(July 1, 2002):493-501.
\doi{10.1016/S0079-6611(02)00065-4}
\item McDougall, Trevor J., and Oliver A. Krzysik. \dQuote{Spiciness.}
Journal of Marine Research 73, no. 5 (September 1, 2015): 141-52.
}
}
\seealso{
Other functions that calculate seawater spiciness: 
\code{\link{swSpiciness0}()},
\code{\link{swSpiciness1}()},
\code{\link{swSpiciness2}()}

Other functions that calculate seawater properties: 
\code{\link{T68fromT90}()},
\code{\link{T90fromT48}()},
\code{\link{T90fromT68}()},
\code{\link{computableWaterProperties}()},
\code{\link{locationForGsw}()},
\code{\link{swAbsoluteSalinity}()},
\code{\link{swAlpha}()},
\code{\link{swAlphaOverBeta}()},
\code{\link{swBeta}()},
\code{\link{swCSTp}()},
\code{\link{swConservativeTemperature}()},
\code{\link{swDepth}()},
\code{\link{swDynamicHeight}()},
\code{\link{swLapseRate}()},
\code{\link{swN2}()},
\code{\link{swPressure}()},
\code{\link{swRho}()},
\code{\link{swRrho}()},
\code{\link{swSCTp}()},
\code{\link{swSR}()},
\code{\link{swSTrho}()},
\code{\link{swSigma}()},
\code{\link{swSigma0}()},
\code{\link{swSigma1}()},
\code{\link{swSigma2}()},
\code{\link{swSigma3}()},
\code{\link{swSigma4}()},
\code{\link{swSigmaT}()},
\code{\link{swSigmaTheta}()},
\code{\link{swSoundAbsorption}()},
\code{\link{swSoundSpeed}()},
\code{\link{swSpecificHeat}()},
\code{\link{swSpiciness0}()},
\code{\link{swSpiciness1}()},
\code{\link{swSpiciness2}()},
\code{\link{swSstar}()},
\code{\link{swTFreeze}()},
\code{\link{swTSrho}()},
\code{\link{swThermalConductivity}()},
\code{\link{swTheta}()},
\code{\link{swViscosity}()},
\code{\link{swZ}()}
}
\author{
Dan Kelley coded this, merely an interface to the code described
by references 1 and 2.
}
\concept{functions that calculate seawater properties}
\concept{functions that calculate seawater spiciness}
