
#' Reshape score matrix for ggplot2.
#'
#' Manually reshapes the data into Long format and adds a refinement group factor.
#'
#' @param score_df (Dataframe) A scored code matrix as generated by [score_codes()].
#' @param refinements Either a list object generated by [import_field_notes()],
#'    or an Integer vector that lists when (in terms of interview sequence)
#'    refinements were made to the interview questions. For example, `c(10, 15)`
#'    means that interview questions were revised twice: First **before** the
#'    10th interview, and then again **before** the 15th interview.
#'
#' @return A dataframe.
#'
#' @md
reshape_for_plots <- function(score_df, refinements = integer(0)) {
    long_df <-
        data.frame(
            Interview  = score_df$itvw_seq,
            Code       = c(rep("Novel", nrow(score_df)), rep("Duplicate", nrow(score_df))),
            n          = c(score_df$n_novel,             score_df$n_duplicate),
            Proportion = c(score_df$prop_novel,          score_df$prop_duplicate),
            CumSum     = c(score_df$cumsum_novel,        rep(NA_real_, nrow(score_df))),
            Refinement = cut(score_df$itvw_seq,
                             breaks = c(-Inf, refinements, Inf),
                             right = FALSE,
                             include.lowest = TRUE)
        )

    # Omit groups that have a count of 0. This removes an issue where the
    # "Novel" group was being drawn with a 2-pixel line at Interview #1,
    # especially visible if the `stroke_duplicate` colour was a prominent one.
    long_df <- long_df[long_df$n > 0, ]

    return(long_df)
}
