% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/all_generic.R
\name{perm_test}
\alias{perm_test}
\alias{perm_test.pca}
\alias{perm_test.cross_projector}
\alias{perm_test.discriminant_projector}
\alias{perm_test.multiblock_biprojector}
\title{Generic Permutation-Based Test}
\arguments{
\item{x}{A fitted model object (e.g. \code{pca}, \code{cross_projector}, \code{discriminant_projector}, \code{multiblock_biprojector}).}

\item{...}{Additional arguments passed down to \code{shuffle_fun} or \code{measure_fun} (if applicable).
Note: For \code{multiblock} methods, \code{Xlist}, \code{comps}, \code{alpha}, and \code{use_rspectra} (for biprojector) are handled as direct named arguments, not via \code{...}.}

\item{X}{(Used by \code{pca}, \code{cross_projector}, \code{discriminant_projector}) The original primary data matrix used to fit \code{x}. Ignored by the \code{multiblock_biprojector} method.}

\item{Y}{(Used by \code{cross_projector}) The secondary data block (n x pY). Ignored by other methods.}

\item{Xlist}{(Used by \code{multiblock_biprojector} [optional, default \code{NULL}] and \code{multiblock_projector} [required]) List of data blocks.}

\item{nperm}{Integer number of permutations (Default: 1000 for PCA, 500 for multiblock methods, 100 otherwise).}

\item{measure_fun}{(Optional; Used by \code{pca}, \code{cross_projector}, \code{discriminant_projector}, \code{multiblock_projector}) A function for computing the statistic(s) of interest. Ignored by \code{multiblock_biprojector}. Signature/default varies by method (see Details).}

\item{shuffle_fun}{(Optional; Used by all methods) A function for permuting the data appropriately. Signature/default varies by method (see Details).}

\item{fit_fun}{(Optional; Used by \code{cross_projector}, \code{discriminant_projector}) A function for re-fitting a new model. Ignored by PCA and multiblock methods. Signature/default varies by method (see Details).}

\item{stepwise}{(Used by \code{pca}) Logical indicating if sequential testing (P3 projection) should be performed. Default \code{TRUE}. (The multiblock methods also perform sequential testing based on \code{alpha} and \code{comps}, but this argument is ignored). Ignored by other methods.}

\item{parallel}{(Used by all methods) Logical; if \code{TRUE}, attempt parallel execution via \code{future.apply::future_lapply}.}

\item{alternative}{(Used by all methods) Character string for the alternative hypothesis: "greater" (default), "less", or "two.sided".}

\item{alpha}{(Used by \code{pca}, \code{multiblock_biprojector}, \code{multiblock_projector}) Significance level for sequential stopping rule (default 0.05). Passed directly as a named argument to these methods.}

\item{comps}{(Used by \code{pca}, \code{multiblock_biprojector}, \code{multiblock_projector}) Maximum number of components to test sequentially (default 4). Passed directly as a named argument to these methods.}

\item{use_svd_solver}{(Used by \code{pca}) Optional string specifying the SVD solver (default "fast").}

\item{use_rspectra}{(Used by \code{multiblock_biprojector}) Logical indicating whether to use RSpectra for eigenvalue calculation (default \code{TRUE}). Passed directly as a named argument.}

\item{predict_method}{(Used by \code{discriminant_projector}) Prediction method (\code{"lda"} or \code{"euclid"}) used by the default measure function (default "lda").}
}
\value{
The structure of the return value depends on the method:
\describe{
\item{\strong{\code{cross_projector}} and \strong{\code{discriminant_projector}}:}{
Returns an object of class \code{perm_test}, a list containing: \code{statistic}, \code{perm_values}, \code{p.value}, \code{alternative}, \code{method}, \code{nperm}, \code{call}.}
\item{\strong{\code{pca}}, \strong{\code{multiblock_biprojector}}, and \strong{\code{multiblock_projector}}:}{
Returns an object inheriting from \code{perm_test} (classes \code{perm_test_pca}, \code{perm_test_multiblock}, or \code{perm_test} respectively for multiblock_projector),
a list containing: \code{component_results} (data frame with observed stat, pval, CIs per component), \code{perm_values} (matrix of permuted stats), \code{alpha} (if applicable), \code{alternative}, \code{method}, \code{nperm} (vector of successful permutations per component), \code{call}.}
}
}
\description{
This generic function implements a permutation-based test to assess the significance
of components or statistics in a fitted model. The actual procedure depends on
the method defined for the specific model class. Typical usage:
}
\details{
\enumerate{
\item Shuffle or permute the data in a way that breaks the structure of interest
(e.g., shuffle labels for supervised methods, shuffle columns/rows for unsupervised).
\item Re-fit or re-project the model on the permuted data. Depending on the class,
this can be done via a \code{fit_fun} or a class-specific approach.
\item Measure the statistic of interest (e.g., variance explained, classification accuracy, canonical correlation).
\item Compare the distribution of permuted statistics to the observed statistic to compute an empirical p-value.
}

S3 methods define the specific defaults and required signatures for the functions
involved in shuffling, fitting, and measuring.

This function provides a framework for permutation testing in various multivariate models.
The specific implementation details, default functions, and relevant arguments vary by method.

\strong{PCA Method (\code{perm_test.pca}):}
Relevant arguments: \code{X}, \code{nperm}, \code{measure_fun}, \code{shuffle_fun}, \code{stepwise}, \code{parallel}, \code{alternative}, \code{alpha}, \code{comps}, \code{use_svd_solver}, \code{...}. Assesses significance of variance explained by each PC (Vitale et al., 2017). Default statistic: F_a. Default shuffle: column-wise. Default uses P3 projection and sequential stopping with \code{alpha}.

\strong{Cross Projector Method (\code{perm_test.cross_projector}):}
Relevant arguments: \code{X}, \code{Y}, \code{nperm}, \code{measure_fun}, \code{shuffle_fun}, \code{fit_fun}, \code{parallel}, \code{alternative}, \code{...}. Tests the X-Y relationship. Default statistic: \code{x2y.mse}. Default shuffle: rows of Y. Default fit: \code{stats::cancor}.

\strong{Discriminant Projector Method (\code{perm_test.discriminant_projector}):}
Relevant arguments: \code{X}, \code{nperm}, \code{measure_fun}, \code{shuffle_fun}, \code{fit_fun}, \code{predict_method}, \code{parallel}, \code{alternative}, \code{...}. Tests class separation. Default statistic: prediction accuracy. Default shuffle: labels. Default fit: \code{MASS::lda}.

\strong{Multiblock Bi-Projector Method (\code{perm_test.multiblock_biprojector}):}
Relevant arguments: \code{Xlist} (optional), \code{nperm}, \code{shuffle_fun}, \code{parallel}, \code{alternative}, \code{alpha}, \code{comps}, \code{use_rspectra}, \code{...}. Tests consensus using fixed internal statistic (eigenvalue) on scores for each component. The statistic is the leading eigenvalue of the covariance matrix of block scores for a given component (T^T, where T columns are scores of block \emph{b} on component \emph{k}). By default, it shuffles rows within each block independently (either from \code{Xlist} if provided via \code{...}, or using the internally stored scores). It performs sequential testing for components specified by \code{comps} using the stopping rule defined by \code{alpha} (both passed via \code{...}).

\strong{Multiblock Projector Method (\code{perm_test.multiblock_projector}):}
Relevant arguments: \code{Xlist} (required), \code{nperm}, \code{measure_fun}, \code{shuffle_fun}, \code{parallel}, \code{alternative}, \code{alpha}, \code{comps}, \code{...}. Tests consensus using \code{measure_fun} (default: mean abs corr) on scores projected from \code{Xlist} using the original model \code{x}. Does not refit.
}
\examples{
# PCA Example
data(iris)
X_iris <- as.matrix(iris[,1:4])
mod_pca <- pca(X_iris, ncomp=4, preproc=center()) # Ensure centering

# Test first 3 components sequentially (faster with more nperm)
# Ensure a future plan is set for parallel=TRUE, e.g., future::plan("multisession")
res_pca <- perm_test(mod_pca, X_iris, nperm=50, comps=3, parallel=FALSE)
print(res_pca)

# PCA Example with row shuffling (tests different null hypothesis)
row_shuffle <- function(dat, ...) dat[sample(nrow(dat)), ]
res_pca_row <- perm_test(mod_pca, X_iris, nperm=50, comps=3,
                         shuffle_fun=row_shuffle, parallel=FALSE)
print(res_pca_row)

}
\references{
Buja, A., & Eyuboglu, N. (1992). Remarks on parallel analysis. \emph{Multivariate Behavioral Research}, 27(4), 509-540. (Relevant for PCA permutation concepts)

Vitale, R., Westerhuis, J. A., Næs, T., Smilde, A. K., de Noord, O. E., & Ferrer, A. (2017).
Selecting the number of factors in principal component analysis by permutation testing—
Numerical and practical aspects. \emph{Journal of Chemometrics}, 31(10), e2937.
\doi{10.1002/cem.2937} (Specific to \code{perm_test.pca})
}
\seealso{
\code{\link{pca}}, \code{\link{cross_projector}}, \code{\link{discriminant_projector}},
\code{\link{multiblock_biprojector}},
\code{\link{measure_interblock_transfer_error}}
}
\concept{perm_test}
