% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/count_binomial.R
\name{count_binom}
\alias{count_binom}
\title{Count How Many Samples Satisfy Linear Inequalities (Binomial)}
\usage{
count_binom(
  k,
  n,
  A,
  b,
  V,
  map,
  prior = c(1, 1),
  M = 10000,
  steps,
  start,
  cmin = 0,
  maxiter = 500,
  burnin = 5,
  progress = TRUE,
  cpu = 1
)
}
\arguments{
\item{k}{vector of observed response frequencies.}

\item{n}{the number of choices per item type.
If \code{k=n=0}, Bayesian inference is relies on the prior distribution only.}

\item{A}{a matrix with one row for each linear inequality constraint and one
column for each of the free parameters. The parameter space is defined
as all probabilities \code{x} that fulfill the order constraints  \code{A*x <= b}.}

\item{b}{a vector of the same length as the number of rows of \code{A}.}

\item{V}{a matrix of vertices (one per row) that define the polytope of
admissible parameters as the convex hull over these points
(if provided, \code{A} and \code{b} are ignored).
Similar as for \code{A}, columns of \code{V} omit the last value for each
multinomial condition (e.g., a1,a2,a3,b1,b2 becomes a1,a2,b1).
Note that this method is comparatively slow since it solves linear-programming problems
to test whether a point is inside  a polytope (Fukuda, 2004) or to run the Gibbs sampler.}

\item{map}{optional: numeric vector of the same length as \code{k} with integers
mapping the frequencies \code{k} to the free parameters/columns of \code{A}/\code{V},
thereby allowing for equality constraints (e.g., \code{map=c(1,1,2,2)}).
Reversed probabilities \code{1-p} are coded by negative integers.
Guessing probabilities of .50 are encoded by zeros. The default assumes
different parameters for each item type: \code{map=1:ncol(A)}}

\item{prior}{a vector with two positive numbers defining the shape parameters
of the beta prior distributions for each binomial rate parameter.}

\item{M}{number of posterior samples drawn from the encompassing model}

\item{steps}{an integer vector that indicates the row numbers at which the matrix \code{A}
is split for a stepwise computation of the Bayes factor (see details).
\code{M} can be a vector with the number of samples drawn
in each step from the (partially) order-constrained models  using Gibbs sampling.
If \code{cmin>0}, samples are drawn for each step until \code{count[i]>=cmin}.}

\item{start}{only relevant if \code{steps} is defined or \code{cmin>0}:
a vector with starting values in the interior of the polytope.
If missing, an approximate maximum-likelihood estimate is used.}

\item{cmin}{if \code{cmin>0}: minimum number of counts per step in the automatic stepwise procedure.
If \code{steps} is not defined, \code{steps=c(1,2,3,4,...)} by default.}

\item{maxiter}{if \code{cmin>0}: maximum number of sampling runs in the automatic stepwise procedure.}

\item{burnin}{number of burnin samples per step that are discarded. Since the
maximum-likelihood estimate is used as a start value (which is updated for each step in
the stepwise procedure in \code{count_multinom}), the \code{burnin}
number can be smaller than in other MCMC applications.}

\item{progress}{whether a progress bar should be shown (if \code{cpu=1}).}

\item{cpu}{either the number of CPUs used for parallel sampling, or a parallel
cluster  (e.g., \code{cl <- parallel::makeCluster(3)}).
All arguments of the function call are passed directly to each core,
and thus the total number of samples is \code{M*number_cpu}.}
}
\value{
a matrix with the columns
\itemize{
    \item\code{count}: number of samples in polytope / that satisfy order constraints
    \item\code{M}: the  total number of samples in each step
    \item\code{steps}: the \code{"steps"} used to sample from the polytope
        (i.e., the row numbers of \code{A} that were considered  stepwise)
}
with the attributes:
\itemize{
   \item\code{proportion}: estimated probability that samples are in polytope
   \item\code{se}: standard error of probability estimate
   \item\code{const_map}: logarithm of the binomial constants that
          have to be considered due to equality constraints
}
}
\description{
Draws prior/posterior samples for product-binomial data and counts how many samples are
inside the convex polytope defined by
(1) the inequalities \code{A*x <= b} or
(2) the convex hull over the vertices \code{V}.
}
\details{
Counts the number of random samples drawn from beta distributions that satisfy
the convex, linear-inequalitiy constraints. The function is useful to compute
the encompassing Bayes factor for testing inequality-constrained models
(see \code{\link{bf_binom}}; Hojtink, 2011).

The stepwise computation of the Bayes factor proceeds as follows:
If the steps are defined as \code{steps=c(5,10)}, the BF is computed in three steps by comparing:
(1) Unconstrained model vs. inequalities in \code{A[1:5,]};
(2) use posterior based on inequalities in \code{A[1:5,]} and check inequalities \code{A[6:10,]};
(3) sample from A[1:10,] and check inequalities in \code{A[11:nrow(A),]} (i.e., all inequalities).
}
\examples{
### a set of linear order constraints:
### x1 < x2 < .... < x6 < .50
A <- matrix(
  c(
    1, -1, 0, 0, 0, 0,
    0, 1, -1, 0, 0, 0,
    0, 0, 1, -1, 0, 0,
    0, 0, 0, 1, -1, 0,
    0, 0, 0, 0, 1, -1,
    0, 0, 0, 0, 0, 1
  ),
  ncol = 6, byrow = TRUE
)
b <- c(0, 0, 0, 0, 0, .50)

### check whether a specific vector is inside the polytope:
A \%*\% c(.05, .1, .12, .16, .19, .23) <= b


### observed frequencies and number of observations:
k <- c(0, 3, 2, 5, 3, 7)
n <- rep(10, 6)

### count prior samples and compare to analytical result
prior <- count_binom(0, 0, A, b, M = 5000, steps = 1:4)
prior # to get the proportion: attr(prior, "proportion")
(.50)^6 / factorial(6)

### count posterior samples + get Bayes factor
posterior <- count_binom(k, n, A, b, M = 2000, steps = 1:4)
count_to_bf(posterior, prior)

### automatic stepwise algorithm
prior <- count_binom(0, 0, A, b, M = 500, cmin = 200)
posterior <- count_binom(k, n, A, b, M = 500, cmin = 200)
count_to_bf(posterior, prior)

}
\references{
Hoijtink, H. (2011). Informative Hypotheses: Theory and Practice for Behavioral and Social Scientists. Boca Raton, FL: Chapman & Hall/CRC.

Fukuda, K. (2004). Is there an efficient way of determining whether a given point q is in the convex hull of a given finite set S of points in Rd? Retrieved from \url{https://www.cs.mcgill.ca/~fukuda/soft/polyfaq/node22.html}
}
\seealso{
\code{\link{bf_binom}}, \code{\link{count_multinom}}
}
