% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mrct_functions.R
\name{mrct.sparse}
\alias{mrct.sparse}
\title{Sparse minimum regularized covariance trace estimator}
\usage{
mrct.sparse(
  data,
  nbasis = dim(data)[2],
  new.p = dim(data)[2],
  h = 0.75,
  alpha = 0.01,
  initializations = 5,
  seed = 123,
  scaling.iterations = 10,
  scaling.tolerance = 10^(-4),
  criterion = "sum",
  sum.percentage = 0.75
)
}
\arguments{
\item{data}{Numeric matrix of a functional data set for which the esimator has to be calculated. Each row contains an observation. They are assumed to be observed on the same (probably sparse) regular grid. The number of grid points must be at least \code{nbasis}.}

\item{nbasis}{Integer. Number of B-spline basis functions for smoothing. The basis will be of order \eqn{4} and therefore, cannot contain less than \eqn{4} functions. The default value will be set to \code{dim(data)[2]}. i.e. the number of time points with a maximum of \eqn{15}.}

\item{new.p}{Integer. Length of the grid of the smoothed curves. The resulting grid will be an equidistant partition of \verb{[rangeval[1],rangeval[length(rangeval)]]}. Default value is \code{dim(data)[2]}}

\item{h}{Numeric value between \eqn{0.5} and \eqn{1}. Ratio of the data which the estimator is based on. Default is set to \eqn{0.75}, i.e. \eqn{75\%} of the data will be used for the estimator.}

\item{alpha}{Numeric (default is \eqn{0.01}). Tikhonov regularization parameter \eqn{\alpha}.}

\item{initializations}{Integer (default is \eqn{5}). Number of random initial subsets.}

\item{seed}{Integer (default is \eqn{123}). Random seed for reproducibility.}

\item{scaling.iterations}{Integer (default is \eqn{5}). The maximum number of times \eqn{k_1} is re-scaled if the error between subsequent
scalingparameters does not fall below \code{scaling.tolerance}.}

\item{scaling.tolerance}{Numeric (default is \eqn{10^{-4}}). The error tolerance for re-scaling. If the error falls below this value, the re-scaling procedure stops.}

\item{criterion}{Character. Criterion based on which the optimal subset is chosen among the final subsets. Possible options are: "\code{cluster}" and the default "\code{sum}".}

\item{sum.percentage}{Numeric value between \eqn{0.5} and \eqn{1}. Corresponding to the "\code{sum}" criterion. Determines the fraction of observations up to which the sum over the sorted functional Mahalanobis distances is calculated (in ascending order). Default is set to \eqn{0.75}, i.e. the sum of the smallest \eqn{75\%} of Mahalanobis distances is calculated. If outliers are present, this value should not be to high, in order not to include any outlying curves.}
}
\value{
A list with two entries
\item{mrct.output}{List. The same output as the function \code{\link[=mrct]{mrct()}}. For more details, see there.}
\item{data.smooth}{Numeric matrix. Collection of the smoothed curves of \code{data} with \code{dim(data)[1]} rows and \code{new.p} columns. Each row corresponds to one observation.}
}
\description{
Robust outlier detection for sparse functional data as a generalization of the \code{minimum} \code{regularized} \code{covariance} \code{trace} (MRCT) estimator \insertCite{oguamalam2023minimum}{mrct}. At first the observations are smoothed
by a B-spline basis and afterwards the MRCT algorithm is performed with the matrix of basis coefficients.
}
\examples{
# Fix seed for reproducibility
set.seed(123)

# Sample outlying indices
cont.ind <- sample(1:50,size=10)

# Generate 50 sparse curves on the interval [0,1] at 10 timepoints with 20\% outliers
y <- mrct.rgauss(x.grid=seq(0,1,length.out=10), N=50, model=1,
                 outliers=cont.ind, method="linear")

# Visualize curves (regular curves grey, outliers black)
colormap <- rep("grey",50); colormap[cont.ind] <- "black"
matplot(x = seq(0,1,length.out=10), y = t(y), type="l", lty="solid",
        col=colormap, xlab="t",ylab="")

# Run sparse MRCT
sparse.mrct.y <- mrct.sparse(data = y, nbasis = 10, h = 0.75, new.p = 50,
                             alpha = 0.1, initializations = 10, criterion = "sum" )

# Visualize smoothed functions
matplot(x=seq(0,1,length.out=50), y=t(sparse.mrct.y$data.smooth),
        type="l", lty="solid", col=colormap, xlab="t", ylab="")

# Visualize alpha-Mahalanobis distance with cutoff (horizontal black line)
# Colors correspond to simulated outliers, shapes to estimated (sparse MRCT) ones
# (circle regular and triangle irregular curves)
shapemap <- rep(1,50); shapemap[sparse.mrct.y$mrct.output$theoretical.w] <- 2
plot(x = 1:50, y = sparse.mrct.y$mrct.output$aMHD.w, col=colormap, pch = shapemap,
     xlab = "Index", ylab = expression(alpha*"-MHD"))
abline(h = sparse.mrct.y$mrct.output$quant.w)

# If you dont have any information on possible outliers,
# alternatively you could use the S3 method plot.mrctsparse()
mrct.sparse.plot(mrct.sparse.object = sparse.mrct.y)
}
\references{
\insertRef{oguamalam2023minimum}{mrct}.
}
