% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/FilterEnsemble.R
\name{mlr_filters_ensemble}
\alias{mlr_filters_ensemble}
\alias{FilterEnsemble}
\title{Filter Ensemble}
\format{
\code{\link[R6:R6Class]{R6Class}} object inheriting from \code{\link[mlr3filters:Filter]{Filter}}.
}
\description{
\code{FilterEnsemble} aggregates several \code{\link[mlr3filters:Filter]{Filter}}s by averaging their scores
(or ranks) with user-defined weights. Each wrapped filter is evaluated on the supplied task,
and the resulting feature scores are combined feature-wise by a convex combination determined
through the \code{weights} parameter. This allows leveraging complementary inductive biases of
multiple filters without committing to a single criterion. The concept was introduced by
Binder et al. (2020). This implementation follows the idea but leaves the exact choice of
weights to the user.
}
\section{Construction}{


\if{html}{\out{<div class="sourceCode">}}\preformatted{FilterEnsemble$new(filters)
}\if{html}{\out{</div>}}
\itemize{
\item \code{filters} :: \code{list} of \code{\link[mlr3filters:Filter]{Filter}}\cr
Filters that are evaluated and aggregated. Each filter must be cloneable and support the
task type and feature types of the ensemble. The ensemble identifier defaults to the wrapped
filter ids concatenated by \code{"."}.
}
}

\section{Parameters}{

\itemize{
\item \code{weights} :: \code{numeric()}\cr
Required non-negative weights, one for each wrapped filter, with at least one strictly positive value.
Values are used as given when calculating the weighted mean. If named, names must match the wrapped filter ids.
\item \code{rank_transform} :: \code{logical(1)}\cr
If \code{TRUE}, ranks of individual filter scores are used instead of the raw scores. Initialized to \code{FALSE}.
\item \code{filter_score_transform} :: \code{function}\cr
Function to be applied to the vector of individual filter scores after they were potentially transformed by
\code{rank_transform} but before weighting and aggregation. Initialized to \code{identity}.
\item \code{aggregator} :: \code{function}\cr
Function to aggregate the (potentially transformed) and weighted filter scores across filters. Must take
arguments \code{w} for weights and \code{na.rm}, the latter of which is always set to \code{TRUE}. Defaults to \code{\link[stats:weighted.mean]{stats::weighted.mean}}.
\item \code{result_score_transform} :: \code{function}\cr
Function to be applied to the vector of aggregated scores after they were potentially transformed by \code{rank_transform} and/or
\code{filter_score_transform}. Initialized to \code{identity}.
}

Parameters of wrapped filters are available via \verb{$param_set} and can be referenced using
the wrapped filter id followed by \code{"."}, e.g. \code{"variance.na.rm"}.
}

\section{Fields}{

\itemize{
\item \verb{$wrapped} :: named \code{list} of \code{\link[mlr3filters:Filter]{Filter}}\cr
Read-only access to the wrapped filters.
}
}

\section{Methods}{

\itemize{
\item \code{get_weights_search_space(weights_param_name = "weights", normalize_weights = "uniform", prefix = "w")}\cr
(\code{character(1)}, \code{character(1)}, \code{character(1)}) -> \code{\link[paradox:ParamSet]{ParamSet}}\cr
Construct a \code{\link[paradox:ParamSet]{ParamSet}} describing a weight search space.
\item \code{get_weights_tunetoken(normalize_weights = "uniform")}\cr
(\code{character(1)}) -> \code{\link[paradox:to_tune]{TuneToken}}\cr
Shortcut returning a \code{\link[paradox:to_tune]{TuneToken}} for tuning the weights.
\item \code{set_weights_to_tune(normalize_weights = "uniform")}\cr
(\code{character(1)}) -> \code{self}\cr
Convenience wrapper that stores the \code{TuneToken} returned by
\code{get_weights_tunetoken()} in \verb{$param_set$values$weights}.
}
}

\section{Internals}{

All wrapped filters are called with \code{nfeat} equal to the number of features to ensure that
complete score vectors are available for aggregation.
Scores are combined per feature by computing a weighted aggregation of transformed (default: \code{identity})
scores or ranks. Additionally, the final scores may also be transformed (default: \code{identity}).

The order of transformations is as follows:
\enumerate{
\item \verb{$calculate} the filter's scores for all features;
\item If \code{rank_transform} is \code{TRUE}, convert filter scores to ranks;
\item Apply \code{filter_score_transform} to the scores / ranks;
\item Calculate the weighted aggregation across all filters using \code{aggregator};
\item Potentially apply \code{result_score_transform} to the vector of scores for each feature aggreagted across filters.
}
}

\section{References}{

Binder M, Moosbauer J, Thomas J, Bischl B (2020).
\dQuote{Multi-objective hyperparameter tuning and feature selection using filter ensembles.}
In \emph{Proceedings of the 2020 Genetic and Evolutionary Computation Conference}, 471--479.
\doi{10.1145/3377930.3389815}.
}

\examples{
\dontshow{if (mlr3misc::require_namespaces("mlr3filters", quietly = TRUE)) withAutoprint(\{ # examplesIf}
library("mlr3")
library("mlr3filters")

task = tsk("sonar")

filter = flt("ensemble",
  filters = list(FilterVariance$new(), FilterAUC$new()))
filter$param_set$values$weights = c(variance = 0.5, auc = 0.5)
filter$calculate(task)
head(as.data.table(filter))

# Weighted median as aggregator
filter$param_set$set_values(aggregator = function(x, w, na.rm) {
  if (na.rm) x <- x[!is.na(x)]
  o <- order(x)
  x <- x[o]
  w <- w[o]
  x[match(TRUE, which(cumsum(w) >= sum(w) / 2))]
})
filter$calculate(task)
head(as.data.table(filter))

# Aggregate reciprocal ranking
filter$param_set$set_values(rank_transform = TRUE, 
  filter_score_transform = function(x) 1 / x, 
  result_score_transform = function(x) rank(1 / x, ties.method = "average"))
filter$calculate(task)
head(as.data.table(filter))
\dontshow{\}) # examplesIf}
}
