#' @title Class Weights for Sample Weighting
#'
#' @usage NULL
#' @name mlr_pipeops_classweights
#' @format [`R6Class`][R6::R6Class] object inheriting from [`PipeOpTaskPreproc`]/[`PipeOp`].
#'
#' @description
#' Adds a class-dependent sample weights column to a [`Task`][mlr3::Task], allowing
#' [`Learner`][mlr3::Learner]s and [`Measure`][mlr3::Measure]s to weight observations
#' differently during training and evaluation. 
#'
#' Weights are assigned per observation based on the target class and can be written
#' to the `"weights_learner"` column, the `"weights_measure"` column, both, or neither.
#'
#' Only binary classification tasks ([`TaskClassif`][mlr3::TaskClassif]) are supported.
#'
#' Note: By default, all weights are set to `1`. To obtain a meaningful effect, the
#' `minor_weight` parameter must be adjusted.
#' 
#' See [`PipeOpClassWeightsEx`] for an extended version of this `PipeOp` which can 
#' handle multiclass classification tasks and offers several methods for automatically 
#' determining weights.
#'
#' @section Construction:
#' ```
#' PipeOpClassWeights$new(id = "classweights", param_vals = list())
#' ```
#'
#' * `id` :: `character(1)` \cr
#'   Identifier of the resulting  object, default `"classweights"`
#' * `param_vals` :: named `list` \cr
#'   List of hyperparameter settings, overwriting the hyperparameter settings that would otherwise be set during construction. Default `list()`.
#'
#' @section Input and Output Channels:
#' Input and output channels are inherited from [`PipeOpTaskPreproc`]. Instead of a [`Task`][mlr3::Task], a
#' [`TaskClassif`][mlr3::TaskClassif] is used as input and output during training and prediction.
#'
#' The output during training is the input [`Task`][mlr3::Task] with an added weights column according to the target class.
#' The output during prediction is the unchanged input.
#'
#' @section State:
#' The `$state` is a named `list` with the `$state` elements inherited from [`PipeOpTaskPreproc`].
#'
#' @section Parameters:
#' The parameters are the parameters inherited from [`PipeOpTaskPreproc`]; however, the `affect_columns` parameter is *not* present. Further parameters are:
#' * `minor_weight` :: `numeric(1)` \cr
#'   Weight given to samples of the minor class. Major class samples have weight `1`. Initialized to `1`.
#' * `weights_learner` :: `logical(1)` \cr
#'   Whether the created weights should be stored as a `weights_learner` column or not. Initialized to `TRUE`.
#' * `weights_measure` :: `logical(1)` \cr
#'   Whether the created weights should be stored as a `weights_measure` column or not. Initialized to `FALSE`.
#'
#' @section Internals:
#' Adds a `.WEIGHTS` column to the [`Task`][mlr3::Task], which is removed from the feature role and mapped to the requested weight roles.
#' There will be a naming conflict if this column already exists and is *not* a weight column already. For potentially pre-existing weight columns, 
#' the weight column role gets dropped, but they remain in the [`DataBackend`][mlr3::DataBackend] of the `Task`.
#' The [`Learner`][mlr3::Learner] must support weights for this `PipeOp` to have an effect.
#'
#' @section Fields:
#' Only fields inherited from [`PipeOp`].
#'
#' @section Methods:
#' Only methods inherited from [`PipeOpTaskPreproc`]/[`PipeOp`].
#'
#' @family PipeOps
#' @template seealso_pipeopslist
#' @include PipeOpTaskPreproc.R
#' @export
#' @examples
#' library("mlr3")
#'
#' task = tsk("spam")
#' opb = po("classweights")
#'
#' # task weights
#' if ("weights_learner" %in% names(task)) {
#'   task$weights_learner  # recent mlr3-versions
#' } else {
#'   task$weights  # old mlr3-versions
#' }
#'
#' # double the instances in the minority class (spam)
#' opb$param_set$values$minor_weight = 2
#' result = opb$train(list(task))[[1L]]
#' if ("weights_learner" %in% names(result)) {
#'   result$weights_learner  # recent mlr3-versions
#' } else {
#'   result$weights  # old mlr3-versions
#' }
PipeOpClassWeights = R6Class("PipeOpClassWeights",
  inherit = PipeOpTaskPreproc,

  public = list(
    initialize = function(id = "classweights", param_vals = list()) {
      ps = ps(
        minor_weight = p_dbl(init = 1, lower = 0, upper = Inf, tags = "train"),
        weights_learner = p_lgl(init = TRUE, tags = c("train", "weights_indicator", "required")),
        weights_measure = p_lgl(init = FALSE, tags = c("train", "weights_indicator", "required"))
      )
      super$initialize(id, param_set = ps, param_vals = param_vals, can_subset_cols = FALSE, task_type = "TaskClassif", tags = "imbalanced data")
    }
  ),
  private = list(
    .train_task = function(task) {
      if ("twoclass" %nin% task$properties) {
        stop("Only binary classification Tasks are supported.")
      }
      pv = self$param_set$get_values(tags = "train")
      if (!pv$weights_learner && !pv$weights_measure) return(task)

      weightcolname = ".WEIGHTS"
      if (weightcolname %in% unlist(task$col_roles)) {
        stopf("Weight column '%s' is already in the Task", weightcolname)
      }

      truth = task$truth()
      minor = names(which.min(table(task$truth())))

      wcol = setnames(data.table(ifelse(truth == minor, pv$minor_weight, 1)), weightcolname)
      task$cbind(wcol)
      task$col_roles$feature = setdiff(task$col_roles$feature, weightcolname)

      weights_indicators = unlist(self$param_set$get_values(tags = "weights_indicator"))
      final_roles = names(weights_indicators)[weights_indicators]
      final_roles = if (all(final_roles %in% mlr_reflections$task_col_roles$classif)) final_roles else "weight"  # support for older mlr3 versions
      task$col_roles[final_roles] = weightcolname

      task
    },

    .predict_task = identity
  )
)

mlr_pipeops$add("classweights", PipeOpClassWeights)
