% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bats.R
\name{bat_optim}
\alias{bat_optim}
\title{Bat Algorithm}
\usage{
bat_optim(D, NP, N_Gen, A, r, Qmin, Qmax, Lower, Upper, FUN, ...)
}
\arguments{
\item{D}{integer: the dimension of the search variables}

\item{NP}{integer: the population size, typically between 10 and 40}

\item{N_Gen}{integer: the number of generations, or iterations}

\item{A}{numeric; loudness, between 0 and 1, can be constant or decreasing}

\item{r}{numeric; pulse rate, must be positive, can be constant or decreasing}

\item{Qmin}{minimum frequency}

\item{Qmax}{maximum frequency}

\item{Lower}{lower bound of the search variables}

\item{Upper}{upper bound of the search variables}

\item{FUN}{the objective function to optimize, must be supplied by a user}

\item{...}{optional arguments to \code{FUN}}
}
\value{
Returns a list of four values: minimum fitness, population of solutions, fitness, best solution(s)
}
\description{
The function \code{bat_optim} implements a nature-inspired metaheuristic algorithm that deals with both continuous and discrete
optimization problems. The algorithm is based on the echolocation behavior of microbats and uses frequency tuning.
}
\details{
\code{bat_optim} implements the standard bat algorithm in three robust steps.
The first step is to initialize the parameters of algorithm to
generate and evaluate the initial population from which to determine the best solution.

Secondly, a population of virtual microbats are moved in a \emph{d}-dimensional search or solution space according to
the updating rules of the algorithm: each bat is encoded with a velocity and a location at each iteration in the
search space. The location is a solution vector, and the current best solution is achieved.

Then the current best solution is improved using random walks. The new solution is evaluated and updated.
See \emph{References} below for more details.

In essence, frequency tuning acts as mutation to vary the solutions locally; hence, increasing the range of
frequencies leads to a global search. The mutation, compared with genetic algorithms, has no crossover but
depends on loudness and pulse emission. So technically, varying loudness and pulse emission rates can also
make the search intensive approaching the global optimality.

One of the advantages of the bat algorithm is that it can converge very quickly at the initial stage and
can switch from exploration to exploitation when the optimality is approaching.
}
\examples{
# find the x-value that gives the minimum of the quadratic function y = x^2 - 3x
# x should then be 1.5
quad_func <- function(D, sol) {
 val = 0
 for (i in 1:D) {
   val <- val + sol[i] * sol[i] - sol[i] * 3
 }
 return(val)
}

# run a simulation using the standard bat algorithm
set.seed(5)  # for reproducive results
fit <- bat_optim(D = 1, NP = 20, N_Gen = 100, A = 0.5, r = 0.5,
                 Qmin = 0, Qmax = 2, Lower = -10, Upper = 10, FUN = quad_func)
x <- fit$best_solution

}
\author{
Seong Hyun Hwang, Rachel Myoung Moon
}
\references{
[1] Yang, X.-S. "A new metaheuristic bat-inspired algorithm." Nature inspired cooperative strategies for optimization (NICSO 2010). Springer Berlin Heidelberg, 2010. 65-74.

[2] Fister, I. Jr., Fister, I., Yang, X.-S., Fong, S., Zhuang, Y. "Bat algorithm: Recent advances." IEEE 15th International Symposium on Computational Intelligence and Informatics (CINTI), IEEE, 2014. 163-167.
}

