% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mhp.R
\name{mhp_compare}
\alias{mhp_compare}
\title{Compare HP vs Modified HP Filter}
\usage{
mhp_compare(x, frequency = c("quarterly", "annual"), max_lambda = 100000L)
}
\arguments{
\item{x}{Numeric vector. The time series to analyze.}

\item{frequency}{Character. Data frequency: "quarterly" (lambda=1600) or "annual" (lambda=100).}

\item{max_lambda}{Integer. Maximum lambda for Modified HP search. Default 100000.}
}
\value{
A data.table with comparison statistics for both methods:
\describe{
  \item{method}{"HP" or "Modified HP"}
  \item{lambda}{Smoothing parameter used}
  \item{cycle_sd}{Standard deviation of cyclical component}
  \item{cycle_mean}{Mean of cyclical component}
  \item{ar1}{First-order autocorrelation of cyclical component}
  \item{cycle_range}{Range of cyclical component (max - min)}
  \item{gcv}{GCV value (NA for standard HP)}
}
}
\description{
Compare the standard HP filter with the Modified HP filter for a single series.
Provides summary statistics for both methods including cycle properties.
}
\details{
The comparison includes:
1. Standard HP filter with fixed lambda (1600 for quarterly, 100 for annual)
2. Modified HP filter with GCV-optimized lambda

Statistics calculated on the cyclical component help assess filter performance:
- Lower cycle SD suggests smoother trend
- AR(1) near 0 suggests successful cycle extraction
- Near-zero mean suggests proper centering
}
\examples{
# Example 1: Quarterly GDP-like series
set.seed(789)
n <- 100
gdp <- cumsum(rnorm(n, mean = 0.7, sd = 0.5)) + 2 * cos(2 * pi * (1:n) / 32)
comparison <- mhp_compare(gdp, frequency = "quarterly", max_lambda = 10000)
print(comparison)

# Example 2: Annual series
set.seed(101)
n_annual <- 50
annual_series <- cumsum(rnorm(n_annual, mean = 2.0, sd = 1.0)) +
  3 * sin(2 * pi * (1:n_annual) / 10)
annual_comparison <- mhp_compare(annual_series, frequency = "annual", max_lambda = 5000)
print(annual_comparison)

# Example 3: Visual comparison
set.seed(2023)
test_series <- cumsum(rnorm(120, mean = 0.5, sd = 0.4)) +
  runif(1, 1, 3) * sin(2 * pi * (1:120) / 30)

comp_result <- mhp_compare(test_series, frequency = "quarterly", max_lambda = 20000)

if (require(ggplot2)) {
  # Create visualization
  hp_result <- hp_filter(test_series, lambda = 1600, as_dt = FALSE)
  mhp_result <- mhp_filter(test_series, max_lambda = 20000, as_dt = FALSE)

  plot_comparison(test_series, frequency = "quarterly", max_lambda = 20000)
}

}
