#' Retrieve information about Meetup Pro networks,
#' including groups and events.
#'
#' Meetup Pro is a premium service for organizations
#' managing multiple Meetup groups.
#' This functionality allows you to access details about
#' the groups within a Pro network
#' and the events they host.
#'
#' @template urlname
#' @template max_results
#' @template handle_multiples
#' @template date_before
#' @template date_after
#' @template asis
#' @param ... Should be empty. Used for parameter expansion
#' @template extra_graphql
#' @param status Which status the events should have.
#' @return tibble with pro network information
#'
#' @examples
#' \dontshow{
#' vcr::insert_example_cassette("get_pro", package = "meetupr")
#' meetupr:::mock_if_no_auth()
#' }
#' urlname <- "rladies"
#' members <- get_pro_groups(urlname)
#'
#' upcoming_events <- get_pro_events(urlname, "upcoming", max_results = 5)
#' \dontshow{
#' vcr::eject_cassette()
#' }
#' @name get_pro
#' @return A tibble with meetup pro information
NULL

#' @export
#' @describeIn get_pro retrieve groups in a pro network
get_pro_groups <- function(
  urlname,
  max_results = NULL,
  handle_multiples = "list",
  extra_graphql = NULL,
  asis = FALSE,
  ...
) {
  rlang::check_dots_empty()
  execute(
    standard_query(
      "get_pro_groups",
      "data.proNetwork.groupsSearch"
    ),
    urlname = urlname,
    first = max_results,
    max_results = max_results,
    handle_multiples = handle_multiples,
    extra_graphql = extra_graphql,
    asis = asis
  ) |>
    process_datetime_fields(c("founded_date", "pro_join_date")) |>
    dplyr::mutate(
      country = get_country_code(country)
    )
}

#' @describeIn get_pro retrieve events from a pro network
#' @export
get_pro_events <- function(
  urlname,
  status = NULL,
  date_before = NULL,
  date_after = NULL,
  max_results = NULL,
  handle_multiples = "list",
  extra_graphql = NULL,
  asis = FALSE,
  ...
) {
  rlang::check_dots_empty()

  execute(
    standard_query(
      "get_pro_events",
      "data.proNetwork.eventsSearch"
    ),
    urlname = urlname,
    first = max_results,
    max_results = max_results,
    status = validate_event_status(status, pro = TRUE),
    handle_multiples = handle_multiples,
    date_before = date_before,
    date_after = date_after,
    extra_graphql = extra_graphql,
    asis = asis
  ) |>
    process_datetime_fields(c("date_time", "pro_join_date"))
}
