% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/package_documentation.R
\docType{package}
\name{manureshed}
\alias{manureshed-package}
\alias{manureshed}
\title{manureshed: Manureshed Analysis with WWTP Integration}
\description{
This package provides comprehensive tools for analyzing agricultural nutrient
balances at multiple spatial scales with optional integration of wastewater
treatment plant (WWTP) nutrient loads for both nitrogen and phosphorus.
}
\details{
All datasets are downloaded on-demand from OSF repository to minimize package
size while maintaining full functionality.
}
\section{Main Functions}{


\subsection{Data Loading and Management:}{
\itemize{
\item \code{\link{load_builtin_nugis}}: Load NuGIS data from OSF
\item \code{\link{load_builtin_boundaries}}: Load spatial boundaries from OSF
\item \code{\link{load_builtin_wwtp}}: Load WWTP data from OSF (2016)
\item \code{\link{check_builtin_data}}: Check available datasets and cache status
\item \code{\link{download_all_data}}: Download all datasets at once
\item \code{\link{download_osf_data}}: Download specific dataset
\item \code{\link{clear_data_cache}}: Clear cached datasets
}
}

\subsection{High-Level Workflows:}{
\itemize{
\item \code{\link{run_builtin_analysis}}: Complete end-to-end analysis workflow
\item \code{\link{quick_analysis}}: Analysis with automatic visualizations
\item \code{\link{batch_analysis_years}}: Multi-year analysis workflow
}
}

\subsection{Agricultural Classification:}{
\itemize{
\item \code{\link{agri_process_nugis}}: Process and standardize NuGIS data
\item \code{\link{agri_classify_nitrogen}}: Classify nitrogen balance
\item \code{\link{agri_classify_phosphorus}}: Classify phosphorus balance
\item \code{\link{agri_classify_complete}}: Complete agricultural pipeline
}
}

\subsection{WWTP Processing:}{
\itemize{
\item \code{\link{load_user_wwtp}}: Load custom WWTP data with flexible formatting
\item \code{\link{wwtp_clean_data}}: Clean and filter WWTP data
\item \code{\link{wwtp_classify_sources}}: Classify WWTP facilities by load size
\item \code{\link{convert_load_units}}: Handle different load units (kg, lbs, tons)
}
}

\subsection{Data Integration:}{
\itemize{
\item \code{\link{integrate_wwtp_agricultural}}: Combine WWTP and agricultural data
\item \code{\link{integrate_complete}}: Complete integration pipeline
\item \code{\link{add_texas_huc8}}: Add Texas HUC8 supplemental data
}
}

\subsection{Visualization and Mapping:}{
\itemize{
\item \code{\link{map_agricultural_classification}}: Map nutrient classifications
\item \code{\link{map_wwtp_points}}: Map WWTP facility locations
\item \code{\link{map_wwtp_influence}}: Map WWTP influence/proportion
\item \code{\link{get_state_boundaries}}: Get US state boundaries for mapping
}
}

\subsection{Spatial Analysis:}{
\itemize{
\item \code{\link{calculate_transition_probabilities}}: Spatial transition analysis
\item \code{\link{create_network_plot}}: Network visualization of transitions
\item \code{\link{add_centroid_coordinates}}: Calculate spatial centroids
}
}

\subsection{Comparison Analysis:}{
\itemize{
\item \code{\link{create_classification_summary}}: Before/after comparison summaries
\item \code{\link{plot_before_after_comparison}}: Comparison bar plots
\item \code{\link{plot_impact_ratios}}: Impact ratio visualizations
\item \code{\link{plot_absolute_changes}}: Absolute change plots
}
}

\subsection{Utility Functions:}{
\itemize{
\item \code{\link{get_nutrient_colors}}: Get color schemes for nutrients
\item \code{\link{clean_category_names}}: Clean classification names for display
\item \code{\link{format_huc8}}: Format HUC8 codes with leading zeros
\item \code{\link{get_cropland_threshold}}: Calculate exclusion thresholds
}
}
}

\section{Spatial Scales}{

The package supports analysis at three spatial scales:
\itemize{
\item \strong{County}: US county boundaries (3,000+ units)
\item \strong{HUC8}: 8-digit Hydrologic Unit Code watersheds (2,000+ units)
\item \strong{HUC2}: 2-digit Hydrologic Unit Code regions (18 units)
}
}

\section{Nutrients Supported}{

The package supports analysis for both major nutrients with appropriate methodologies:
\itemize{
\item \strong{Nitrogen}: Uses 0.5 availability factor for manure nitrogen in calculations
\item \strong{Phosphorus}: Direct calculation without availability factor
}
Users can analyze one nutrient, both nutrients, or different combinations
in the same workflow: \code{nutrients = c("nitrogen", "phosphorus")}
}

\section{Classification System}{

Spatial units are classified into five categories based on nutrient balance:
\itemize{
\item \strong{Source}: Net nutrient surplus available for export
\item \strong{Sink Deficit}: Total deficit requiring nutrient imports
\item \strong{Sink Fertilizer}: Fertilizer surplus available for manure import
\item \strong{Within Watershed/County}: Balanced for internal nutrient transfers
\item \strong{Excluded}: Insufficient cropland for meaningful analysis (<500 ha equivalent)
}
}

\section{Data Sources}{

The package provides access to comprehensive built-in datasets:
\subsection{NuGIS Data:}{
\itemize{
\item County-level data (1987 - 2016)
\item HUC8 watershed data (1987 - 2016)
\item HUC2 regional data (1987 - 2016)
\item All nutrient balance components (manure, fertilizer, removal, fixation)
}
}

\subsection{Spatial Boundaries:}{
\itemize{
\item US county boundaries (CONUS)
\item HUC8 watershed boundaries
\item HUC2 regional boundaries
\item All in Albers Equal Area Conic projection (EPSG:5070)
}
}

\subsection{WWTP Data:}{
\itemize{
\item Nitrogen discharge data (2007 - 2016)
\item Phosphorus discharge data (2007 - 2016)
\item Pre-processed and classification-ready
\item Includes facility metadata and spatial coordinates
}
}

\subsection{Supplemental Data:}{
\itemize{
\item Texas HUC8 data (automatically integrated for HUC8 analyses)
\item Texas spatial boundaries
}
}
}

\section{OSF Data Repository}{

All datasets are hosted on OSF and downloaded on-demand:
\itemize{
\item \strong{Repository}: https://osf.io/g39xa/
\item \strong{Automatic caching}: Data downloaded once and reused
\item \strong{Flexible loading}: Load only the data you need
\item \strong{Version control}: Permanent DOI for reproducibility
\item \strong{Size efficiency}: Package <1MB, full datasets ~25MB
}
}

\section{WWTP Data Flexibility}{

The package handles varying EPA WWTP data formats across different years:
\itemize{
\item \strong{Unit Conversion}: Automatic conversion between kg, lbs, pounds, and tons
\item \strong{Column Mapping}: Flexible mapping to handle EPA naming changes
\item \strong{Header Detection}: Support for different header row positions
\item \strong{Built-in 2016}: Ready-to-use cleaned data for immediate analysis
\item \strong{Custom Integration}: Easy integration of user data for other years
}
}

\section{Workflow Examples}{

\preformatted{
# Check what data is available
check_builtin_data()

# Download all datasets (optional, ~40MB)
download_all_data()

# Basic analysis using built-in data - any year 2007-2016
results <- run_builtin_analysis(
  scale = "huc8",
  year = 2012,  # Any year 2007-2016 now supported
  nutrients = c("nitrogen", "phosphorus"),
  include_wwtp = TRUE
)

wwtp_n_2010 <- load_builtin_wwtp("nitrogen", year = 2010)
wwtp_p_2015 <- load_builtin_wwtp("phosphorus", year = 2015)


# Quick analysis with automatic visualizations
viz_results <- quick_analysis(
  scale = "county",
  year = 2016,
  nutrients = "nitrogen",
  include_wwtp = TRUE
)

# Historical analysis without WWTP
historical <- run_builtin_analysis(
  scale = "huc8",
  year = 2010,
  nutrients = c("nitrogen", "phosphorus"),
  include_wwtp = FALSE
)

# Load specific datasets manually
county_2016 <- load_builtin_nugis("county", 2016)
boundaries <- load_builtin_boundaries("county")
wwtp_n <- load_builtin_wwtp("nitrogen")
}
}

\section{Analysis Outputs}{

The package generates comprehensive outputs for each nutrient analyzed:
\itemize{
\item \strong{Spatial Data}: Classification results as sf objects
\item \strong{Maps}: Classification, influence, and facility maps
\item \strong{Networks}: Transition probability visualizations
\item \strong{Comparisons}: Before/after WWTP integration analysis
\item \strong{Data Files}: CSV centroids, RDS spatial data
\item \strong{Metadata}: Analysis parameters and processing information
}
}

\section{Performance and Scalability}{

\itemize{
\item Optimized for CONUS-scale analysis (1000s of spatial units)
\item Memory-efficient spatial operations
\item On-demand data loading reduces memory footprint
\item Progress reporting for long-running analyses
\item Automatic garbage collection for memory management
}
}

\seealso{
Useful links:
\itemize{
  \item \url{https://osf.io/g39xa/}
  \item \url{https://github.com/cwru-sdle/manureshed}
  \item \url{https://exelegch.github.io/manureshed-docs/}
}

}
\author{
\strong{Maintainer}: Olatunde D. Akanbi \email{olatunde.akanbi@case.edu} (\href{https://orcid.org/0000-0001-7719-2619}{ORCID}) [copyright holder]

Authors:
\itemize{
  \item Vibha Mandayam (\href{https://orcid.org/0009-0008-8628-9904}{ORCID})
  \item Atharva Gupta (\href{https://orcid.org/0009-0004-5372-0260}{ORCID})
  \item K. Colton Flynn (\href{https://orcid.org/0000-0002-5718-1071}{ORCID})
  \item Jeffrey Yarus (\href{https://orcid.org/0000-0002-9331-9568}{ORCID})
  \item Erika I. Barcelos \email{erika.barcelos@case.edu} (\href{https://orcid.org/0000-0002-9273-8488}{ORCID}) [copyright holder]
  \item Roger H. French \email{roger.french@case.edu} (\href{https://orcid.org/0000-0002-6162-0532}{ORCID}) [copyright holder]
}

}
