% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lotriNearPD.R
\name{lotriNearPD}
\alias{lotriNearPD}
\title{C++ implementation of Matrix's nearPD}
\usage{
lotriNearPD(
  x,
  keepDiag = FALSE,
  do2eigen = TRUE,
  doDykstra = TRUE,
  only.values = FALSE,
  ensureSymmetry = !isSymmetric(x),
  eig.tol = 1e-06,
  conv.tol = 1e-07,
  posd.tol = 1e-08,
  maxit = 100L,
  trace = FALSE
)
}
\arguments{
\item{x}{numeric \eqn{n \times n}{n * n} approximately positive
    definite matrix, typically an approximation to a correlation or
    covariance matrix.  If \code{x} is not symmetric (and
    \code{ensureSymmetry} is not false), \code{\link[Matrix]{symmpart}(x)} is used.}

\item{keepDiag}{logical, generalizing \code{corr}: if \code{TRUE}, the
    resulting matrix should have the same diagonal
    (\code{\link{diag}(x)}) as the input matrix.}

\item{do2eigen}{logical indicating if a `posdefify()` (like in the
package `sfsmisc`) eigen step should be applied to the result of
the Higham algorithm}

\item{doDykstra}{logical indicating if Dykstra's correction should be
    used; true by default.  If false, the algorithm is basically the
    direct fixpoint iteration
    \eqn{Y_k = P_U(P_S(Y_{k-1}))}{Y(k) = P_U(P_S(Y(k-1)))}.}

\item{only.values}{logical; if \code{TRUE}, the result is just the
    vector of eigenvalues of the approximating matrix.}

\item{ensureSymmetry}{logical; by default, \code{\link[Matrix]{symmpart}(x)}
is used whenever \code{isSymmetric(x)} is not true.  The user
can explicitly set this to \code{TRUE} or \code{FALSE}, saving the
symmetry test. \emph{Beware} however that setting it \code{FALSE}
for an \bold{a}symmetric input \code{x}, is typically nonsense!}

\item{eig.tol}{defines relative positiveness of eigenvalues compared
    to largest one, \eqn{\lambda_1}. Eigenvalues \eqn{\lambda_k} are
    treated as if zero when \eqn{\lambda_k / \lambda_1 \le eig.tol}.}

\item{conv.tol}{convergence tolerance for Higham algorithm.}

\item{posd.tol}{tolerance for enforcing positive definiteness (in the
    final \code{posdefify} step when \code{do2eigen} is \code{TRUE}).}

\item{maxit}{maximum number of iterations allowed.}

\item{trace}{logical or integer specifying if convergence monitoring
    should be traced.}
}
\value{
unlike the matrix package, this simply returns the nearest
  positive definite matrix
}
\description{
With `ensureSymmetry` it makes sure it is symmetric by applying 0.5*(t(x) + x) before using lotriNearPD
}
\details{
This implements the algorithm of Higham (2002), and then (if
\code{do2eigen} is true) forces positive definiteness using code from
`sfsmisc::posdefify()`.  The algorithm of Knol and ten
Berge (1989) (not implemented here) is more general in that it
allows constraints to (1) fix some rows (and columns) of the matrix and
(2) force the smallest eigenvalue to have a certain value.

Note that setting \code{corr = TRUE} just sets \code{diag(.) <- 1}
within the algorithm.

Higham (2002) uses Dykstra's correction, but the version by Jens
 \enc{Oehlschlägel}{Oehlschlaegel} did not use it (accidentally),
  and still gave reasonable results; this simplification, now only
 used if \code{doDykstra = FALSE},
  was active in \code{nearPD()} up to Matrix version 0.999375-40.
}
\examples{

set.seed(27)
m <- matrix(round(rnorm(25),2), 5, 5)
m <- m + t(m)
diag(m) <- pmax(0, diag(m)) + 1
(m <- round(cov2cor(m), 2))

near.m <- lotriNearPD(m)
round(near.m, 2)
norm(m - near.m) # 1.102 / 1.08

round(lotriNearPD(m, only.values=TRUE), 9)

## A longer example, extended from Jens' original,
## showing the effects of some of the options:

pr <- matrix(c(1,     0.477, 0.644, 0.478, 0.651, 0.826,
               0.477, 1,     0.516, 0.233, 0.682, 0.75,
               0.644, 0.516, 1,     0.599, 0.581, 0.742,
               0.478, 0.233, 0.599, 1,     0.741, 0.8,
               0.651, 0.682, 0.581, 0.741, 1,     0.798,
               0.826, 0.75,  0.742, 0.8,   0.798, 1),
               nrow = 6, ncol = 6)

nc  <- lotriNearPD(pr)

}
\references{
%% more in /u/maechler/R/Pkgs/sfsmisc/man/posdefify.Rd
  Cheng, Sheung Hun and Higham, Nick (1998)
  A Modified Cholesky Algorithm Based on a Symmetric Indefinite Factorization;
  \emph{SIAM J. Matrix Anal.\\ Appl.}, \bold{19}, 1097--1110.

  Knol DL, ten Berge JMF (1989)
  Least-squares approximation of an improper correlation matrix by a
  proper one.
  \emph{Psychometrika} \bold{54}, 53--61.

  Higham, Nick (2002)
  Computing the nearest correlation matrix - a problem from finance;
  \emph{IMA Journal of Numerical Analysis} \bold{22}, 329--343.
}
\seealso{
A first version of this (with non-optional \code{corr=TRUE})
has been available as `sfsmisc::nearcor()` and
more simple versions with a similar purpose
`sfsmisc::posdefify()`
}
\author{
Jens \enc{Oehlschlägel}{Oehlschlaegel} donated a first
  version to Matrix.  Subsequent changes by the Matrix package
  authors, later modifications to C++ by Matthew Fidler
}
