% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kdevinecop.R
\name{kdevinecop}
\alias{kdevinecop}
\title{Kernel estimation of vine copula densities}
\usage{
kdevinecop(
  data,
  matrix = NA,
  method = "TLL2",
  renorm.iter = 3L,
  mult = 1,
  test.level = NA,
  trunc.level = NA,
  treecrit = "tau",
  cores = 1,
  info = FALSE
)
}
\arguments{
\item{data}{(\eqn{n x d}) matrix of copula data (have to lie in \eqn{[0,1^d]}).}

\item{matrix}{R-Vine matrix (\eqn{n x d}) specifying the structure of the vine;
if \code{NA} (default) the structure selection heuristic of Dissman et al.
(2013) is applied.}

\item{method}{see \code{\link[kdecopula:kdecop]{kdecop}}.}

\item{renorm.iter}{see \code{\link[kdecopula:kdecop]{kdecop}}.}

\item{mult}{see \code{\link[kdecopula:kdecop]{kdecop}}.}

\item{test.level}{significance level for independence test. If you provide a
number in \eqn{[0, 1]}, an independence test
(\code{\link[VineCopula:BiCopIndTest]{BiCopIndTest}}) will be performed for
each pair; if the null hypothesis of independence cannot be rejected, the
independence copula will be set for this pair. If \code{test.level = NA}
(default), no independence test will be performed.}

\item{trunc.level}{integer; the truncation level. All pair copulas in trees
above the truncation level will be set to independence.}

\item{treecrit}{criterion for structure selection; defaults to \code{"tau"}.}

\item{cores}{integer; if \code{cores > 1}, estimation will be parallized
within each tree (using \code{\link[foreach]{foreach}}).}

\item{info}{logical; if \code{TRUE}, additional information about the
estimate will be gathered (see \code{\link[kdecopula:kdecop]{kdecop}}).}
}
\value{
An object of class \code{kdevinecop}. That is, a list containing
\item{T1, T2, ...}{lists of the estimted pair copulas in each tree,}
\item{matrix}{the structure matrix of the vine,}
\item{info}{additional information about the fit (if \code{info = TRUE}).}
}
\description{
The function estimates a vine copula density using kernel estimators for the
pair copulas (based on the \link[kdecopula]{kdecopula-package}).
}
\examples{
data(wdbc, package = "kdecopula")
# rank-transform to copula data (margins are uniform)
u <- VineCopula::pobs(wdbc[, 5:7], ties = "average")
\dontshow{u <- u[1:30, ]}
fit <- kdevinecop(u)                   # estimate density
dkdevinecop(c(0.1, 0.1, 0.1), fit)     # evaluate density estimate
contour(fit)                           # contour matrix (Gaussian scale)
pairs(rkdevinecop(500, fit))           # plot simulated data

}
\references{
Nagler, T., Czado, C. (2016) \cr Evading the curse of
dimensionality in nonparametric density estimation with simplified vine
copulas. \cr \emph{Journal of Multivariate Analysis 151, 69-89
(doi:10.1016/j.jmva.2016.07.003)}

Nagler, T., Schellhase, C. and Czado, C. (2017) \cr Nonparametric
estimation of simplified vine copula models: comparison of methods
arXiv:1701.00845

Dissmann, J., Brechmann, E. C., Czado, C., and Kurowicka, D. (2013). \cr
Selecting and estimating regular vine copulae and application to financial
returns. \cr
Computational Statistics & Data Analysis, 59(0):52--69.
}
\seealso{
\code{\link{dkdevinecop}},
\code{\link[kdecopula:kdecop]{kdecop}},
\code{\link[VineCopula:BiCopIndTest]{BiCopIndTest}},
\code{\link[foreach]{foreach}}
}
