% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lp.R
\name{lpSetup}
\alias{lpSetup}
\title{Constructing LP problem}
\usage{
lpSetup(
  env,
  sset,
  orig.sset = NULL,
  equal.coef0 = NULL,
  equal.coef1 = NULL,
  shape = TRUE,
  direct = FALSE,
  rescale = TRUE,
  solver
)
}
\arguments{
\item{env}{environment containing the matrices defining the LP/QCQP
problem.}

\item{sset}{List of IV-like estimates and the corresponding gamma
terms.}

\item{orig.sset}{list, only used for bootstraps. The list contains
the gamma moments for each element in the S-set, as well as the
IV-like coefficients.}

\item{equal.coef0}{character, name of terms in \code{m0} that
should have common coefficients with the corresponding terms in
\code{m1}.}

\item{equal.coef1}{character, name of terms in \code{m1} that
should have common coefficients with the corresponding terms in
\code{m0}.}

\item{shape}{boolean, default set to TRUE. Switch to determine
whether or not to include shape restrictions in the LP/QCQP
problem.}

\item{direct}{boolean, set to \code{TRUE} if the direct MTR
regression is used.}

\item{rescale}{boolean, set to \code{TRUE} if the MTR components
should be rescaled to improve stability in the LP/QCQP
optimization.}

\item{solver}{string, name of the package used to solve the LP/QCQP
problem.}
}
\value{
A list of matrices and vectors necessary to define an
    LP/QCQP problem.
}
\description{
If the user passes IV-like moments to the function, then the
function constructs the components of the LP problem. If no IV-like
moments are passed, then the function constructs the linear
constraints of the QCQP problem. Note that the LP/QCQP model will
be saved inside an environment variable, which is to be passed
through the argument \code{env}. This is done for efficient use of
memory. The environment \code{env} is supposed to already contain a
list under the entry \code{$mbobj} containing the matrices defining
the shape constraints. This list of shape constraints \code{$mbobj}
should contain three entries corresponding to a system of linear
equations of the form \code{Ax <=> b}: \code{mbA}, the matrix
defining the constraints, \code{A}; \code{mbs}, a vector indicating
whether a row in \code{mbA} is an equality or inequality constraint
(for Gurobi and MOSEK, use '<=', '>=', '='; for CPLEX,
use 'L', 'G', and 'E'); \code{mbrhs}, a vector of the right hand
side values defining the constraint of the form i.e. the vector
\code{b}. Depending on the linear programming solver used, this
function will return different output specific to the solver.
}
\examples{
dtm <- ivmte:::gendistMosquito()

## Declare empty list to be updated (in the event multiple IV like
## specifications are provided
sSet <- list()

## Declare MTR formulas
formula0 = ~ 1 + u
formula1 = ~ 1 + u

## Construct object that separates out non-spline components of MTR
## formulas from the spline components. The MTR functions are
## obtained from this object by the function 'genSSet'.
splinesList = list(removeSplines(formula0), removeSplines(formula1))

## Construct MTR polynomials
polynomials0 <- polyparse(formula = formula0,
                          data = dtm,
                          uname = u,
                          as.function = FALSE)
polynomials1 <- polyparse(formula = formula1,
                          data = dtm,
                          uname = u,
                           as.function = FALSE)

## Generate propensity score model
propensityObj <- propensity(formula = d ~ z,
                            data = dtm,
                            link = "linear")

## Generate IV estimates
ivEstimates <- ivEstimate(formula = ey ~ d | z,
                          data = dtm,
                          components = l(intercept, d),
                          treat = d,
                          list = FALSE)

## Generate target gamma moments
targetGamma <- genTarget(treat = "d",
                         m0 = ~ 1 + u,
                         m1 = ~ 1 + u,
                         target = "atu",
                         data = dtm,
                         splinesobj = splinesList,
                         pmodobj = propensityObj,
                         pm0 = polynomials0,
                         pm1 = polynomials1)

## Construct S-set. which contains the coefficients and weights
## corresponding to various IV-like estimands
sSet <- genSSet(data = dtm,
                sset = sSet,
                sest = ivEstimates,
                splinesobj = splinesList,
                pmodobj = propensityObj$phat,
                pm0 = polynomials0,
                pm1 = polynomials1,
                ncomponents = 2,
                scount = 1,
                yvar = "ey",
                dvar = "d",
                means = TRUE)
## Only the entry $sset is required
sSet <- sSet$sset

## Define additional upper- and lower-bound constraints for the LP
## problem.  The code below imposes a lower bound of 0.2 and upper
## bound of 0.8 on the MTRs.
A <- matrix(0, nrow = 22, ncol = 4)
A <- cbind(A, rbind(cbind(1, seq(0, 1, 0.1)),
                    matrix(0, nrow = 11, ncol = 2)))
A <- cbind(A, rbind(matrix(0, nrow = 11, ncol = 2),
                    cbind(1, seq(0, 1, 0.1))))
sense <- c(rep(">", 11), rep("<", 11))
rhs <- c(rep(0.2, 11), rep(0.8, 11))

## Construct LP object to be interpreted and solved by
## lpSolveAPI. Note that an environment has to be created for the LP
## object. The matrices defining the shape restrictions must be stored
## as a list under the entry \code{$mbobj} in the environment.
modelEnv <- new.env()
modelEnv$mbobj <- list(mbA = A,
                    mbs = sense,
                    mbrhs = rhs)
## Convert the matrices defining the shape constraints into a format
## that is suitable for the LP solver.
lpSetup(env = modelEnv,
        sset = sSet,
        solver = "lpsolveapi")
## Setup LP model so that it is solving for the bounds.
lpSetupBound(env = modelEnv,
             g0 = targetGamma$gstar0,
             g1 = targetGamma$gstar1,
             sset = sSet,
             criterion.tol = 0,
             criterion.min = 0,
             solver = "lpsolveapi")
## Declare any LP solver options as a list.
lpOptions <- optionsLpSolveAPI(list(epslevel = "tight"))
## Obtain the bounds.
bounds <- bound(env = modelEnv,
                sset = sSet,
                solver = "lpsolveapi",
                solver.options = lpOptions)
cat("The bounds are [",  bounds$min, ",", bounds$max, "].\n")

}
