% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/use_model.R
\name{decompose_volume}
\alias{decompose_volume}
\title{Decompose Intraday Volume into Several Components}
\usage{
decompose_volume(purpose, model, data, burn_in_days = 0)
}
\arguments{
\item{purpose}{String \code{"analysis"/"forecast"}. Indicates the purpose of using the provided model.}

\item{model}{A model object of class "\code{volume_model}" from \code{fit_volume()}.}

\item{data}{An n_bin * n_day matrix or an \code{xts} object storing intraday volume.}

\item{burn_in_days}{Number of initial days in the burn-in period for forecast. Samples from the first \code{burn_in_days} are used to warm up the model and then are discarded.}
}
\value{
A list containing the following elements:
       \itemize{
       \item{\code{original_signal}: }{A vector of original intraday volume;}
       \item{\code{smooth_signal} / \code{forecast_signal}: }{A vector of smooth/forecast intraday volume;}
       \item{\code{smooth_components} /\code{forecast_components}: }{A list of smooth/forecast components: daily, seasonal, intraday dynamic, and residual components.}
       \item{\code{error}: }{A list of three error measures: mae, mape, and rmse.}
       }
}
\description{
This function decomposes the intraday volume into daily, seasonal, and intraday dynamic components according to (Chen et al., 2016).
If \code{purpose = “analysis”} (aka Kalman smoothing), the optimal components are conditioned on both the past and future observations.
Its mathematical expression is \eqn{\hat{x}_{\tau} = E[x_{\tau}|\{y_{j}\}_{j=1}^{M}]}{x*(\tau) = E[x(\tau) | y(j), j = 1, ... , M]},
             where \eqn{M} is the total number of bins in the dataset.

             If \code{purpose = “forecast”} (aka Kalman forecasting), the optimal components are conditioned on only the past observations.
             Its mathematical expression is \eqn{\hat{x}_{\tau+1} = E[x_{\tau+1}|\{y_{j}\}_{j=1}^{\tau}]}{x*(\tau+1) = E[x(\tau + 1) | y(j), j = 1, ... , \tau]}.

             Three measures are used to evaluate the model performance:
             \itemize{\item{Mean absolute error (MAE):
                            \eqn{\frac{1}{M}\sum_{\tau=1}^M|\hat{y}_{\tau} - y_{\tau}|}{\sum (|y*(\tau) - y(\tau)|) / M} ;}
                      \item{Mean absolute percent error (MAPE):
                            \eqn{\frac{1}{M}\sum_{\tau=1}^M\frac{|\hat{y}_{\tau} - y_{\tau}|}{y_{\tau}}}{\sum (|y*(\tau) - y(\tau)| / y(\tau)) / M} ;}
                      \item{Root mean square error (RMSE):
                            \eqn{\sqrt{\sum_{\tau=1}^M\frac{\left(\hat{y}_{\tau} - y_{\tau}\right)^2}{M}}}{[\sum ((y*(\tau) - y(\tau))^2 / M)]^0.5} .}
             }
}
\examples{
library(intradayModel)
data(volume_aapl)
volume_aapl_training <- volume_aapl[, 1:20]
volume_aapl_testing <- volume_aapl[, 21:50]
model_fit <- fit_volume(volume_aapl_training, fixed_pars = list(a_mu = 0.5, var_mu = 0.05),
                        init_pars = list(a_eta = 0.5))

# analyze training volume
analysis_result <- decompose_volume(purpose = "analysis", model_fit, volume_aapl_training)

# forecast testing volume
forecast_result <- decompose_volume(purpose = "forecast", model_fit, volume_aapl_testing)

# forecast testing volume with burn-in
forecast_result <- decompose_volume(purpose = "forecast", model_fit, volume_aapl[, 1:50],
                             burn_in_days = 20)

}
\references{
Chen, R., Feng, Y., and Palomar, D. (2016). Forecasting intraday trading volume: A Kalman filter approach. Available at SSRN 3101695.
}
\author{
Shengjie Xiu, Yifan Yu and Daniel P. Palomar
}
