#include <RcppArmadillo.h>
#include "ieTest.h"

using namespace Rcpp;


// Function that tests a null hypothesis for the mediation effect using
// the maxP-test. Function also requires additional values, either in the form
// of estimates and covariances, or direct cumulative probabilities. 
// If estimates and variances are passed, the user must specify what 
// distribution is to be used (t- and normal available currently). 
// The function performs the test using either
// given or calculated U values, returning the p-value.

//' MaxP test for the indirect effect - Single mediator path
//'
//' This function takes 
//' estimates and covariances, or 2 U values, for the maxP test 
//' for either one mediator of one path of an unordered mediation scenario.
//' If estimates are passed to the function, the user must specify 
//' what distribution is to be used to find the cumulative probabilities.
//' The maximum p-value is returned. 
//'
//' @param u1,u2 The U values to be used in the test. Given priority over estimates, but both must be supplied.
//' @param V1Dist String value specifying the distribution of the estimate of the independent variable on the mediator. Ignored if u1 and u2 are supplied.
//' @param V1 Value of the estimate of the independent variable on the mediator. Ignored if u1 and u2 are supplied.
//' @param V1_VAR Value of the variance of the estimate of the independent variable on the mediator. Ignored if u1 and u2 are supplied.
//' @param V1_DF Degrees of freedom for V1. Only needed if t-distribution is used.
//' @param V2Dist String value specifying the distribution of the estimate of the mediator (and interaction term) on the response.
//' @param V2 Value of the estimate of the mediator on the response. Ignored if u1 and u2 are supplied.
//' @param V2_VAR Value of the variance of the estimate of the mediator on the response.. Ignored if u1 and u2 are supplied.
//' @param V2_DF Degrees of freedom for V2. Only needed if t-distribution is used.. 
//' @param V2b Value of the estimate of the effect of the interaction of the independent and mediator variable on the response. Ignored if u1 and u2 are supplied.
//' @param V2b_VAR Value of the variance of the estimate of the effect of the interaction of the independent and mediator variable on the response. Ignored if u1 and u2 are supplied.
//' @param V2bmult Value indicating the value of the independent variable used for the interaction. Typically 1.
//' @param V1_V2_cov Value of the covariance between V1 and V2. Typically 0 for fully observed data. 
//' @param V1_V2b_cov Value of the covariance between V1 and V2b. Typically 0 for fully observed data.
//' @param V2_V2b_cov Value of the covariance between V2 and V2b.
//' @param V1_0 Null value for V1.
//' @param V2_0 Null value for V2.
//' @param V2b_0 Null value for V2b.
//' @returns The p-value of the test in the form of the larger of the p-values for the individual parameters.
//' @export
//' @examples
//' maxp_one(u1 = .02, u2= .015)
// [[Rcpp::export]]
double maxp_one(Rcpp::Nullable<Rcpp::NumericVector> u1 = R_NilValue,
               Rcpp::Nullable<Rcpp::NumericVector> u2 = R_NilValue,
               Nullable<CharacterVector> V1Dist = R_NilValue,
               Rcpp::Nullable<Rcpp::NumericVector> V1 = R_NilValue, 
               Rcpp::Nullable<Rcpp::NumericVector> V1_VAR = R_NilValue,
               Nullable<int> V1_DF = R_NilValue,
               Nullable<CharacterVector> V2Dist = R_NilValue,
               Rcpp::Nullable<Rcpp::NumericVector> V2 = R_NilValue, 
               Rcpp::Nullable<Rcpp::NumericVector> V2_VAR = R_NilValue,
               Rcpp::Nullable<Rcpp::NumericVector> V2_DF = R_NilValue,
               double V2b = 0, double V2b_VAR = 0,
               const int V2bmult = 1,
               double V1_V2_cov = 0, double V1_V2b_cov = 0,
               double V2_V2b_cov = 0,
               double V1_0 = 0, double V2_0 = 0, double V2b_0 = 0){
  
  // Figure out which distribution is being used
  // Use given u1 and u2 values if passed
  
  // Check if calculations need to be done (or if U values are passed in)
  // Then check if correlation between variables and do transformation
  // if necessary.

  /* Check if calculations */
  // If only one u value is specified, return with error;
  if( ( u1.isNull() && u2.isNotNull() )  ||
  ( u1.isNotNull() && u2.isNull() )){
    Rcout << "Both U values must be specified." << "\n";
    return -1;
  }
  
  // Declare u1 and u2 to either be filled in with inputs or calculated;
  double u1_ = .5;
  double u2_ = .5;
  
  // Calculate u values if neither specified. Otherwise just use what is given down below.
  if(u1.isNull() && u2.isNull()){
    
    double V1_ = 0;
    double V2_ = 0;
    double V1_VAR_ = 0;
    double V2_VAR_ = 0;
    CharacterVector V1Dist_;
    CharacterVector V2Dist_;
    
    if(V1Dist.isNull() && V2Dist.isNull()){
      Rcout << "A distribution must be specified for parameters to be calculated." << "\n";
      return -1;
    }
    
    // If one distribution is specified, but the other is unspecified, set both to the specified value.
    // Both dists as null is returned with an error above.
    if(V1Dist.isNull()){
      V1Dist_ = CharacterVector(V2Dist);
    }else{
      V1Dist_ = CharacterVector(V1Dist);
    }
    
    if(V2Dist.isNull()){
      V2Dist_ = CharacterVector(V1Dist);
    }else{
      V2Dist_ = CharacterVector(V2Dist);
    }
    
    
    // If V1 or V2 (including variances) are missing, return with error;
    if(V1.isNull() || V2.isNull() || V1_VAR.isNull() || V2_VAR.isNull()){
      Rcout << "You must specify the two effects and their variances" << "\n";
      return -1;
    }
    
    // If correlation between v1 and either v2 or V2b, then transform;
    if( V1_V2_cov != 0 || V1_V2b_cov != 0 ){
      arma::vec tParam(3);
      tParam(0) = NumericVector(V1)[0]; tParam(1) = NumericVector(V2)[0]; 
      tParam(2) = V2b ;
      arma::vec tNull(3);
      tNull(0) = V1_0; tNull(1) = V2_0; tNull(2) = V2b_0;
      
      arma::mat tSig(3, 3);
      
      tSig(0, 0) = NumericVector(V1_VAR)[0]; 
      tSig(1, 1) = NumericVector(V2_VAR)[0]; 
      tSig(2, 2) = V2b_VAR;
      tSig(0, 1) = V1_V2_cov; tSig(1, 0) = V1_V2_cov;
      tSig(0, 2) = V1_V2b_cov; tSig(2, 0) = V1_V2b_cov;
      tSig(1, 2) = V2_V2b_cov; tSig(2, 1) = V2_V2b_cov;
      
      if(tSig(2, 2) == 0){tSig(2, 2) = 1;}
      
      arma::mat transMeans = corrTrans(tParam, tNull, tSig);
      
      // Update values with transformation;
      V1_ = transMeans(0, 0); 
      V2_ = transMeans(1, 0); 
      V2b = transMeans(2, 0);
      V1_0 = transMeans(0, 1); V2_0 = transMeans(1, 1); V2b_0 = transMeans(2, 1);
      
      V1_VAR_ = 1; V2_VAR_ = 1; 
      if(V2b_VAR != 0){V2b_VAR = 1;}
      V1_V2_cov = 0; V1_V2b_cov = 0; V2_V2b_cov = 0;
    }else{
      V1_ = NumericVector(V1)[0];
      V2_ = NumericVector(V2)[0];
      V1_VAR_ = NumericVector(V1_VAR)[0];
      V2_VAR_ = NumericVector(V2_VAR)[0];
    }
    
    // Now calculate U values if possible;
    // u1;
    if( (V1Dist_[0] == "Normal") ||
        (V1Dist_[0] == "normal") ||
        (V1Dist_[0] == "N") ||
        (V1Dist_[0] == "n")){
      u1_ = normU(V1_, V1_VAR_, 0, 0, 1, 0, V1_0, 0);
    }else if((V1Dist_[0] == "T") ||
      (V1Dist_[0] == "t")){
      if(V1_DF.isNull()){
        Rcout << "Degrees of Freedom must be specified for t-distribution" << "\n";
        return 1;
      }
      u1_ = tU(V1_, V1_VAR_, NumericVector(V1_DF)[0], 0, 0, 1, 0, V1_0, 0, 1);
    }
    
    // u2;
    if( (V2Dist_[0] == "Normal") ||
        (V2Dist_[0] == "normal") ||
        (V2Dist_[0] == "N") ||
        (V2Dist_[0] == "n")){
      u2_ = normU(V2_, V2_VAR_, V2b, V2b_VAR, V2bmult, V2_V2b_cov, V2_0, V2b_0);
    }else if((V2Dist_[0] == "T") ||
      (V2Dist_[0] == "t")){
      if(V2_DF.isNull()){
        Rcout << "Degrees of Freedom must be specified for t-distribution" << "\n";
        return 1;
      }
      u2_ = tU(V2_, V2_VAR_, NumericVector(V2_DF)[0], V2b, V2b_VAR, V2bmult, V2_V2b_cov, V2_0, V2b_0, 1e5);
      // try removing 1e5 at the package build
    }
  }else{
    u1_ = NumericVector(u1)[0];
    u2_ = NumericVector(u2)[0];
  }
  
  // Put both u values in lower left corner of region.
  u1_ = std::min(u1_, 1.0 - u1_);
  u2_ = std::min(u2_, 1.0 - u2_);
  // Here U1 and U2 have been calculated, so we can run ps-test;
 
  return std::max(u1_*2.0, u2_*2.0);
  
}
