% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gm.R
\name{gm}
\alias{gm}
\title{Graph Matching Methods}
\usage{
gm(A, B, seeds = NULL, similarity = NULL, method = "indefinite", ...)
}
\arguments{
\item{A}{A matrix, igraph object, or list of either.}

\item{B}{A matrix, igraph object, or list of either.}

\item{seeds}{A vector of integers or logicals, a matrix or a data frame. If
the seed pairs have the same indices in both graphs then seeds can be a
vector. If not, seeds must be a matrix or a data frame, with the first
column being the indices of \eqn{G_1} and the second column being the
corresponding indices of \eqn{G_2}.}

\item{similarity}{A matrix. An \code{n-by-n} matrix containing vertex
similarities. Mandatory for the "IsoRank" method.}

\item{method}{Choice for graph matching methods. One of "indefinite",
"convex", "PATH", "percolation", "IsoRank", "Umeyama", or a user-defined
graph matching function. Please check Details and Examples sections for
instructions on how to define your own function.}

\item{...}{Arguments passed to graph matching methods. Please refer to
Details section for more information.}
}
\value{
\code{gm} returns an object of class "\code{\link{graphMatch}}".
See \link{graphMatch-class} and links therein for details
on the \code{graphMatch} class.

Please also refer to the help page for each implemented method, i.e.
\link[=graph_match_indefinite]{"indefinite"},
\link[=graph_match_convex]{"convex"},
\link[=graph_match_PATH]{"PATH"},
\link[=graph_match_percolation]{"percolation"},
\link[=graph_match_IsoRank]{"IsoRank"},
and \link[=graph_match_Umeyama]{"Umeyama"}
for details on the corresponding returned list.
}
\description{
\code{gm} is used to match a pair of given graphs, with
  specifications of the adjacency matrices of for a pair of graphs, possible
  prior knowledge, and a graph matching method.
}
\details{
If \code{method} is a function, it should take two matrices or
  igraph objects, seeds and similarity scores as arguments for minimum.
  Additionally, it can also take other arguments if needed. The self-defined
  function should return a graphMatch class object with matching
  correspondence, sizes of two input graphs, matching formula, and other
  algorithm hyperparameter details.

  The \code{method} argument can also take one of the implemented algorithms,
  including \link[=graph_match_indefinite]{"indefinite"},
  \link[=graph_match_convex]{"convex"}, \link[=graph_match_PATH]{"PATH"},
  \link[=graph_match_percolation]{"percolation"}, \link[=graph_match_IsoRank]{"IsoRank"},
  and \link[=graph_match_Umeyama]{"Umeyama"}.
  In this case, one can pass additional arguments to the \code{gm} function
  according to the specified method.
  For a detailed list of additional arguments for each one of the implemented method,
  please click on the corresponding method name for its help page.
  
  Most graph matching functions include as list elements additional details
  about the match. Call \code{names()} on a \code{graphMatch} object to see
  the available details. As an example, PATH, IsoRank, Umeyama, Indefinite,
  and Convex each include \code{soft}, which is the matrix found by the
  algorithm prior to projection onto the set of permutation matrices.
  Similarly, PATH, Indefinite, and Convex return \code{iter}, the number of
  iterations, and IsoRank (with greedy LAP) and Percolation return
  \code{match_order}, the order that the node-pairs were added to the match.
}
\examples{
# match G_1 & G_2 with some known node pairs as seeds
set.seed(123)
cgnp_pair <- sample_correlated_gnp_pair(n = 10, corr =  0.5, p =  0.5)
g1 <- cgnp_pair$graph1
g2 <- cgnp_pair$graph2
seeds <- 1:10 <= 4

m_rds <- gm(g1, g2, seeds, method = "indefinite", start = "rds", max_iter = 20)
summary(m_rds, g1, g2, true_label = 1:10)


# match two multi-layer graphs
set.seed(123)
gp_list <- replicate(3, sample_correlated_gnp_pair(20, .3, .5), simplify = FALSE)
A <- lapply(gp_list, function(gp)gp[[1]])
B <- lapply(gp_list, function(gp)gp[[2]])

m_perco <- gm(A, B, seeds, method = "percolation", ExpandWhenStuck = FALSE)
summary(m_perco, A, B)

sim <- as.matrix(init_start(start = "bari", nns = 20, soft_seeds = 1:5))
m_Iso <- gm(A, B, similarity = sim, method = "IsoRank", lap_method = "greedy")
summary(m_Iso, A, B)

# customized graph matching algorithm
graph_match_rand <- function(A, B, seeds = NULL, similarity = NULL, rand_seed){
  nm <- min(nrow(A), nrow(B))
  set.seed(rand_seed)
  m <- data.frame(sample(nrow(A), nm), corr_B = sample(nrow(B), nm))
  m <- as.graphMatch(m)
  m$rand_seed <- rand_seed
  m
}

m_self <- gm(g1, g2, method = graph_match_rand, rand_seed = 123)
summary(m_self, g1, g2)

}
