% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hypothesize.R
\name{union_test}
\alias{union_test}
\title{Union Test (OR via De Morgan's Law)}
\usage{
union_test(...)
}
\arguments{
\item{...}{\code{hypothesis_test} objects or numeric p-values to combine.}
}
\value{
A \code{hypothesis_test} object of subclass \code{union_test} containing:
\describe{
\item{stat}{The minimum p-value (used as the test statistic)}
\item{p.value}{\eqn{\min(p_1, \ldots, p_k)}}
\item{dof}{Number of component tests}
\item{n_tests}{Number of tests combined}
\item{component_pvals}{Vector of individual p-values}
}
}
\description{
Combines hypothesis tests using the OR rule: rejects when ANY component
test rejects.
}
\details{
The union test implements the OR operation in the Boolean algebra of
hypothesis tests. It is defined via De Morgan's law:

\deqn{\text{union}(t_1, \ldots, t_k) =
  \text{NOT}(\text{AND}(\text{NOT}(t_1), \ldots, \text{NOT}(t_k)))}

This is not an approximation --- it is the definition. The implementation
is literally the De Morgan law applied to \code{\link[=complement_test]{complement_test()}} and
\code{\link[=intersection_test]{intersection_test()}}.

The resulting p-value is \eqn{\min(p_1, \ldots, p_k)}.
}
\section{Multiplicity Warning}{

The uncorrected \eqn{\min(p)} is anti-conservative when testing multiple
hypotheses. If you need to control the family-wise error rate, apply
\code{\link[=adjust_pval]{adjust_pval()}} to the component tests before combining, or use
\code{\link[=fisher_combine]{fisher_combine()}} which pools evidence differently.

The raw union test is appropriate when you genuinely want to reject a
global null if any sub-hypothesis is false, without multiplicity
correction --- for example, in screening or exploratory analysis.
}

\section{Boolean Algebra}{

Together with \code{\link[=intersection_test]{intersection_test()}} (AND) and \code{\link[=complement_test]{complement_test()}} (NOT),
this forms a complete Boolean algebra over hypothesis tests:

\itemize{
\item AND: \code{\link[=intersection_test]{intersection_test()}} --- reject when all reject
\item OR: \code{union_test()} --- reject when any rejects
\item NOT: \code{\link[=complement_test]{complement_test()}} --- reject when original fails to reject
}

De Morgan's laws hold by construction:
\itemize{
\item \code{union(a, b) = NOT(AND(NOT(a), NOT(b)))}
\item \code{intersection(a, b) = NOT(OR(NOT(a), NOT(b)))}
}
}

\examples{
# Screen three biomarkers: reject if ANY is significant
t1 <- wald_test(estimate = 0.5, se = 0.3)
t2 <- wald_test(estimate = 2.1, se = 0.8)
t3 <- wald_test(estimate = 1.0, se = 0.4)
union_test(t1, t2, t3)

# De Morgan's law in action
a <- wald_test(estimate = 2.0, se = 1.0)
b <- wald_test(estimate = 1.5, se = 0.8)
# These are equivalent:
pval(union_test(a, b))
pval(complement_test(intersection_test(complement_test(a), complement_test(b))))

}
\seealso{
\code{\link[=intersection_test]{intersection_test()}} for AND, \code{\link[=complement_test]{complement_test()}} for NOT,
\code{\link[=fisher_combine]{fisher_combine()}} for evidence pooling
}
