/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Aprivate.h"  
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5Iprivate.h"  
#include "H5FOprivate.h" 
#include "H5Lprivate.h"  
#include "H5MFprivate.h" 
#include "H5MMprivate.h" 
#include "H5Opkg.h"      
#include "H5Pprivate.h"  
#include "H5SLprivate.h" 

typedef struct H5O_copy_search_comm_dt_key_t {
    H5T_t        *dt;     
    unsigned long fileno; 
} H5O_copy_search_comm_dt_key_t;

typedef struct H5O_copy_search_comm_dt_ud_t {
    H5SL_t    *dst_dt_list;  
    H5G_loc_t *dst_root_loc; 
    H5O_loc_t  obj_oloc;     
} H5O_copy_search_comm_dt_ud_t;

static herr_t H5O__copy_free_addrmap_cb(void *item, void *key, void *op_data);
static herr_t H5O__copy_header_real(const H5O_loc_t *oloc_src, H5O_loc_t *oloc_dst ,
                                    H5O_copy_t *cpy_info, H5O_type_t *obj_type, void **udata);
static herr_t H5O__copy_header(const H5O_loc_t *oloc_src, H5O_loc_t *oloc_dst , hid_t ocpypl_id,
                               hid_t lcpl_id);
static herr_t H5O__copy_obj(H5G_loc_t *src_loc, H5G_loc_t *dst_loc, const char *dst_name, hid_t ocpypl_id,
                            hid_t lcpl_id);
static herr_t H5O__copy_free_comm_dt_cb(void *item, void *key, void *op_data);
static int    H5O__copy_comm_dt_cmp(const void *dt1, const void *dt2);
static herr_t H5O__copy_search_comm_dt_cb(hid_t group, const char *name, const H5L_info2_t *linfo,
                                          void *udata);
static htri_t H5O__copy_search_comm_dt(H5F_t *file_src, H5O_t *oh_src, H5O_loc_t *oloc_dst ,
                                       H5O_copy_t *cpy_info);
static herr_t H5O__copy_insert_comm_dt(H5F_t *file_src, H5O_t *oh_src, H5O_loc_t *oloc_dst,
                                       H5O_copy_t *cpy_info);

H5FL_DEFINE(H5O_addr_map_t);

H5FL_DEFINE(H5O_copy_search_comm_dt_key_t);

H5FL_DEFINE(haddr_t);

herr_t
H5O__copy(const H5G_loc_t *loc, const char *src_name, H5G_loc_t *dst_loc, const char *dst_name,
          hid_t ocpypl_id, hid_t lcpl_id)
{
    H5G_loc_t  src_loc;             
    H5G_name_t src_path;            
    H5O_loc_t  src_oloc;            
    bool       dst_exists;          
    bool       loc_found = false;   
    bool       obj_open  = false;   
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(loc);
    assert(src_name && *src_name);
    assert(dst_loc);
    assert(dst_name && *dst_name);

    
    dst_exists = false;
    if (H5L_exists_tolerant(dst_loc, dst_name, &dst_exists) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "unable to check if destination name exists");
    if (dst_exists)
        HGOTO_ERROR(H5E_OHDR, H5E_EXISTS, FAIL, "destination object already exists");

    
    src_loc.oloc = &src_oloc;
    src_loc.path = &src_path;
    H5G_loc_reset(&src_loc);

    
    if (H5G_loc_find(loc, src_name, &src_loc ) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "source object not found");
    loc_found = true;

    
    if (H5O_open(&src_oloc) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTOPENOBJ, FAIL, "unable to open object");
    obj_open = true;

    
    if (H5O__copy_obj(&src_loc, dst_loc, dst_name, ocpypl_id, lcpl_id) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, FAIL, "unable to copy object");

done:
    if (loc_found && H5G_loc_free(&src_loc) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTRELEASE, FAIL, "can't free location");
    if (obj_open && H5O_close(&src_oloc, NULL) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CLOSEERROR, FAIL, "unable to release object header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__copy_header_real(const H5O_loc_t *oloc_src, H5O_loc_t *oloc_dst , H5O_copy_t *cpy_info,
                      H5O_type_t *obj_type, void **udata )
{
    H5O_addr_map_t *addr_map = NULL; 
    H5O_t          *oh_src   = NULL; 
    H5O_t          *oh_dst   = NULL; 
    unsigned        mesgno   = 0;
    haddr_t         addr_new = HADDR_UNDEF;
    bool           *deleted  = NULL; 
    bool        inserted = false; 
    size_t      null_msgs;        
    size_t      orig_dst_msgs;    
    H5O_mesg_t *mesg_src;         
    H5O_mesg_t *mesg_dst;         
    const H5O_msg_class_t *copy_type;        
    const H5O_obj_class_t *obj_class = NULL; 
    void                  *cpy_udata = NULL; 
    uint64_t               dst_oh_size;      
    size_t                 dst_oh_null;      
    size_t                 dst_oh_gap;       
    uint8_t               *current_pos;      
    size_t                 msghdr_size;
    herr_t                 ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE_TAG(oloc_src->addr)

    assert(oloc_src);
    assert(oloc_src->file);
    assert(H5_addr_defined(oloc_src->addr));
    assert(oloc_dst->file);
    assert(cpy_info);

    
    if (NULL == (obj_class = H5O__obj_class(oloc_src)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL, "unable to determine object type");

    
    cpy_info->shared_fo = H5FO_opened(oloc_src->file, oloc_src->addr);

    
    if (NULL == (oh_src = H5O_protect(oloc_src, H5AC__READ_ONLY_FLAG, false)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to load object header");

    
    if (obj_class->get_copy_file_udata && (NULL == (cpy_udata = (obj_class->get_copy_file_udata)())))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL, "unable to retrieve copy user data");

    
    if (cpy_info->merge_comm_dt && obj_class->type == H5O_TYPE_NAMED_DATATYPE) {
        unsigned long fileno_src; 
        unsigned long fileno_dst; 
        htri_t        merge;      

        
        H5F_GET_FILENO(oloc_src->file, fileno_src);
        H5F_GET_FILENO(oloc_dst->file, fileno_dst);
        if (fileno_src == fileno_dst) {
            merge          = true;
            oloc_dst->addr = oloc_src->addr;
        } 
        else
            
            if ((merge = H5O__copy_search_comm_dt(oloc_src->file, oh_src, oloc_dst, cpy_info)) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't search for matching committed datatype");

        if (merge) {
            
            
            if (NULL == (addr_map = H5FL_MALLOC(H5O_addr_map_t)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

            
            addr_map->src_obj_pos.fileno = fileno_src;
            addr_map->src_obj_pos.addr   = oloc_src->addr;
            addr_map->dst_addr           = oloc_dst->addr;
            addr_map->is_locked          = true; 
            addr_map->inc_ref_count      = 0;    
            addr_map->obj_class          = obj_class;
            addr_map->udata              = cpy_udata;

            
            if (H5SL_insert(cpy_info->map_list, addr_map, &(addr_map->src_obj_pos)) < 0) {
                addr_map = H5FL_FREE(H5O_addr_map_t, addr_map);
                HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, FAIL, "can't insert object into skip list");
            } 

            HGOTO_DONE(SUCCEED);
        } 
    }     

    
    if (H5O__flush_msgs(oloc_src->file, oh_src) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTFLUSH, FAIL, "unable to flush object header messages");

    
    if (NULL == (oh_dst = H5FL_CALLOC(H5O_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    oh_dst->version = oh_src->version;

    
    if (oh_dst->version > H5O_obj_ver_bounds[H5F_HIGH_BOUND(oloc_dst->file)])
        HGOTO_ERROR(H5E_OHDR, H5E_BADRANGE, FAIL, "destination object header version out of bounds");

    oh_dst->flags          = oh_src->flags;
    oh_dst->link_msgs_seen = oh_src->link_msgs_seen;
    oh_dst->attr_msgs_seen = oh_src->attr_msgs_seen;
    oh_dst->sizeof_size    = H5F_SIZEOF_SIZE(oloc_dst->file);
    oh_dst->sizeof_addr    = H5F_SIZEOF_ADDR(oloc_dst->file);
    oh_dst->swmr_write     = !!(H5F_INTENT(oloc_dst->file) & H5F_ACC_SWMR_WRITE);

    
    oh_dst->atime = oh_src->atime;
    oh_dst->mtime = oh_src->mtime;
    oh_dst->ctime = oh_src->ctime;
    oh_dst->btime = oh_src->btime;

    
    oh_dst->max_compact = oh_src->max_compact;
    oh_dst->min_dense   = oh_src->min_dense;

    
    if (oh_dst->swmr_write) {
        
        if (NULL == (oh_dst->proxy = H5AC_proxy_entry_create()))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTCREATE, FAIL, "can't create object header proxy");
    } 
    else
        oh_dst->proxy = NULL;

    
    oh_dst->alloc_nchunks = oh_dst->nchunks = 0;

    
    if (NULL == (oh_dst->chunk = H5FL_SEQ_MALLOC(H5O_chunk_t, (size_t)1)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    oh_dst->alloc_nchunks = 1;

    
    if (NULL == (deleted = (bool *)H5MM_malloc(sizeof(bool) * oh_src->nmesgs)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");
    memset(deleted, false, sizeof(bool) * oh_src->nmesgs);

    
    null_msgs = 0;
    for (mesgno = 0; mesgno < oh_src->nmesgs; mesgno++) {
        
        mesg_src = &(oh_src->mesg[mesgno]);

        
        assert(!mesg_src->dirty); 

        
        copy_type = mesg_src->type;

        
        if (H5O_UNKNOWN_ID != mesg_src->type->id && H5O_NULL_ID != mesg_src->type->id)
            if (0 == mesg_src->raw_size)
                HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "message of type '%s' has zero size",
                            mesg_src->type->name);

        
        if (H5O_CONT_ID == mesg_src->type->id || H5O_NULL_ID == mesg_src->type->id) {
            deleted[mesgno] = true;
            ++null_msgs;
            copy_type = H5O_MSG_NULL;
        } 
        assert(copy_type);

        if (copy_type->pre_copy_file) {
            
            H5O_LOAD_NATIVE(oloc_src->file, 0, oh_src, mesg_src, FAIL)

            
            cpy_info->file_dst = oloc_dst->file;

            
            if ((copy_type->pre_copy_file)(oloc_src->file, mesg_src->native, &(deleted[mesgno]), cpy_info,
                                           cpy_udata) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL,
                            "unable to perform 'pre copy' operation on message");

            
            if (deleted[mesgno])
                
                ++null_msgs;
        } 
    }     

    
    if (cpy_info->preserve_null)
        oh_dst->alloc_nmesgs = oh_dst->nmesgs = oh_src->nmesgs;
    else
        oh_dst->alloc_nmesgs = oh_dst->nmesgs = (oh_src->nmesgs - null_msgs);

    
    if (oh_dst->alloc_nmesgs > 0)
        if (NULL == (oh_dst->mesg = H5FL_SEQ_CALLOC(H5O_mesg_t, oh_dst->alloc_nmesgs)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    null_msgs = 0;
    for (mesgno = 0; mesgno < oh_dst->nmesgs; mesgno++) {
        
        if (false == cpy_info->preserve_null) {
            while (deleted[mesgno + null_msgs]) {
                ++null_msgs;
                assert(mesgno + null_msgs < oh_src->nmesgs);
            } 
        }     

        
        mesg_src = &(oh_src->mesg[mesgno + null_msgs]);
        mesg_dst = &(oh_dst->mesg[mesgno]);

        
        mesg_dst->crt_idx  = mesg_src->crt_idx;
        mesg_dst->flags    = mesg_src->flags;
        mesg_dst->raw_size = mesg_src->raw_size;
        mesg_dst->type     = mesg_src->type;

        
        if (cpy_info->preserve_null && deleted[mesgno]) {
            mesg_dst->type  = H5O_MSG_NULL;
            mesg_dst->flags = 0;
            mesg_dst->dirty = true;
        } 

        
        
        copy_type = mesg_dst->type;
        assert(copy_type);

        
        if (copy_type->copy_file) {
            bool     recompute_size; 
            unsigned mesg_flags;     

            
            H5O_LOAD_NATIVE(oloc_src->file, 0, oh_src, mesg_src, FAIL)

            
            mesg_flags = (unsigned)mesg_dst->flags & ~H5O_MSG_FLAG_SHARED & ~H5O_MSG_FLAG_SHAREABLE;

            
            recompute_size = false;
            if (NULL == (mesg_dst->native =
                             H5O__msg_copy_file(copy_type, oloc_src->file, mesg_src->native, oloc_dst->file,
                                                &recompute_size, &mesg_flags, cpy_info, cpy_udata)))
                HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, FAIL, "unable to copy object header message");

            
            if (!(mesg_flags & H5O_MSG_FLAG_SHARED) != !(mesg_dst->flags & H5O_MSG_FLAG_SHARED))
                recompute_size = true;

            
            mesg_dst->flags = (uint8_t)mesg_flags;

            
            
            if (recompute_size)
                mesg_dst->raw_size = H5O_ALIGN_OH(
                    oh_dst, H5O_msg_raw_size(oloc_dst->file, mesg_dst->type->id, false, mesg_dst->native));

            
            mesg_dst->dirty = true;
        } 
    }     

    

    

    
    dst_oh_size = 0;
    for (mesgno = 0; mesgno < oh_dst->nmesgs; mesgno++) {
        dst_oh_size += (uint64_t)H5O_SIZEOF_MSGHDR_OH(oh_dst);
        dst_oh_size += oh_dst->mesg[mesgno].raw_size;
    } 

    
    if (oh_dst->version > H5O_VERSION_1) {
        
        oh_dst->flags = (uint8_t)(oh_dst->flags & ~H5O_HDR_CHUNK0_SIZE);

        
        if (dst_oh_size > 4294967295)
            oh_dst->flags |= H5O_HDR_CHUNK0_8;
        else if (dst_oh_size > 65535)
            oh_dst->flags |= H5O_HDR_CHUNK0_4;
        else if (dst_oh_size > 255)
            oh_dst->flags |= H5O_HDR_CHUNK0_2;
    } 

    
    dst_oh_gap = dst_oh_null = 0;
    if (dst_oh_size < H5O_MIN_SIZE) {
        size_t delta = (size_t)(H5O_MIN_SIZE - dst_oh_size); 

        
        assert((oh_dst->flags & H5O_HDR_CHUNK0_SIZE) == H5O_HDR_CHUNK0_1);

        
        if ((oh_dst->version > H5O_VERSION_1) && (delta < H5O_SIZEOF_MSGHDR_OH(oh_dst)))
            dst_oh_gap = delta;
        else {
            
            if (delta < H5O_SIZEOF_MSGHDR_OH(oh_dst))
                delta = H5O_SIZEOF_MSGHDR_OH(oh_dst);

            dst_oh_null = delta;
        }

        
        dst_oh_size += delta;

        
        assert(dst_oh_size <= 255);
    } 

    
    dst_oh_size += (uint64_t)H5O_SIZEOF_HDR(oh_dst);

    
    if (HADDR_UNDEF ==
        (oh_dst->chunk[0].addr = H5MF_alloc(oloc_dst->file, H5FD_MEM_OHDR, (hsize_t)dst_oh_size)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "file allocation failed for object header");
    addr_new = oh_dst->chunk[0].addr;

    
    
    if (NULL == (oh_dst->chunk[0].image = H5FL_BLK_CALLOC(chunk_image, (size_t)dst_oh_size)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    oh_dst->chunk[0].size = (size_t)dst_oh_size;
    oh_dst->chunk[0].gap  = dst_oh_gap;

    
    oh_dst->nchunks = 1;

    
    assert(H5O_SIZEOF_MSGHDR_OH(oh_src) == H5O_SIZEOF_MSGHDR_OH(oh_dst));
    msghdr_size = H5O_SIZEOF_MSGHDR_OH(oh_dst);

    current_pos = oh_dst->chunk[0].image;

    
    if (oh_dst->version > H5O_VERSION_1)
        H5MM_memcpy(current_pos, H5O_HDR_MAGIC, (size_t)H5_SIZEOF_MAGIC);
    current_pos += H5O_SIZEOF_HDR(oh_dst) - H5O_SIZEOF_CHKSUM_OH(oh_dst);

    
    null_msgs = 0;
    for (mesgno = 0; mesgno < oh_dst->nmesgs; mesgno++) {
        
        if (false == cpy_info->preserve_null) {
            while (deleted[mesgno + null_msgs]) {
                ++null_msgs;
                assert(mesgno + null_msgs < oh_src->nmesgs);
            } 
        }     

        
        mesg_src = &(oh_src->mesg[mesgno + null_msgs]);
        mesg_dst = &(oh_dst->mesg[mesgno]);

        
        if (!mesg_dst->dirty)
            
            H5MM_memcpy(current_pos, mesg_src->raw - msghdr_size, msghdr_size + mesg_src->raw_size);

        
        mesg_dst->raw = current_pos + msghdr_size;

        
        current_pos += mesg_dst->raw_size + msghdr_size;
    } 

    
    orig_dst_msgs = oh_dst->nmesgs;

    
    if (dst_oh_null > 0) {
        size_t null_idx; 

        
        if (oh_dst->nmesgs + 1 > oh_dst->alloc_nmesgs)
            if (H5O__alloc_msgs(oh_dst, (size_t)1) < 0)
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "can't allocate more space for messages");

        
        
        null_idx                        = oh_dst->nmesgs++;
        oh_dst->mesg[null_idx].type     = H5O_MSG_NULL;
        oh_dst->mesg[null_idx].dirty    = true;
        oh_dst->mesg[null_idx].native   = NULL;
        oh_dst->mesg[null_idx].raw      = current_pos + msghdr_size;
        oh_dst->mesg[null_idx].raw_size = dst_oh_null - msghdr_size;
        oh_dst->mesg[null_idx].chunkno  = 0;
    } 

    
    assert(current_pos + dst_oh_gap + dst_oh_null + H5O_SIZEOF_CHKSUM_OH(oh_dst) ==
           (size_t)dst_oh_size + oh_dst->chunk[0].image);

    
    assert(H5_addr_defined(addr_new));
    oloc_dst->addr = addr_new;

    
    if (cpy_info->merge_comm_dt && obj_class->type == H5O_TYPE_NAMED_DATATYPE)
        if (H5O__copy_insert_comm_dt(oloc_src->file, oh_src, oloc_dst, cpy_info) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't insert committed datatype into destination list");

    
    if (NULL == (addr_map = H5FL_MALLOC(H5O_addr_map_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    
    H5F_GET_FILENO(oloc_src->file, addr_map->src_obj_pos.fileno);
    addr_map->src_obj_pos.addr = oloc_src->addr;
    addr_map->dst_addr         = oloc_dst->addr;
    addr_map->is_locked        = true; 
    addr_map->inc_ref_count    = 0;    
    addr_map->obj_class        = obj_class;
    addr_map->udata            = cpy_udata;

    
    if (H5SL_insert(cpy_info->map_list, addr_map, &(addr_map->src_obj_pos)) < 0) {
        addr_map = H5FL_FREE(H5O_addr_map_t, addr_map);
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, FAIL, "can't insert object into skip list");
    } 

    
    null_msgs = 0;
    for (mesgno = 0; mesgno < orig_dst_msgs; mesgno++) {
        
        if (false == cpy_info->preserve_null) {
            while (deleted[mesgno + null_msgs]) {
                ++null_msgs;
                assert(mesgno + null_msgs < oh_src->nmesgs);
            } 
        }     

        
        mesg_src = &(oh_src->mesg[mesgno + null_msgs]);
        mesg_dst = &(oh_dst->mesg[mesgno]);

        
        
        copy_type = mesg_dst->type;
        assert(copy_type);

        if (copy_type->post_copy_file && mesg_src->native) {
            unsigned mesg_flags; 

            
            assert(mesg_dst->type == mesg_src->type);
            assert(mesg_dst->native);

            
            mesg_flags = (unsigned)mesg_dst->flags;

            
            cpy_info->oh_dst = oh_dst;

            
            if ((copy_type->post_copy_file)(oloc_src, mesg_src->native, oloc_dst, mesg_dst->native,
                                            &mesg_flags, cpy_info) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL,
                            "unable to perform 'post copy' operation on message");

            
            assert(mesg_flags == (unsigned)mesg_dst->flags);
        } 
    }     

    
    addr_map->is_locked = false;

    
    if (addr_map->inc_ref_count) {
        H5_CHECK_OVERFLOW(addr_map->inc_ref_count, hsize_t, unsigned);
        oh_dst->nlink += (unsigned)addr_map->inc_ref_count;
    } 

    
    if (H5AC_retag_copied_metadata(oloc_dst->file, oloc_dst->addr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTTAG, FAIL, "unable to re-tag metadata entries");

    
    H5_BEGIN_TAG(oloc_dst->addr)

    
    if (H5AC_insert_entry(oloc_dst->file, H5AC_OHDR, oloc_dst->addr, oh_dst, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR_TAG(H5E_OHDR, H5E_CANTINSERT, FAIL, "unable to cache object header");
    oh_dst   = NULL;
    inserted = true;

    
    H5_END_TAG

    
    if (obj_type) {
        assert(udata);
        *obj_type = obj_class->type;
        *udata    = cpy_udata;
    } 

done:
    
    if (deleted)
        H5MM_free(deleted);

    
    if (oh_src && H5O_unprotect(oloc_src, oh_src, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to release object header");

    
    if (ret_value < 0) {
        if (oh_dst && !inserted) {
            if (H5O__free(oh_dst, true) < 0)
                HDONE_ERROR(H5E_OHDR, H5E_CANTFREE, FAIL, "unable to destroy object header data");
            if (H5O_loc_reset(oloc_dst) < 0)
                HDONE_ERROR(H5E_OHDR, H5E_CANTFREE, FAIL, "unable to destroy object header data");
        } 

        if (addr_map == NULL && cpy_udata) {
            if (obj_class && obj_class->free_copy_file_udata)
                obj_class->free_copy_file_udata(cpy_udata);
        } 
    }

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5O_copy_header_map(const H5O_loc_t *oloc_src, H5O_loc_t *oloc_dst , H5O_copy_t *cpy_info,
                    bool inc_depth, H5O_type_t *obj_type, void **udata )
{
    H5O_addr_map_t *addr_map = NULL; 
    H5_obj_t        src_obj_pos;     
    bool            inc_link;        
    herr_t          ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(oloc_src);
    assert(oloc_src->file);
    assert(oloc_dst);
    assert(oloc_dst->file);
    assert(cpy_info);

    
    H5F_GET_FILENO(oloc_src->file, src_obj_pos.fileno);
    src_obj_pos.addr = oloc_src->addr;

    
    addr_map = (H5O_addr_map_t *)H5SL_search(cpy_info->map_list, &src_obj_pos);

    
    if (addr_map == NULL) {
        

        
        
        if (inc_depth)
            cpy_info->curr_depth++;

        
        if (H5O__copy_header_real(oloc_src, oloc_dst, cpy_info, obj_type, udata) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, FAIL, "unable to copy object");

        
        if (inc_depth)
            cpy_info->curr_depth--;

        
        inc_link = true;

        
        ret_value++;
    } 
    else {
        
        oloc_dst->addr = addr_map->dst_addr;

        
        if (obj_type) {
            assert(udata);
            *obj_type = addr_map->obj_class->type;
            *udata    = addr_map->udata;
        } 

        
        if (addr_map->is_locked) {
            addr_map->inc_ref_count++;
            inc_link = false;
        } 
        else
            inc_link = true;
    } 

    
    if (inc_link)
        if (H5O_link(oloc_dst, 1) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL, "unable to increment object link count");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__copy_free_addrmap_cb(void *_item, void H5_ATTR_UNUSED *key, void H5_ATTR_UNUSED *op_data)
{
    H5O_addr_map_t *item = (H5O_addr_map_t *)_item;

    FUNC_ENTER_PACKAGE_NOERR

    assert(item);

    
    if (item->udata) {
        assert(item->obj_class);
        assert(item->obj_class->free_copy_file_udata);
        (item->obj_class->free_copy_file_udata)(item->udata);
    } 

    
    item = H5FL_FREE(H5O_addr_map_t, item);

    FUNC_LEAVE_NOAPI(0)
} 

static herr_t
H5O__copy_header(const H5O_loc_t *oloc_src, H5O_loc_t *oloc_dst , hid_t ocpypl_id, hid_t lcpl_id)
{
    H5O_copy_t                   cpy_info;       
    H5P_genplist_t              *ocpy_plist;     
    H5O_copy_dtype_merge_list_t *dt_list = NULL; 
    H5O_mcdt_cb_info_t           cb_info;        
    unsigned                     cpy_option = 0; 
    herr_t                       ret_value  = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(oloc_src);
    assert(oloc_src->file);
    assert(H5_addr_defined(oloc_src->addr));
    assert(oloc_dst->file);

    
    memset(&cpy_info, 0, sizeof(H5O_copy_t));

    
    if (NULL == (ocpy_plist = (H5P_genplist_t *)H5I_object(ocpypl_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list");

    
    if (H5P_get(ocpy_plist, H5O_CPY_OPTION_NAME, &cpy_option) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get object copy flag");

    
    if (H5P_peek(ocpy_plist, H5O_CPY_MERGE_COMM_DT_LIST_NAME, &dt_list) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get merge committed datatype list");

    
    if (H5P_get(ocpy_plist, H5O_CPY_MCDT_SEARCH_CB_NAME, &cb_info) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get callback info");

    
    if ((cpy_option & H5O_COPY_SHALLOW_HIERARCHY_FLAG) > 0) {
        cpy_info.copy_shallow = true;
        cpy_info.max_depth    = 1;
    } 
    else
        cpy_info.max_depth = -1; 
    cpy_info.curr_depth = 0;
    if ((cpy_option & H5O_COPY_EXPAND_SOFT_LINK_FLAG) > 0)
        cpy_info.expand_soft_link = true;
    if ((cpy_option & H5O_COPY_EXPAND_EXT_LINK_FLAG) > 0)
        cpy_info.expand_ext_link = true;
    if ((cpy_option & H5O_COPY_EXPAND_REFERENCE_FLAG) > 0)
        cpy_info.expand_ref = true;
    if ((cpy_option & H5O_COPY_WITHOUT_ATTR_FLAG) > 0)
        cpy_info.copy_without_attr = true;
    if ((cpy_option & H5O_COPY_PRESERVE_NULL_FLAG) > 0)
        cpy_info.preserve_null = true;
    if ((cpy_option & H5O_COPY_MERGE_COMMITTED_DTYPE_FLAG) > 0)
        cpy_info.merge_comm_dt = true;

    
    cpy_info.dst_dt_suggestion_list = dt_list;

    
    cpy_info.mcdt_cb = cb_info.func;
    cpy_info.mcdt_ud = cb_info.user_data;

    
    cpy_info.lcpl_id = lcpl_id;

    
    if (NULL == (cpy_info.map_list = H5SL_create(H5SL_TYPE_OBJ, NULL)))
        HGOTO_ERROR(H5E_SLIST, H5E_CANTCREATE, FAIL, "cannot make skip list");

    
    if (H5O__copy_header_real(oloc_src, oloc_dst, &cpy_info, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, FAIL, "unable to copy object");

done:
    if (cpy_info.map_list)
        H5SL_destroy(cpy_info.map_list, H5O__copy_free_addrmap_cb, NULL);
    if (cpy_info.dst_dt_list)
        H5SL_destroy(cpy_info.dst_dt_list, H5O__copy_free_comm_dt_cb, NULL);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__copy_obj(H5G_loc_t *src_loc, H5G_loc_t *dst_loc, const char *dst_name, hid_t ocpypl_id, hid_t lcpl_id)
{
    H5G_name_t new_path;                 
    H5O_loc_t  new_oloc;                 
    H5G_loc_t  new_loc;                  
    H5F_t     *cached_dst_file;          
    bool       entry_inserted = false;   
    herr_t     ret_value      = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(src_loc);
    assert(src_loc->oloc->file);
    assert(dst_loc);
    assert(dst_loc->oloc->file);
    assert(dst_name);

    
    new_loc.oloc = &new_oloc;
    new_loc.path = &new_path;
    H5G_loc_reset(&new_loc);
    new_oloc.file = dst_loc->oloc->file;

    
    cached_dst_file = dst_loc->oloc->file;

    
    if (H5O__copy_header(src_loc->oloc, &new_oloc, ocpypl_id, lcpl_id) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, FAIL, "unable to copy object");

    
    dst_loc->oloc->file = cached_dst_file;

    
    if (H5L_link(dst_loc, dst_name, &new_loc, lcpl_id) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to insert link");
    entry_inserted = true;

done:
    
    if (entry_inserted)
        H5G_loc_free(&new_loc);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__copy_free_comm_dt_cb(void *item, void *_key, void H5_ATTR_UNUSED *_op_data)
{
    haddr_t                       *addr = (haddr_t *)item;
    H5O_copy_search_comm_dt_key_t *key  = (H5O_copy_search_comm_dt_key_t *)_key;

    FUNC_ENTER_PACKAGE_NOERR

    assert(addr);
    assert(key);
    assert(key->dt);

    key->dt = (H5T_t *)H5O_msg_free(H5O_DTYPE_ID, key->dt);
    key     = H5FL_FREE(H5O_copy_search_comm_dt_key_t, key);
    addr    = H5FL_FREE(haddr_t, addr);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static int
H5O__copy_comm_dt_cmp(const void *_key1, const void *_key2)
{
    const H5O_copy_search_comm_dt_key_t *key1      = (const H5O_copy_search_comm_dt_key_t *)_key1;
    const H5O_copy_search_comm_dt_key_t *key2      = (const H5O_copy_search_comm_dt_key_t *)_key2;
    int                                  ret_value = 0;

    FUNC_ENTER_PACKAGE_NOERR

    
    if (key1->fileno != key2->fileno) {
        if (key1->fileno < key2->fileno)
            HGOTO_DONE(-1);
        if (key1->fileno > key2->fileno)
            HGOTO_DONE(1);
    } 

    ret_value = H5T_cmp(key1->dt, key2->dt, false);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__copy_search_comm_dt_attr_cb(const H5A_t *attr, void *_udata)
{
    H5O_copy_search_comm_dt_ud_t  *udata        = (H5O_copy_search_comm_dt_ud_t *)_udata;
    H5T_t                         *dt           = NULL;    
    H5O_copy_search_comm_dt_key_t *key          = NULL;    
    haddr_t                       *addr         = NULL;    
    bool                           obj_inserted = false;   
    herr_t                         ret_value    = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(attr);
    assert(udata);
    assert(udata->dst_dt_list);
    assert(H5_addr_defined(udata->obj_oloc.addr));

    
    if (NULL == (dt = H5A_type(attr)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get attribute datatype");

    
    if (H5T_is_named(dt)) {
        
        if (NULL == (key = H5FL_MALLOC(H5O_copy_search_comm_dt_key_t)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

        
        if (NULL == (key->dt = (H5T_t *)H5O_msg_copy(H5O_DTYPE_ID, dt, NULL)))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL, "unable to copy datatype message");

        
        H5F_GET_FILENO(udata->obj_oloc.file, key->fileno);

        if (!H5SL_search(udata->dst_dt_list, key)) {
            
            if (NULL == (addr = H5FL_MALLOC(haddr_t)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

            
            *addr = ((H5O_shared_t *)(key->dt))->u.loc.oh_addr;
            if (H5SL_insert(udata->dst_dt_list, addr, key) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, FAIL, "can't insert object into skip list");
            obj_inserted = true;
        } 
    }     

done:
    
    if (!obj_inserted) {
        if (key) {
            if (key->dt)
                key->dt = (H5T_t *)H5O_msg_free(H5O_DTYPE_ID, key->dt);
            key = H5FL_FREE(H5O_copy_search_comm_dt_key_t, key);
        } 
        if (addr) {
            assert(ret_value < 0);
            addr = H5FL_FREE(haddr_t, addr);
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__copy_search_comm_dt_check(H5O_loc_t *obj_oloc, H5O_copy_search_comm_dt_ud_t *udata)
{
    H5O_copy_search_comm_dt_key_t *key          = NULL;  
    haddr_t                       *addr         = NULL;  
    bool                           obj_inserted = false; 
    H5A_attr_iter_op_t             attr_op;              
    const H5O_obj_class_t         *obj_class = NULL;     
    herr_t                         ret_value = SUCCEED;  

    FUNC_ENTER_PACKAGE

    
    assert(obj_oloc);
    assert(udata);
    assert(udata->dst_dt_list);
    assert(udata->dst_root_loc);

    
    if ((obj_class = H5O__obj_class(obj_oloc)) == NULL)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL, "unable to determine object type");

    
    if (obj_class->type == H5O_TYPE_NAMED_DATATYPE) {
        
        if (NULL == (key = H5FL_MALLOC(H5O_copy_search_comm_dt_key_t)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

        
        if (NULL == (key->dt = (H5T_t *)H5O_msg_read(obj_oloc, H5O_DTYPE_ID, NULL)))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't read DTYPE message");

        
        H5F_GET_FILENO(obj_oloc->file, key->fileno);

        
        if (!H5SL_search(udata->dst_dt_list, key)) {
            
            if (NULL == (addr = H5FL_MALLOC(haddr_t)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

            
            *addr = obj_oloc->addr;
            if (H5SL_insert(udata->dst_dt_list, addr, key) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, FAIL, "can't insert object into skip list");
            obj_inserted = true;
        } 
    }     
    else if (obj_class->type == H5O_TYPE_DATASET) {
        
        if (NULL == (key = H5FL_MALLOC(H5O_copy_search_comm_dt_key_t)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

        
        if (NULL == (key->dt = (H5T_t *)H5O_msg_read(obj_oloc, H5O_DTYPE_ID, NULL)))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't read DTYPE message");

        
        if (H5T_is_named(key->dt)) {
            
            H5F_GET_FILENO(obj_oloc->file, key->fileno);

            if (!H5SL_search(udata->dst_dt_list, key)) {
                
                if (NULL == (addr = H5FL_MALLOC(haddr_t)))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

                
                *addr = ((H5O_shared_t *)(key->dt))->u.loc.oh_addr;
                if (H5SL_insert(udata->dst_dt_list, addr, key) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, FAIL, "can't insert object into skip list");
                obj_inserted = true;
            } 
        }     
    }         

    
    attr_op.op_type      = H5A_ATTR_OP_LIB;
    attr_op.u.lib_op     = H5O__copy_search_comm_dt_attr_cb;
    udata->obj_oloc.file = obj_oloc->file;
    udata->obj_oloc.addr = obj_oloc->addr;
    if (H5O_attr_iterate_real((hid_t)-1, obj_oloc, H5_INDEX_NAME, H5_ITER_NATIVE, (hsize_t)0, NULL, &attr_op,
                              udata) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_BADITER, FAIL, "error iterating over attributes");

done:
    
    if (!obj_inserted) {
        if (key) {
            if (key->dt)
                key->dt = (H5T_t *)H5O_msg_free(H5O_DTYPE_ID, key->dt);
            key = H5FL_FREE(H5O_copy_search_comm_dt_key_t, key);
        } 
        if (addr) {
            assert(ret_value < 0);
            addr = H5FL_FREE(haddr_t, addr);
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__copy_search_comm_dt_cb(hid_t H5_ATTR_UNUSED group, const char *name, const H5L_info2_t *linfo,
                            void *_udata)
{
    H5O_copy_search_comm_dt_ud_t *udata =
        (H5O_copy_search_comm_dt_ud_t *)_udata; 
    H5G_loc_t  obj_loc;                         
    H5O_loc_t  obj_oloc;                        
    H5G_name_t obj_path;                        
    bool       obj_found = false;               
    herr_t     ret_value = H5_ITER_CONT;        

    FUNC_ENTER_PACKAGE

    
    assert(name);
    assert(linfo);
    assert(udata);
    assert(udata->dst_dt_list);
    assert(udata->dst_root_loc);

    
    if (linfo->type == H5L_TYPE_HARD) {
        
        obj_loc.oloc = &obj_oloc;
        obj_loc.path = &obj_path;
        H5G_loc_reset(&obj_loc);

        
        if (H5G_loc_find(udata->dst_root_loc, name, &obj_loc ) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_NOTFOUND, H5_ITER_ERROR, "object not found");
        obj_found = true;

        
        if (H5O__copy_search_comm_dt_check(&obj_oloc, udata) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, H5_ITER_ERROR, "can't check object");
    } 

done:
    
    if (obj_found && H5G_loc_free(&obj_loc) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTRELEASE, H5_ITER_ERROR, "can't free location");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5O__copy_search_comm_dt(H5F_t *file_src, H5O_t *oh_src, H5O_loc_t *oloc_dst ,
                         H5O_copy_t *cpy_info)
{
    H5O_copy_search_comm_dt_key_t *key = NULL;                  
    haddr_t                       *dst_addr;                    
    H5G_loc_t                      dst_root_loc = {NULL, NULL}; 
    H5O_copy_search_comm_dt_ud_t   udata;                       
    herr_t                         ret_value = false;           

    FUNC_ENTER_PACKAGE

    
    assert(oh_src);
    assert(oloc_dst);
    assert(oloc_dst->file);
    assert(H5F_ID_EXISTS(oloc_dst->file));
    assert(cpy_info);

    
    if (NULL == (key = H5FL_MALLOC(H5O_copy_search_comm_dt_key_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    if (NULL == (key->dt = (H5T_t *)H5O_msg_read_oh(file_src, oh_src, H5O_DTYPE_ID, NULL)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't read DTYPE message");

    
    H5F_GET_FILENO(oloc_dst->file, key->fileno);

    
    if (!cpy_info->dst_dt_list) {
        
        if (NULL == (cpy_info->dst_dt_list = H5SL_create(H5SL_TYPE_GENERIC, H5O__copy_comm_dt_cmp)))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTCREATE, FAIL, "can't create skip list for committed datatypes");

        
        if (cpy_info->dst_dt_suggestion_list) {
            H5O_copy_dtype_merge_list_t *suggestion = cpy_info->dst_dt_suggestion_list;
            H5G_loc_t                    obj_loc;  
            H5O_loc_t                    obj_oloc; 
            H5G_name_t                   obj_path; 

            
            if (NULL == (dst_root_loc.oloc = H5G_oloc(H5G_rootof(oloc_dst->file))))
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get object location for root group");
            if (NULL == (dst_root_loc.path = H5G_nameof(H5G_rootof(oloc_dst->file))))
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get path for root group");

            
            obj_loc.oloc = &obj_oloc;
            obj_loc.path = &obj_path;
            H5G_loc_reset(&obj_loc);

            
            udata.dst_dt_list   = cpy_info->dst_dt_list;
            udata.dst_root_loc  = &dst_root_loc;
            udata.obj_oloc.file = NULL;
            udata.obj_oloc.addr = HADDR_UNDEF;

            
            while (suggestion) {
                bool exists = false;

                
                if (H5G_loc_exists(&dst_root_loc, suggestion->path, &exists ) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTFIND, FAIL, "can't check object's existence");

                if (exists) {
                    
                    if (H5G_loc_find(&dst_root_loc, suggestion->path, &obj_loc ) < 0)
                        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't retrieve object location");

                    
                    if (H5O__copy_search_comm_dt_check(&obj_oloc, &udata) < 0) {
                        if (H5G_loc_free(&obj_loc) < 0)
                            HERROR(H5E_OHDR, H5E_CANTRELEASE, "can't free location");
                        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't check object");
                    } 

                    
                    if (H5G_loc_free(&obj_loc) < 0)
                        HGOTO_ERROR(H5E_OHDR, H5E_CANTRELEASE, FAIL, "can't free location");
                } 

                
                suggestion = suggestion->next;
            } 
        }     
    }

    if (!cpy_info->dst_dt_list_complete) {
        
        if (cpy_info->dst_dt_suggestion_list &&
            NULL != (dst_addr = (haddr_t *)H5SL_search(cpy_info->dst_dt_list, key))) {
            oloc_dst->addr = *dst_addr;
            ret_value      = true;
        } 
        else {
            H5O_mcdt_search_ret_t search_cb_ret = H5O_MCDT_SEARCH_CONT;

            
            if (cpy_info->mcdt_cb) {
                
                H5_BEFORE_USER_CB(FAIL)
                    {
                        search_cb_ret = cpy_info->mcdt_cb(cpy_info->mcdt_ud);
                    }
                H5_AFTER_USER_CB(FAIL)
                if (H5O_MCDT_SEARCH_ERROR == search_cb_ret)
                    HGOTO_ERROR(H5E_OHDR, H5E_CALLBACK, FAIL, "callback returned error");
            }

            if (search_cb_ret == H5O_MCDT_SEARCH_CONT) {
                
                
                if (!dst_root_loc.oloc) {
                    assert(!dst_root_loc.path);
                    if (NULL == (dst_root_loc.oloc = H5G_oloc(H5G_rootof(oloc_dst->file))))
                        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                                    "unable to get object location for root group");
                    if (NULL == (dst_root_loc.path = H5G_nameof(H5G_rootof(oloc_dst->file))))
                        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get path for root group");
                } 
                else
                    assert(dst_root_loc.path);

                
                udata.dst_dt_list   = cpy_info->dst_dt_list;
                udata.dst_root_loc  = &dst_root_loc;
                udata.obj_oloc.file = NULL;
                udata.obj_oloc.addr = HADDR_UNDEF;

                
                if (H5G_visit(&dst_root_loc, "/", H5_INDEX_NAME, H5_ITER_NATIVE, H5O__copy_search_comm_dt_cb,
                              &udata) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_BADITER, FAIL, "object visitation failed");
                cpy_info->dst_dt_list_complete = true;
            } 
            else if (search_cb_ret != H5O_MCDT_SEARCH_STOP)
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unknown return value for callback");
        } 
    }     

    
    if (cpy_info->dst_dt_list_complete) {
        if (NULL != (dst_addr = (haddr_t *)H5SL_search(cpy_info->dst_dt_list, key))) {
            oloc_dst->addr = *dst_addr;
            ret_value      = true;
        } 
    }     

done:
    if (key) {
        if (key->dt)
            key->dt = (H5T_t *)H5O_msg_free(H5O_DTYPE_ID, key->dt);
        key = H5FL_FREE(H5O_copy_search_comm_dt_key_t, key);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__copy_insert_comm_dt(H5F_t *file_src, H5O_t *oh_src, H5O_loc_t *oloc_dst, H5O_copy_t *cpy_info)
{
    H5O_copy_search_comm_dt_key_t *key       = NULL;    
    haddr_t                       *addr      = NULL;    
    herr_t                         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(oh_src);
    assert(oloc_dst);
    assert(oloc_dst->file);
    assert(oloc_dst->addr != HADDR_UNDEF);
    assert(cpy_info);
    assert(cpy_info->dst_dt_list);

    
    if (NULL == (key = H5FL_MALLOC(H5O_copy_search_comm_dt_key_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    if (NULL == (key->dt = (H5T_t *)H5O_msg_read_oh(file_src, oh_src, H5O_DTYPE_ID, NULL)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't read DTYPE message");

    
    H5F_GET_FILENO(oloc_dst->file, key->fileno);

    
    if (NULL == (addr = H5FL_MALLOC(haddr_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    *addr = oloc_dst->addr;
    if (H5SL_insert(cpy_info->dst_dt_list, addr, key) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, FAIL, "can't insert object into skip list");

done:
    if (ret_value < 0) {
        if (key) {
            if (key->dt)
                key->dt = (H5T_t *)H5O_msg_free(H5O_DTYPE_ID, key->dt);
            key = H5FL_FREE(H5O_copy_search_comm_dt_key_t, key);
        } 
        if (addr)
            addr = H5FL_FREE(haddr_t, addr);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 
