/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MFprivate.h" 
#include "H5MMprivate.h" 
#include "H5Opkg.h"      

static herr_t H5O__add_gap(H5F_t *f, H5O_t *oh, unsigned chunkno, bool *chk_dirtied, size_t idx,
                           uint8_t *new_gap_loc, size_t new_gap_size);
static herr_t H5O__eliminate_gap(H5O_t *oh, bool *chk_dirtied, H5O_mesg_t *mesg, uint8_t *new_gap_loc,
                                 size_t new_gap_size);
static herr_t H5O__alloc_null(H5F_t *f, H5O_t *oh, size_t null_idx, const H5O_msg_class_t *new_type,
                              void *new_native, size_t new_size);
static htri_t H5O__alloc_extend_chunk(H5F_t *f, H5O_t *oh, unsigned chunkno, size_t size, size_t *msg_idx);
static herr_t H5O__alloc_find_best_nonnull(const H5F_t *f, const H5O_t *oh, size_t *size,
                                           H5O_msg_alloc_info_t *found_msg);
static herr_t H5O__alloc_new_chunk(H5F_t *f, H5O_t *oh, size_t size, size_t *new_idx);
static herr_t H5O__alloc_find_best_null(const H5O_t *oh, size_t size, size_t *mesg_idx);
static htri_t H5O__move_cont(H5F_t *f, H5O_t *oh, unsigned cont_u);
static htri_t H5O__move_msgs_forward(H5F_t *f, H5O_t *oh);
static htri_t H5O__merge_null(H5F_t *f, H5O_t *oh);
static htri_t H5O__remove_empty_chunks(H5F_t *f, H5O_t *oh);
static herr_t H5O__alloc_shrink_chunk(H5F_t *f, H5O_t *oh, unsigned chunkno);

H5FL_EXTERN(H5O_cont_t);

static herr_t
H5O__add_gap(H5F_t H5_ATTR_NDEBUG_UNUSED *f, H5O_t *oh, unsigned chunkno, bool *chk_dirtied, size_t idx,
             uint8_t *new_gap_loc, size_t new_gap_size)
{
    bool   merged_with_null;    
    size_t u;                   
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(oh);
    assert(oh->version > H5O_VERSION_1);
    assert(chk_dirtied);
    assert(new_gap_loc);
    assert(new_gap_size);

#ifndef NDEBUG
    if (chunkno > 0) {
        unsigned chk_proxy_status = 0; 

        
        if (H5AC_get_entry_status(f, oh->chunk[chunkno].addr, &chk_proxy_status) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL,
                        "unable to check metadata cache status for object header chunk proxy");

        
        assert(chk_proxy_status & H5AC_ES__IS_PROTECTED);
    }  
#endif 

    
    merged_with_null = false;
    for (u = 0; u < oh->nmesgs && !merged_with_null; u++) {
        
        
        if (H5O_NULL_ID == oh->mesg[u].type->id && oh->mesg[u].chunkno == chunkno && u != idx) {
            
            assert(oh->chunk[chunkno].gap == 0);

            
            if (H5O__eliminate_gap(oh, chk_dirtied, &oh->mesg[u], new_gap_loc, new_gap_size) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, FAIL, "can't eliminate gap in chunk");

            
            merged_with_null = true;
        } 
    }     

    
    if (!merged_with_null) {
        
        for (u = 0; u < oh->nmesgs; u++)
            if (oh->mesg[u].chunkno == chunkno && oh->mesg[u].raw > new_gap_loc)
                oh->mesg[u].raw -= new_gap_size;

        
        memmove(new_gap_loc, new_gap_loc + new_gap_size,
                (size_t)((oh->chunk[chunkno].image + (oh->chunk[chunkno].size - H5O_SIZEOF_CHKSUM_OH(oh))) -
                         (new_gap_loc + new_gap_size)));

        
        new_gap_size += oh->chunk[chunkno].gap;

        
        if (new_gap_size >= (size_t)H5O_SIZEOF_MSGHDR_OH(oh)) {
            H5O_mesg_t *null_msg; 

            
            if (oh->nmesgs >= oh->alloc_nmesgs)
                if (H5O__alloc_msgs(oh, (size_t)1) < 0)
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "can't allocate more space for messages");

            
            oh->chunk[chunkno].gap += new_gap_size;

            
            null_msg           = &(oh->mesg[oh->nmesgs++]);
            null_msg->type     = H5O_MSG_NULL;
            null_msg->native   = NULL;
            null_msg->raw_size = new_gap_size - (size_t)H5O_SIZEOF_MSGHDR_OH(oh);
            null_msg->raw      = (oh->chunk[chunkno].image + oh->chunk[chunkno].size) -
                            (H5O_SIZEOF_CHKSUM_OH(oh) + null_msg->raw_size);
            null_msg->chunkno = chunkno;

            
            if (null_msg->raw_size)
                memset(null_msg->raw, 0, null_msg->raw_size);

            
            null_msg->dirty = true;

            
            oh->chunk[chunkno].gap = 0;
        } 
        else
            oh->chunk[chunkno].gap = new_gap_size;

        
        *chk_dirtied = true;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__eliminate_gap(H5O_t *oh, bool *chk_dirtied, H5O_mesg_t *mesg, uint8_t *gap_loc, size_t gap_size)
{
    uint8_t *move_start, *move_end; 
    bool     null_before_gap;       

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(oh);
    assert(oh->version > H5O_VERSION_1);
    assert(chk_dirtied);
    assert(mesg);
    assert(gap_loc);
    assert(gap_size);

    
    null_before_gap = (bool)(mesg->raw < gap_loc);

    
    if (null_before_gap) {
        move_start = mesg->raw + mesg->raw_size;
        move_end   = gap_loc;
    } 
    else {
        move_start = gap_loc + gap_size;
        move_end   = mesg->raw - H5O_SIZEOF_MSGHDR_OH(oh);
    } 

    
    if (move_end > move_start) {
        unsigned u; 

        
        
        for (u = 0; u < oh->nmesgs; u++) {
            uint8_t *msg_start; 

            msg_start = oh->mesg[u].raw - H5O_SIZEOF_MSGHDR_OH(oh);
            if (oh->mesg[u].chunkno == mesg->chunkno && (msg_start >= move_start && msg_start < move_end)) {
                
                if (null_before_gap)
                    oh->mesg[u].raw += gap_size;
                else
                    oh->mesg[u].raw -= gap_size;
            } 
        }     

        
        if (null_before_gap)
            
            memmove(move_start + gap_size, move_start, (size_t)(move_end - move_start));
        else {
            
            memmove(move_start - gap_size, move_start, (size_t)(move_end - move_start));

            
            mesg->raw -= gap_size;
        } 
    }
    else if (move_end == move_start && !null_before_gap) {
        
        memmove(move_start - gap_size, move_start, mesg->raw_size + (size_t)H5O_SIZEOF_MSGHDR_OH(oh));

        
        mesg->raw -= gap_size;
    } 

    
    memset(mesg->raw + mesg->raw_size, 0, gap_size);

    
    mesg->raw_size += gap_size;

    
    oh->chunk[mesg->chunkno].gap = 0;

    
    mesg->dirty  = true;
    *chk_dirtied = true;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__alloc_null(H5F_t *f, H5O_t *oh, size_t null_idx, const H5O_msg_class_t *new_type, void *new_native,
                size_t new_size)
{
    H5O_chunk_proxy_t *chk_proxy   = NULL;  
    bool               chk_dirtied = false; 
    H5O_mesg_t        *alloc_msg;           
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(oh);
    assert(new_type);
    assert(new_size);

    
    alloc_msg = &oh->mesg[null_idx];

    
    if (NULL == (chk_proxy = H5O__chunk_protect(f, oh, alloc_msg->chunkno)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to load object header chunk");

    
    if (alloc_msg->raw_size > new_size) {
        
        if ((alloc_msg->raw_size - new_size) < (size_t)H5O_SIZEOF_MSGHDR_OH(oh)) {
            size_t gap_size = alloc_msg->raw_size - new_size; 

            
            alloc_msg->raw_size = new_size;

            
            if (H5O__add_gap(f, oh, alloc_msg->chunkno, &chk_dirtied, null_idx,
                             alloc_msg->raw + alloc_msg->raw_size, gap_size) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, FAIL, "can't insert gap in chunk");
        } 
        else {
            size_t new_mesg_size =
                new_size + (size_t)H5O_SIZEOF_MSGHDR_OH(oh); 
            H5O_mesg_t *null_msg;                            

            
            if (oh->nmesgs >= oh->alloc_nmesgs) {
                if (H5O__alloc_msgs(oh, (size_t)1) < 0)
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "can't allocate more space for messages");

                
                alloc_msg = &oh->mesg[null_idx];
            } 

            
            null_msg           = &(oh->mesg[oh->nmesgs++]);
            null_msg->type     = H5O_MSG_NULL;
            null_msg->native   = NULL;
            null_msg->raw      = alloc_msg->raw + new_mesg_size;
            null_msg->raw_size = alloc_msg->raw_size - new_mesg_size;
            null_msg->chunkno  = alloc_msg->chunkno;

            
            null_msg->dirty = true;
            chk_dirtied     = true;

            
            if (oh->chunk[null_msg->chunkno].gap > 0) {
                unsigned null_chunkno = null_msg->chunkno; 

                
                if (H5O__eliminate_gap(oh, &chk_dirtied, null_msg,
                                       ((oh->chunk[null_chunkno].image + oh->chunk[null_chunkno].size) -
                                        (H5O_SIZEOF_CHKSUM_OH(oh) + oh->chunk[null_chunkno].gap)),
                                       oh->chunk[null_chunkno].gap) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTREMOVE, FAIL, "can't eliminate gap in chunk");
            } 

            
            alloc_msg->raw_size = new_size;
        } 
    }     

    
    alloc_msg->type   = new_type;
    alloc_msg->native = new_native;

    
    alloc_msg->dirty = true;
    chk_dirtied      = true;

done:
    
    if (chk_proxy && H5O__chunk_unprotect(f, chk_proxy, chk_dirtied) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to unprotect object header chunk");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__alloc_msgs(H5O_t *oh, size_t min_alloc)
{
    size_t      old_alloc;           
    size_t      na;                  
    H5O_mesg_t *new_mesg;            
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(oh);

    
    old_alloc = oh->alloc_nmesgs;
    na        = oh->alloc_nmesgs + MAX(oh->alloc_nmesgs, min_alloc); 

    
    if (NULL == (new_mesg = H5FL_SEQ_REALLOC(H5O_mesg_t, oh->mesg, na)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    oh->alloc_nmesgs = na;
    oh->mesg         = new_mesg;

    
    memset(&oh->mesg[old_alloc], 0, (oh->alloc_nmesgs - old_alloc) * sizeof(H5O_mesg_t));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5O__alloc_extend_chunk(H5F_t *f, H5O_t *oh, unsigned chunkno, size_t size, size_t *msg_idx)
{
    H5O_chunk_proxy_t *chk_proxy   = NULL;  
    bool               chk_dirtied = false; 
    size_t             delta;               
    size_t             aligned_size = H5O_ALIGN_OH(oh, size);
    uint8_t           *old_image;                 
    size_t             old_size;                  
    htri_t             was_extended;              
    size_t             extend_msg        = 0;     
    bool               extended_msg      = false; 
    uint8_t            new_size_flags    = 0;     
    bool               adjust_size_flags = false; 
    size_t             extra_prfx_size   = 0;     
    size_t             u;                         
    htri_t             ret_value = true;          

    FUNC_ENTER_PACKAGE

    
    assert(f != NULL);
    assert(oh != NULL);
    assert(chunkno < oh->nchunks);
    assert(size > 0);
    assert(msg_idx != NULL);
    assert(H5_addr_defined(oh->chunk[chunkno].addr));

    
    for (u = 0; u < oh->nmesgs; u++) {
        
        
        if (oh->mesg[u].chunkno == chunkno && H5O_NULL_ID == oh->mesg[u].type->id &&
            ((oh->mesg[u].raw + oh->mesg[u].raw_size) ==
             ((oh->chunk[chunkno].image + oh->chunk[chunkno].size) -
              (oh->chunk[chunkno].gap + H5O_SIZEOF_CHKSUM_OH(oh))))) {

            extend_msg   = u;
            extended_msg = true;
            break;
        } 
    }     

    
    if (extended_msg) {
        assert(oh->chunk[chunkno].gap == 0);
        delta = aligned_size - oh->mesg[extend_msg].raw_size;
    } 
    else
        delta = (aligned_size + (size_t)H5O_SIZEOF_MSGHDR_OH(oh)) - oh->chunk[chunkno].gap;
    delta = H5O_ALIGN_OH(oh, delta);

    
    if (oh->version > H5O_VERSION_1 && chunkno == 0) {
        uint64_t chunk0_size;                                                     
        size_t   orig_prfx_size = (size_t)1 << (oh->flags & H5O_HDR_CHUNK0_SIZE); 

        assert(oh->chunk[0].size >= (size_t)H5O_SIZEOF_HDR(oh));
        chunk0_size = oh->chunk[0].size - (size_t)H5O_SIZEOF_HDR(oh);

        
        if (orig_prfx_size < 8 && (chunk0_size + delta) > 4294967295) {
            extra_prfx_size   = 8 - orig_prfx_size;
            new_size_flags    = H5O_HDR_CHUNK0_8;
            adjust_size_flags = true;
        } 
        
        else if (orig_prfx_size < 4 && (chunk0_size + delta) > 65535) {
            extra_prfx_size   = 4 - orig_prfx_size;
            new_size_flags    = H5O_HDR_CHUNK0_4;
            adjust_size_flags = true;
        } 
        
        else if (orig_prfx_size < 2 && (chunk0_size + delta) > 255) {
            extra_prfx_size   = 2 - orig_prfx_size;
            new_size_flags    = H5O_HDR_CHUNK0_2;
            adjust_size_flags = true;
        } 
    }     

    
    if (NULL == (chk_proxy = H5O__chunk_protect(f, oh, chunkno)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to load object header chunk");

    
    was_extended = H5MF_try_extend(f, H5FD_MEM_OHDR, oh->chunk[chunkno].addr,
                                   (hsize_t)(oh->chunk[chunkno].size), (hsize_t)(delta + extra_prfx_size));
    if (was_extended < 0) 
        HGOTO_ERROR(H5E_OHDR, H5E_CANTEXTEND, FAIL, "can't tell if we can extend chunk");
    else if (was_extended == false) 
        HGOTO_DONE(false);

    
    if (adjust_size_flags) {
        oh->flags = (uint8_t)(oh->flags & ~H5O_HDR_CHUNK0_SIZE);
        oh->flags |= new_size_flags;

        
        if (H5AC_mark_entry_dirty(oh) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTMARKDIRTY, FAIL, "unable to mark object header as dirty");
    } 

    
    if (extended_msg) {
        
        oh->mesg[extend_msg].raw_size += delta;
    } 
    
    else {
        
        if (oh->nmesgs >= oh->alloc_nmesgs)
            if (H5O__alloc_msgs(oh, (size_t)1) < 0)
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "can't allocate more space for messages");

        
        extend_msg = oh->nmesgs++;

        
        oh->mesg[extend_msg].type   = H5O_MSG_NULL;
        oh->mesg[extend_msg].native = NULL;
        oh->mesg[extend_msg].raw    = ((oh->chunk[chunkno].image + oh->chunk[chunkno].size) -
                                    (H5O_SIZEOF_CHKSUM_OH(oh) + oh->chunk[chunkno].gap)) +
                                   H5O_SIZEOF_MSGHDR_OH(oh);
        oh->mesg[extend_msg].raw_size = (delta + oh->chunk[chunkno].gap) - (size_t)H5O_SIZEOF_MSGHDR_OH(oh);
        oh->mesg[extend_msg].chunkno  = chunkno;
    } 

    
    oh->mesg[extend_msg].dirty = true;
    chk_dirtied                = true;

    
    old_image = oh->chunk[chunkno].image;
    old_size  = oh->chunk[chunkno].size;
    oh->chunk[chunkno].size += delta + extra_prfx_size;
    oh->chunk[chunkno].image = H5FL_BLK_REALLOC(chunk_image, old_image, oh->chunk[chunkno].size);
    if (NULL == oh->chunk[chunkno].image)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTALLOC, FAIL, "can't reallocate extended object header chunk");
    oh->chunk[chunkno].gap = 0;

    
    memset(oh->chunk[chunkno].image + old_size, 0, oh->chunk[chunkno].size - old_size);

    
    if (adjust_size_flags)
        memmove(oh->chunk[0].image + H5O_SIZEOF_HDR(oh) - H5O_SIZEOF_CHKSUM_OH(oh),
                oh->chunk[0].image + H5O_SIZEOF_HDR(oh) - H5O_SIZEOF_CHKSUM_OH(oh) - extra_prfx_size,
                old_size - (size_t)H5O_SIZEOF_HDR(oh) + extra_prfx_size);

    
    for (u = 0; u < oh->nmesgs; u++) {
        
        if (oh->mesg[u].chunkno == chunkno)
            oh->mesg[u].raw = oh->chunk[chunkno].image + extra_prfx_size + (oh->mesg[u].raw - old_image);

        
        
        if (chunkno > 0 && (H5O_CONT_ID == oh->mesg[u].type->id) &&
            (((H5O_cont_t *)(oh->mesg[u].native))->chunkno == chunkno)) {
            H5O_chunk_proxy_t *chk_proxy2   = NULL;                
            bool               chk_dirtied2 = false;               
            unsigned           cont_chunkno = oh->mesg[u].chunkno; 

            
            if (NULL == (chk_proxy2 = H5O__chunk_protect(f, oh, cont_chunkno)))
                HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to load object header chunk");

            
            assert(((H5O_cont_t *)(oh->mesg[u].native))->size == old_size);
            ((H5O_cont_t *)(oh->mesg[u].native))->size = oh->chunk[chunkno].size;

            
            oh->mesg[u].dirty = true;
            chk_dirtied2      = true;

            
            if (H5O__chunk_unprotect(f, chk_proxy2, chk_dirtied2) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to unprotect object header chunk");
        } 
    }     

    
    if (H5O__chunk_resize(oh, chk_proxy) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTRESIZE, FAIL, "unable to resize object header chunk");

    
    *msg_idx = extend_msg;

done:
    
    if (chk_proxy && H5O__chunk_unprotect(f, chk_proxy, chk_dirtied) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to unprotect object header chunk");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__alloc_find_best_nonnull(const H5F_t *f, const H5O_t *oh, size_t *size, H5O_msg_alloc_info_t *found_msg)
{
    H5O_mesg_t *curr_msg;   
    size_t      cont_size;  
    size_t      multi_size; 
    unsigned    u;          

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(oh);
    assert(size);
    assert(*size > 0);
    assert(found_msg);

    
    cont_size  = H5O_ALIGN_OH(oh, (size_t)(H5F_SIZEOF_ADDR(f) + H5F_SIZEOF_SIZE(f)));
    multi_size = 0;
    for (u = 0, curr_msg = &oh->mesg[0]; u < oh->nmesgs; u++, curr_msg++) {
        
        if (H5O_CONT_ID != curr_msg->type->id) {
            unsigned msg_chunkno = curr_msg->chunkno; 
            uint8_t *end_chunk_data =
                (oh->chunk[msg_chunkno].image + oh->chunk[msg_chunkno].size) -
                (H5O_SIZEOF_CHKSUM_OH(oh) + oh->chunk[msg_chunkno].gap); 
            uint8_t *end_msg    = curr_msg->raw + curr_msg->raw_size;    
            size_t   gap_size   = 0; 
            size_t   null_size  = 0; 
            unsigned null_msgno = 0; 
            size_t   total_size;     

            
            if (end_msg == end_chunk_data)
                gap_size = oh->chunk[msg_chunkno].gap;
            else {
                H5O_mesg_t *tmp_msg; 
                unsigned    v;       

                
                for (v = 0, tmp_msg = &oh->mesg[0]; v < oh->nmesgs; v++, tmp_msg++) {
                    if (tmp_msg->type->id == H5O_NULL_ID &&
                        (tmp_msg->raw - H5O_SIZEOF_MSGHDR_OH(oh)) == end_msg) {
                        null_msgno = v;
                        null_size  = (size_t)H5O_SIZEOF_MSGHDR_OH(oh) + tmp_msg->raw_size;
                        break;
                    } 

                    

                } 
            }     

            
            total_size = curr_msg->raw_size + gap_size + null_size;

            
            if (total_size >= cont_size) {
                bool better = false; 

                
                if (found_msg->msgno < 0)
                    better = true;
                else {
                    
                    
                    if (found_msg->id == H5O_ATTR_ID && curr_msg->type->id != H5O_ATTR_ID)
                        better = true;
                    
                    else {
                        
                        if (total_size < found_msg->total_size)
                            better = true;
                        
                        
                        else if (total_size == found_msg->total_size) {
                            if (msg_chunkno < found_msg->chunkno)
                                better = true;
                        } 
                    }     
                }         

                
                if (better) {
                    found_msg->msgno      = (int)u;
                    found_msg->id         = curr_msg->type->id;
                    found_msg->chunkno    = msg_chunkno;
                    found_msg->gap_size   = gap_size;
                    found_msg->null_size  = null_size;
                    found_msg->total_size = total_size;
                    found_msg->null_msgno = null_msgno;
                } 
            }     
            else if (found_msg->msgno < 0 && msg_chunkno == oh->nchunks - 1)
                
                multi_size += curr_msg->raw_size + (size_t)H5O_SIZEOF_MSGHDR_OH(oh);
        } 
    }     

    
    if (found_msg->msgno < 0)
        *size += multi_size;
    else
        *size += (size_t)H5O_SIZEOF_MSGHDR_OH(oh) + oh->mesg[found_msg->msgno].raw_size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5O__alloc_chunk(H5F_t *f, H5O_t *oh, size_t size, size_t found_null, const H5O_msg_alloc_info_t *found_msg,
                 size_t *new_idx)
{
    H5O_mesg_t        *curr_msg;            
    H5O_chunk_proxy_t *chk_proxy;           
    size_t             cont_size;           
    size_t             idx;                 
    uint8_t           *p    = NULL;         
    H5O_cont_t        *cont = NULL;         
    unsigned           chunkno;             
    haddr_t            new_chunk_addr;      
    unsigned           u;                   
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(oh);
    assert(found_msg);
    assert(new_idx);

    
    size = MAX(H5O_MIN_SIZE, size + (size_t)H5O_SIZEOF_MSGHDR_OH(oh));
    assert(size == H5O_ALIGN_OH(oh, size));

    
    size += H5O_SIZEOF_CHKHDR_OH(oh);

    
    new_chunk_addr = H5MF_alloc(f, H5FD_MEM_OHDR, (hsize_t)size);
    if (!H5_addr_defined(new_chunk_addr))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTALLOC, FAIL, "unable to allocate space for new chunk");

    
    if (oh->nchunks >= oh->alloc_nchunks) {
        size_t       na = MAX(H5O_NCHUNKS, oh->alloc_nchunks * 2); 
        H5O_chunk_t *x;

        if (NULL == (x = H5FL_SEQ_REALLOC(H5O_chunk_t, oh->chunk, na)))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTALLOC, FAIL, "can't allocate larger chunk array, na = %llu", (unsigned long long)na);
        oh->alloc_nchunks = na;
        oh->chunk         = x;
    } 

    H5_CHECKED_ASSIGN(chunkno, unsigned, oh->nchunks, size_t);
    oh->nchunks++;
    oh->chunk[chunkno].addr = new_chunk_addr;
    oh->chunk[chunkno].size = size;
    oh->chunk[chunkno].gap  = 0;
    if (NULL == (oh->chunk[chunkno].image = p = H5FL_BLK_CALLOC(chunk_image, size)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTALLOC, FAIL, "can't allocate image for chunk, size = %llu", (unsigned long long)size);
    oh->chunk[chunkno].chunk_proxy = NULL;

    
    if (oh->version > H5O_VERSION_1) {
        H5MM_memcpy(p, H5O_CHK_MAGIC, (size_t)H5_SIZEOF_MAGIC);
        p += H5_SIZEOF_MAGIC;
    } 

    
    if (oh->nmesgs + 3 > oh->alloc_nmesgs)
        if (H5O__alloc_msgs(oh, (size_t)3) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "can't allocate more space for messages");

    
    cont_size = H5O_ALIGN_OH(oh, (size_t)(H5F_SIZEOF_ADDR(f) + H5F_SIZEOF_SIZE(f)));
    if (found_null >= oh->nmesgs) {
        if (found_msg->msgno < 0) {
            
            H5O_mesg_t *null_msg; 

            
            if (NULL == (chk_proxy = H5O__chunk_protect(f, oh, chunkno - 1)))
                HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to load object header chunk");

            
            for (u = 0, curr_msg = &oh->mesg[0]; u < oh->nmesgs; u++, curr_msg++)
                if (curr_msg->chunkno == chunkno - 1) {
                    if (curr_msg->type->id == H5O_NULL_ID) {
                        
                        curr_msg->type = H5O_MSG_DELETED;
                        oh->num_deleted_mesgs++;
                    }
                    else {
                        assert(curr_msg->type->id != H5O_CONT_ID);

                        if (size < curr_msg->raw_size + (size_t)H5O_SIZEOF_MSGHDR_OH(oh))
                            HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "invalid size");

                        
                        H5MM_memcpy(p, curr_msg->raw - (size_t)H5O_SIZEOF_MSGHDR_OH(oh),
                                    curr_msg->raw_size + (size_t)H5O_SIZEOF_MSGHDR_OH(oh));

                        
                        curr_msg->chunkno = chunkno;
                        curr_msg->raw     = p + H5O_SIZEOF_MSGHDR_OH(oh);

                        
                        p += (size_t)H5O_SIZEOF_MSGHDR_OH(oh) + curr_msg->raw_size;
                        size -= (size_t)H5O_SIZEOF_MSGHDR_OH(oh) + curr_msg->raw_size;
                    } 
                }     

            
            found_null       = oh->nmesgs++;
            null_msg         = &(oh->mesg[found_null]);
            null_msg->type   = H5O_MSG_NULL;
            null_msg->dirty  = true;
            null_msg->native = NULL;
            null_msg->raw    = oh->chunk[chunkno - 1].image +
                            ((chunkno == 1) ? H5O_SIZEOF_HDR(oh) : H5O_SIZEOF_CHKHDR_OH(oh)) -
                            H5O_SIZEOF_CHKSUM_OH(oh) + H5O_SIZEOF_MSGHDR_OH(oh);
            null_msg->raw_size =
                oh->chunk[chunkno - 1].size -
                ((chunkno == 1) ? (size_t)H5O_SIZEOF_HDR(oh) : (size_t)H5O_SIZEOF_CHKHDR_OH(oh)) -
                (size_t)H5O_SIZEOF_MSGHDR_OH(oh);
            null_msg->chunkno = chunkno - 1;

            assert(null_msg->raw_size >= cont_size);

            
            oh->chunk[chunkno - 1].gap = 0;

            
            if (H5O__chunk_unprotect(f, chk_proxy, true) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to unprotect object header chunk");
        } 
        else {
            
            H5O_mesg_t *null_msg; 

            
            if (NULL == (chk_proxy = H5O__chunk_protect(f, oh, oh->mesg[found_msg->msgno].chunkno)))
                HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to load object header chunk");

            
            found_null         = oh->nmesgs++;
            null_msg           = &(oh->mesg[found_null]);
            null_msg->type     = H5O_MSG_NULL;
            null_msg->native   = NULL;
            null_msg->raw      = oh->mesg[found_msg->msgno].raw;
            null_msg->raw_size = oh->mesg[found_msg->msgno].raw_size;
            null_msg->chunkno  = oh->mesg[found_msg->msgno].chunkno;

            
            H5MM_memcpy(p, oh->mesg[found_msg->msgno].raw - H5O_SIZEOF_MSGHDR_OH(oh),
                        oh->mesg[found_msg->msgno].raw_size + (size_t)H5O_SIZEOF_MSGHDR_OH(oh));

            
            oh->mesg[found_msg->msgno].raw     = p + H5O_SIZEOF_MSGHDR_OH(oh);
            oh->mesg[found_msg->msgno].chunkno = chunkno;

            
            p += (size_t)H5O_SIZEOF_MSGHDR_OH(oh) + oh->mesg[found_msg->msgno].raw_size;
            size -= (size_t)H5O_SIZEOF_MSGHDR_OH(oh) + oh->mesg[found_msg->msgno].raw_size;

            
            if (found_msg->gap_size > 0) {
                
                assert(oh->chunk[null_msg->chunkno].gap == found_msg->gap_size);
                null_msg->raw_size += found_msg->gap_size;
                oh->chunk[null_msg->chunkno].gap = 0;
            } 
            else if (found_msg->null_size > 0) {
                H5O_mesg_t *old_null_msg =
                    &oh->mesg[found_msg->null_msgno]; 

                
                assert((null_msg->raw + null_msg->raw_size) ==
                       (old_null_msg->raw - H5O_SIZEOF_MSGHDR_OH(oh)));
                null_msg->raw_size += found_msg->null_size;

                
                H5O__msg_free_mesg(old_null_msg);

                
                old_null_msg->type = H5O_MSG_DELETED;
                oh->num_deleted_mesgs++;
            } 

            
            null_msg->dirty = true;

            
            if (H5O__chunk_unprotect(f, chk_proxy, true) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to unprotect object header chunk");
        } 
    }     

    
    
    idx                    = oh->nmesgs++;
    oh->mesg[idx].type     = H5O_MSG_NULL;
    oh->mesg[idx].dirty    = true;
    oh->mesg[idx].native   = NULL;
    oh->mesg[idx].raw      = p + H5O_SIZEOF_MSGHDR_OH(oh);
    oh->mesg[idx].raw_size = size - (size_t)(H5O_SIZEOF_CHKHDR_OH(oh) + H5O_SIZEOF_MSGHDR_OH(oh));
    oh->mesg[idx].chunkno  = chunkno;

    
    if (H5O__chunk_add(f, oh, chunkno, oh->mesg[found_null].chunkno) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, FAIL, "can't add new chunk to cache");

    
    if (NULL == (cont = H5FL_MALLOC(H5O_cont_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");
    cont->addr    = oh->chunk[chunkno].addr;
    cont->size    = oh->chunk[chunkno].size;
    cont->chunkno = chunkno;

    
    if (H5O__alloc_null(f, oh, found_null, H5O_MSG_CONT, cont, cont_size) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, FAIL, "can't split null message");

    
    *new_idx = idx;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__alloc_new_chunk(H5F_t *f, H5O_t *oh, size_t size, size_t *new_idx)
{
    size_t               cont_size;           
    size_t               idx;                 
    H5O_msg_alloc_info_t found_msg;           
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(oh);
    assert(size > 0);
    size = H5O_ALIGN_OH(oh, size);

    
    idx       = oh->nmesgs;
    cont_size = H5O_ALIGN_OH(oh, (size_t)(H5F_SIZEOF_ADDR(f) + H5F_SIZEOF_SIZE(f)));
    if (H5O__alloc_find_best_null(oh, cont_size, &idx) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_NOTFOUND, FAIL, "error while locating best null header message");

    
    if (idx >= oh->nmesgs) {
        found_msg.msgno = -1;
        if (H5O__alloc_find_best_nonnull(f, oh, &size, &found_msg) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_NOTFOUND, FAIL, "error while locating best non-null header message");
    } 

    
    if (H5O__alloc_chunk(f, oh, size, idx, &found_msg, new_idx) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTALLOC, FAIL, "can't allocate new object header chunk");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__alloc_find_best_null(const H5O_t *oh, size_t size, size_t *mesg_idx)
{
    size_t  idx;        
    ssize_t found_null; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(oh);
    assert(size > 0);
    assert(mesg_idx);

    
    found_null = -1;
    for (idx = 0; idx < oh->nmesgs; idx++) {
        if (H5O_NULL_ID == oh->mesg[idx].type->id) {
            
            if (oh->mesg[idx].raw_size == size) {
                
                if (found_null < 0)
                    found_null = (ssize_t)idx;
                else
                    
                    if (oh->mesg[idx].chunkno < oh->mesg[found_null].chunkno) {
                        found_null = (ssize_t)idx;

                        
                        
                        if (0 == oh->mesg[idx].chunkno)
                            break;
                    } 
            }         
            
            else if (oh->mesg[idx].raw_size > size) {
                
                if (found_null < 0)
                    found_null = (ssize_t)idx;
                
                else if (oh->mesg[idx].raw_size < oh->mesg[found_null].raw_size) {
                    found_null = (ssize_t)idx;
                }
                else {
                    
                    if (oh->mesg[idx].raw_size == oh->mesg[found_null].raw_size) {
                        if (oh->mesg[idx].chunkno < oh->mesg[found_null].chunkno)
                            found_null = (ssize_t)idx;
                    } 
                }     
            }         
            
        } 
    }     
    if (found_null >= 0)
        *mesg_idx = (size_t)found_null;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5O__alloc(H5F_t *f, H5O_t *oh, const H5O_msg_class_t *type, const void *mesg, size_t *mesg_idx)
{
    size_t raw_size;            
    size_t aligned_size;        
    size_t idx;                 
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(oh);
    assert(type);
    assert(mesg);
    assert(mesg_idx);

    
    raw_size = (type->raw_size)(f, false, mesg);
    if (0 == raw_size)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL, "can't compute object header message size");
    if (raw_size >= H5O_MESG_MAX_SIZE)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL, "object header message is too large");
    aligned_size = H5O_ALIGN_OH(oh, raw_size);

    
    idx = oh->nmesgs;
    if (H5O__alloc_find_best_null(oh, aligned_size, &idx) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_NOTFOUND, FAIL, "error while locating best null header message");

    
    if (idx >= oh->nmesgs) {
        unsigned chunkno;

        
        for (chunkno = 0; chunkno < oh->nchunks; chunkno++) {
            htri_t tri_result; 

            if ((tri_result = H5O__alloc_extend_chunk(f, oh, chunkno, raw_size, &idx)) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTEXTEND, FAIL, "can't extend existing chunk");
            if (tri_result == true)
                break;
        } 

        
        if (idx >= oh->nmesgs)
            if (H5O__alloc_new_chunk(f, oh, raw_size, &idx) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_NOSPACE, FAIL, "unable to create a new object header data chunk");
    } 
    assert(idx < oh->nmesgs);

    
    if (H5O__alloc_null(f, oh, idx, type, NULL, aligned_size) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, FAIL, "can't split null message");

    
    if (H5AC_mark_entry_dirty(oh) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTMARKDIRTY, FAIL, "unable to mark object header as dirty");

    
    *mesg_idx = idx;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__release_mesg(H5F_t *f, H5O_t *oh, H5O_mesg_t *mesg, bool adj_link)
{
    H5O_chunk_proxy_t *chk_proxy   = NULL;    
    bool               chk_dirtied = false;   
    herr_t             ret_value   = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(oh);
    assert(mesg);

    
    if (adj_link)
        
        if (H5O__delete_mesg(f, oh, mesg) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTDELETE, FAIL,
                        "unable to delete file space for object header message");

    
    if (NULL == (chk_proxy = H5O__chunk_protect(f, oh, mesg->chunkno)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to protect object header chunk");

    
    H5O__msg_free_mesg(mesg);

    
    mesg->type = H5O_MSG_NULL;
    assert(mesg->raw + mesg->raw_size <= (oh->chunk[mesg->chunkno].image + oh->chunk[mesg->chunkno].size) -
                                             (H5O_SIZEOF_CHKSUM_OH(oh) + oh->chunk[mesg->chunkno].gap));
    memset(mesg->raw, 0, mesg->raw_size);

    
    mesg->flags = 0;

    
    mesg->dirty = true;
    chk_dirtied = true;

    
    if (oh->chunk[mesg->chunkno].gap) {
        
        if (H5O__eliminate_gap(oh, &chk_dirtied, mesg,
                               ((oh->chunk[mesg->chunkno].image + oh->chunk[mesg->chunkno].size) -
                                (H5O_SIZEOF_CHKSUM_OH(oh) + oh->chunk[mesg->chunkno].gap)),
                               oh->chunk[mesg->chunkno].gap) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTREMOVE, FAIL, "can't eliminate gap in chunk");
    } 

done:
    
    if (chk_proxy && H5O__chunk_unprotect(f, chk_proxy, chk_dirtied) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to unprotect object header chunk");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5O__move_cont(H5F_t *f, H5O_t *oh, unsigned cont_u)
{
    H5O_chunk_proxy_t *chk_proxy = NULL;    
    H5O_mesg_t        *cont_msg;            
    unsigned           deleted_chunkno;     
    bool               chk_dirtied = false; 
    htri_t             ret_value   = true;  

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(oh);

    
    cont_msg = &oh->mesg[cont_u];
    H5O_LOAD_NATIVE(f, 0, oh, cont_msg, FAIL)
    deleted_chunkno = ((H5O_cont_t *)(cont_msg->native))->chunkno;

    
    if (deleted_chunkno == (oh->nchunks - 1)) {
        size_t      nonnull_size; 
        H5O_mesg_t *curr_msg;     
        size_t      gap_size;     
        size_t      v;            

        
        nonnull_size = 0;
        for (v = 0, curr_msg = &oh->mesg[0]; v < oh->nmesgs; v++, curr_msg++) {
            if (curr_msg->chunkno == deleted_chunkno) {
                
                if (curr_msg->type->id != H5O_NULL_ID) {
                    assert(curr_msg->type->id != H5O_CONT_ID);
                    nonnull_size += curr_msg->raw_size + (size_t)H5O_SIZEOF_MSGHDR_OH(oh);
                } 
            }     
        }         

        
        gap_size = oh->chunk[cont_msg->chunkno].gap;

        
        
        if (nonnull_size &&
            nonnull_size <= (gap_size + cont_msg->raw_size + (size_t)H5O_SIZEOF_MSGHDR_OH(oh))) {
            uint8_t *move_start, *move_end; 
            unsigned cont_chunkno;          

            
            move_start   = cont_msg->raw - H5O_SIZEOF_MSGHDR_OH(oh);
            move_end     = cont_msg->raw + cont_msg->raw_size;
            cont_chunkno = cont_msg->chunkno;

            
            if (H5O__release_mesg(f, oh, cont_msg, false) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTDELETE, FAIL, "unable to convert into null message");

            
            if (NULL == (chk_proxy = H5O__chunk_protect(f, oh, cont_chunkno)))
                HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to protect object header chunk");

            
            for (v = 0, curr_msg = &oh->mesg[0]; v < oh->nmesgs; v++, curr_msg++)
                
                if (curr_msg->chunkno == deleted_chunkno) {
                    
                    if (curr_msg->type->id != H5O_NULL_ID) {
                        size_t move_size; 

                        
                        move_size = curr_msg->raw_size + (size_t)H5O_SIZEOF_MSGHDR_OH(oh);

                        
                        H5MM_memcpy(move_start, curr_msg->raw - H5O_SIZEOF_MSGHDR_OH(oh), move_size);
                        curr_msg->raw     = move_start + H5O_SIZEOF_MSGHDR_OH(oh);
                        curr_msg->chunkno = cont_chunkno;
                        chk_dirtied       = true;

                        
                        move_start += move_size;
                    } 
                }     

            
            if (H5O__chunk_delete(f, oh, deleted_chunkno) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTDELETE, FAIL, "unable to remove chunk from cache");

            assert(move_start <= (move_end + gap_size));

            
            
            gap_size += (size_t)(move_end - move_start);
            if (gap_size >= (size_t)H5O_SIZEOF_MSGHDR_OH(oh)) {
                
                cont_msg->raw_size = gap_size - (size_t)H5O_SIZEOF_MSGHDR_OH(oh);
                cont_msg->raw      = move_start + H5O_SIZEOF_MSGHDR_OH(oh);
                cont_msg->dirty    = true;
                chk_dirtied        = true;
            } 
            else {
                
                if (gap_size > 0) {
                    
                    if (H5O__add_gap(f, oh, cont_chunkno, &chk_dirtied, cont_u, move_start, gap_size) < 0)
                        HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, FAIL, "can't insert gap in chunk");
                } 

                
                H5O__msg_free_mesg(cont_msg);
                if (cont_u < (oh->nmesgs - 1))
                    memmove(&oh->mesg[cont_u], &oh->mesg[cont_u + 1],
                            ((oh->nmesgs - 1) - cont_u) * sizeof(H5O_mesg_t));
                oh->nmesgs--;
            } 

            
            
            for (v = oh->nmesgs - 1, curr_msg = &oh->mesg[v]; v < oh->nmesgs; v--, curr_msg--)
                
                if (curr_msg->chunkno == deleted_chunkno) {
                    
                    if (curr_msg->type->id == H5O_NULL_ID) {
                        
                        H5O__msg_free_mesg(curr_msg);
                        chk_dirtied = true;

                        
                        if (v < (oh->nmesgs - 1))
                            memmove(&oh->mesg[v], &oh->mesg[v + 1],
                                    ((oh->nmesgs - 1) - v) * sizeof(H5O_mesg_t));
                        oh->nmesgs--;
                    } 
                }     

            
            oh->chunk[deleted_chunkno].image = H5FL_BLK_FREE(chunk_image, oh->chunk[deleted_chunkno].image);
            oh->nchunks--;
        } 
        else
            ret_value = false;
    } 
    else
        ret_value = false;

done:
    
    if (chk_proxy && H5O__chunk_unprotect(f, chk_proxy, chk_dirtied) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to unprotect object header chunk");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5O__move_msgs_forward(H5F_t *f, H5O_t *oh)
{
    H5O_chunk_proxy_t *null_chk_proxy      = NULL;  
    H5O_chunk_proxy_t *curr_chk_proxy      = NULL;  
    H5O_chunk_proxy_t *cont_targ_chk_proxy = NULL;  
    bool               null_chk_dirtied    = false; 
    bool               curr_chk_dirtied    = false; 
    bool               packed_msg;                  
    bool               did_packing = false;         
    htri_t             ret_value   = FAIL;          

    FUNC_ENTER_PACKAGE

    
    assert(oh);

    
    
    do {
        H5O_mesg_t *curr_msg; 
        unsigned    u;        

        
        packed_msg = false;

        
        for (u = 0, curr_msg = &oh->mesg[0]; u < oh->nmesgs; u++, curr_msg++) {
            if (H5O_NULL_ID == curr_msg->type->id) {
                H5O_chunk_t *chunk; 

                
                chunk = &(oh->chunk[curr_msg->chunkno]);
                if ((curr_msg->raw + curr_msg->raw_size) !=
                    ((chunk->image + chunk->size) - (H5O_SIZEOF_CHKSUM_OH(oh) + chunk->gap))) {
                    H5O_mesg_t *nonnull_msg; 
                    unsigned    v;           

                    
                    for (v = 0, nonnull_msg = &oh->mesg[0]; v < oh->nmesgs; v++, nonnull_msg++) {
                        
                        if ((curr_msg->chunkno == nonnull_msg->chunkno) &&
                            ((curr_msg->raw + curr_msg->raw_size) ==
                             (nonnull_msg->raw - H5O_SIZEOF_MSGHDR_OH(oh)))) {
                            
                            
                            if (H5O_NULL_ID != nonnull_msg->type->id) {
                                
                                if (NULL == (null_chk_proxy = H5O__chunk_protect(f, oh, curr_msg->chunkno)))
                                    HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL,
                                                "unable to load object header chunk");

                                
                                memmove(curr_msg->raw - H5O_SIZEOF_MSGHDR_OH(oh),
                                        nonnull_msg->raw - H5O_SIZEOF_MSGHDR_OH(oh),
                                        nonnull_msg->raw_size + (size_t)H5O_SIZEOF_MSGHDR_OH(oh));

                                
                                nonnull_msg->raw = curr_msg->raw;

                                
                                curr_msg->raw =
                                    nonnull_msg->raw + nonnull_msg->raw_size + H5O_SIZEOF_MSGHDR_OH(oh);

                                
                                
                                curr_msg->dirty = true;

                                
                                if (H5O__chunk_unprotect(f, null_chk_proxy, true) < 0)
                                    HGOTO_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL,
                                                "unable to unprotect object header chunk");
                                null_chk_proxy = NULL;

                                
                                packed_msg = true;
                            } 

                            
                            break;
                        } 
                    }     
                    
                    assert(v < oh->nmesgs);
                } 
            }     
            else {
                H5O_mesg_t *null_msg; 
                size_t      v;        

                
                if (H5O_CONT_ID == curr_msg->type->id) {
                    htri_t status; 

                    if ((status = H5O__move_cont(f, oh, u)) < 0)
                        HGOTO_ERROR(H5E_OHDR, H5E_CANTDELETE, FAIL,
                                    "Error in moving messages into cont message");
                    else if (status > 0) { 
                        packed_msg = true;
                        break;
                    } 
                }     

                
                for (v = 0, null_msg = &oh->mesg[0]; v < oh->nmesgs; v++, null_msg++) {
                    if (H5O_NULL_ID == null_msg->type->id && curr_msg->chunkno > null_msg->chunkno &&
                        curr_msg->raw_size <= null_msg->raw_size) {
                        unsigned old_chunkno; 
                        uint8_t *old_raw;

                        
                        old_chunkno = curr_msg->chunkno;
                        old_raw     = curr_msg->raw;

                        
                        if (NULL == (null_chk_proxy = H5O__chunk_protect(f, oh, null_msg->chunkno)))
                            HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL,
                                        "unable to load object header chunk");
                        if (NULL == (curr_chk_proxy = H5O__chunk_protect(f, oh, curr_msg->chunkno)))
                            HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL,
                                        "unable to load object header chunk");

                        
                        if (oh->swmr_write && (H5O_CONT_ID == curr_msg->type->id)) {
                            void *null_chk_mdc_obj; 

                            
                            null_chk_mdc_obj = (null_msg->chunkno == 0 ? (void *)oh : (void *)null_chk_proxy);

                            
                            assert(curr_msg->chunkno > 0);
                            assert(((H5O_cont_t *)(curr_msg->native))->chunkno > 0);

                            
                            if (NULL == (cont_targ_chk_proxy = H5O__chunk_protect(
                                             f, oh, ((H5O_cont_t *)(curr_msg->native))->chunkno)))
                                HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL,
                                            "unable to load object header chunk");

                            
                            
                            if (cont_targ_chk_proxy->fd_parent) {
                                
                                assert(cont_targ_chk_proxy);
                                assert(curr_chk_proxy);
                                assert((void *)curr_chk_proxy == cont_targ_chk_proxy->fd_parent);

                                if (H5AC_destroy_flush_dependency(curr_chk_proxy, cont_targ_chk_proxy) < 0)
                                    HGOTO_ERROR(H5E_OHDR, H5E_CANTUNDEPEND, FAIL,
                                                "unable to destroy flush dependency");

                                cont_targ_chk_proxy->fd_parent = NULL;
                            } 

                            
                            if (0 != null_msg->chunkno) {
                                
                                assert(null_chk_mdc_obj);
                                assert(((H5C_cache_entry_t *)null_chk_mdc_obj)->type);
                                assert(((H5C_cache_entry_t *)null_chk_mdc_obj)->type->id == H5AC_OHDR_CHK_ID);

                                
                                if (H5AC_create_flush_dependency(null_chk_mdc_obj, cont_targ_chk_proxy) < 0)
                                    HGOTO_ERROR(H5E_OHDR, H5E_CANTDEPEND, FAIL,
                                                "unable to create flush dependency");

                                cont_targ_chk_proxy->fd_parent = null_chk_mdc_obj;
                            } 

                            
                            if (H5O__chunk_unprotect(f, cont_targ_chk_proxy, false) < 0)
                                HGOTO_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL,
                                            "unable to unprotect object header chunk");
                            cont_targ_chk_proxy = NULL;
                        } 

                        
                        H5MM_memcpy(null_msg->raw - H5O_SIZEOF_MSGHDR_OH(oh),
                                    curr_msg->raw - H5O_SIZEOF_MSGHDR_OH(oh),
                                    curr_msg->raw_size + (size_t)H5O_SIZEOF_MSGHDR_OH(oh));

                        
                        curr_msg->chunkno = null_msg->chunkno;
                        curr_msg->raw     = null_msg->raw;
                        curr_chk_dirtied  = true;

                        
                        if (curr_msg->raw_size == null_msg->raw_size) {
                            
                            
                            null_msg->chunkno = old_chunkno;
                            null_msg->raw     = old_raw;

                            
                            null_msg->dirty  = true;
                            null_chk_dirtied = true;

                            
                            if (H5O__chunk_unprotect(f, curr_chk_proxy, curr_chk_dirtied) < 0)
                                HGOTO_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL,
                                            "unable to unprotect object header chunk");
                            curr_chk_proxy   = NULL;
                            curr_chk_dirtied = false;

                            
                            if (oh->chunk[old_chunkno].gap > 0) {
                                
                                if (H5O__eliminate_gap(
                                        oh, &null_chk_dirtied, null_msg,
                                        ((oh->chunk[old_chunkno].image + oh->chunk[old_chunkno].size) -
                                         (H5O_SIZEOF_CHKSUM_OH(oh) + oh->chunk[old_chunkno].gap)),
                                        oh->chunk[old_chunkno].gap) < 0)
                                    HGOTO_ERROR(H5E_OHDR, H5E_CANTREMOVE, FAIL,
                                                "can't eliminate gap in chunk");
                            } 

                            
                            if (H5O__chunk_unprotect(f, null_chk_proxy, null_chk_dirtied) < 0)
                                HGOTO_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL,
                                            "unable to unprotect object header chunk");
                            null_chk_proxy   = NULL;
                            null_chk_dirtied = false;
                        } 
                        else {
                            size_t new_null_msg; 

                            
                            if ((null_msg->raw_size - curr_msg->raw_size) <
                                (size_t)H5O_SIZEOF_MSGHDR_OH(oh)) {
                                size_t gap_size =
                                    null_msg->raw_size - curr_msg->raw_size; 

                                
                                null_msg->raw_size = curr_msg->raw_size;

                                
                                null_msg->dirty  = true;
                                null_chk_dirtied = true;

                                
                                if (H5O__add_gap(f, oh, null_msg->chunkno, &null_chk_dirtied, v,
                                                 null_msg->raw + null_msg->raw_size, gap_size) < 0)
                                    HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, FAIL, "can't insert gap in chunk");

                                
                                new_null_msg = v;
                            } 
                            else {
                                
                                null_msg->raw += curr_msg->raw_size + (size_t)H5O_SIZEOF_MSGHDR_OH(oh);
                                null_msg->raw_size -= curr_msg->raw_size + (size_t)H5O_SIZEOF_MSGHDR_OH(oh);

                                
                                null_msg->dirty  = true;
                                null_chk_dirtied = true;

                                
                                if (oh->nmesgs >= oh->alloc_nmesgs) {
                                    if (H5O__alloc_msgs(oh, (size_t)1) < 0)
                                        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                                                    "can't allocate more space for messages");

                                    
                                    curr_msg = &oh->mesg[u];
                                } 

                                
                                new_null_msg = oh->nmesgs++;
                            } 

                            
                            if (H5O__chunk_unprotect(f, null_chk_proxy, null_chk_dirtied) < 0)
                                HGOTO_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL,
                                            "unable to unprotect object header chunk");
                            null_chk_proxy   = NULL;
                            null_chk_dirtied = false;

                            
                            oh->mesg[new_null_msg].type     = H5O_MSG_NULL;
                            oh->mesg[new_null_msg].native   = NULL;
                            oh->mesg[new_null_msg].raw      = old_raw;
                            oh->mesg[new_null_msg].raw_size = curr_msg->raw_size;
                            oh->mesg[new_null_msg].chunkno  = old_chunkno;

                            
                            oh->mesg[new_null_msg].dirty = true;
                            curr_chk_dirtied             = true;

                            
                            if (oh->chunk[old_chunkno].gap > 0) {
                                
                                if (H5O__eliminate_gap(
                                        oh, &curr_chk_dirtied, &oh->mesg[new_null_msg],
                                        ((oh->chunk[old_chunkno].image + oh->chunk[old_chunkno].size) -
                                         (H5O_SIZEOF_CHKSUM_OH(oh) + oh->chunk[old_chunkno].gap)),
                                        oh->chunk[old_chunkno].gap) < 0)
                                    HGOTO_ERROR(H5E_OHDR, H5E_CANTREMOVE, FAIL,
                                                "can't eliminate gap in chunk");
                            } 

                            
                            if (H5O__chunk_unprotect(f, curr_chk_proxy, curr_chk_dirtied) < 0)
                                HGOTO_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL,
                                            "unable to unprotect object header chunk");
                            curr_chk_proxy   = NULL;
                            curr_chk_dirtied = false;
                        } 

                        
                        packed_msg = true;

                        
                        
                        break;
                    } 
                }     

                
                
                if (packed_msg)
                    break;
            } 
        }     

        
        if (packed_msg)
            did_packing = true;
    } while (packed_msg);

    
    ret_value = (htri_t)did_packing;

done:
    if (ret_value < 0) {
        if (null_chk_proxy && H5O__chunk_unprotect(f, null_chk_proxy, null_chk_dirtied) < 0)
            HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to unprotect null object header chunk");
        if (curr_chk_proxy && H5O__chunk_unprotect(f, curr_chk_proxy, curr_chk_dirtied) < 0)
            HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to unprotect current object header chunk");
        if (cont_targ_chk_proxy && H5O__chunk_unprotect(f, cont_targ_chk_proxy, false) < 0)
            HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL,
                        "unable to unprotect continuation message target object header chunk");
    } 
    else
        assert(!null_chk_proxy && !curr_chk_proxy && !cont_targ_chk_proxy);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5O__merge_null(H5F_t *f, H5O_t *oh)
{
    bool   merged_msg;          
    bool   did_merging = false; 
    htri_t ret_value   = FAIL;  

    FUNC_ENTER_PACKAGE

    
    assert(oh != NULL);

    
    
    do {
        H5O_mesg_t *curr_msg; 
        unsigned    u;        

        
        merged_msg = false;

        
        for (u = 0, curr_msg = &oh->mesg[0]; u < oh->nmesgs; u++, curr_msg++) {
            if (H5O_NULL_ID == curr_msg->type->id) {
                H5O_mesg_t *curr_msg2; 
                unsigned    v;         

                
                assert(oh->chunk[curr_msg->chunkno].gap == 0);

                
                for (v = 0, curr_msg2 = &oh->mesg[0]; v < oh->nmesgs; v++, curr_msg2++) {
                    if (u != v && H5O_NULL_ID == curr_msg2->type->id &&
                        curr_msg->chunkno == curr_msg2->chunkno) {
                        ssize_t adj_raw      = 0; 
                        size_t  adj_raw_size = 0; 

                        
                        if ((curr_msg->raw + curr_msg->raw_size) ==
                            (curr_msg2->raw - H5O_SIZEOF_MSGHDR_OH(oh))) {
                            
                            adj_raw      = 0;
                            adj_raw_size = (size_t)H5O_SIZEOF_MSGHDR_OH(oh) + curr_msg2->raw_size;

                            
                            merged_msg = true;
                        } 
                        
                        else if ((curr_msg->raw - H5O_SIZEOF_MSGHDR_OH(oh)) ==
                                 (curr_msg2->raw + curr_msg2->raw_size)) {
                            
                            adj_raw = -((ssize_t)((size_t)H5O_SIZEOF_MSGHDR_OH(oh) + curr_msg2->raw_size));
                            adj_raw_size = (size_t)H5O_SIZEOF_MSGHDR_OH(oh) + curr_msg2->raw_size;

                            
                            merged_msg = true;
                        } 

                        
                        if (merged_msg) {
                            H5O_chunk_proxy_t *curr_chk_proxy; 
                            htri_t             result;

                            
                            H5O__msg_free_mesg(curr_msg2);

                            
                            if (NULL == (curr_chk_proxy = H5O__chunk_protect(f, oh, curr_msg->chunkno)))
                                HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL,
                                            "unable to load object header chunk");

                            
                            curr_msg->raw += adj_raw;
                            curr_msg->raw_size += adj_raw_size;

                            
                            curr_msg->dirty = true;

                            
                            if (H5O__chunk_unprotect(f, curr_chk_proxy, true) < 0)
                                HGOTO_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL,
                                            "unable to unprotect object header chunk");

                            
                            if (v < (oh->nmesgs - 1))
                                memmove(&oh->mesg[v], &oh->mesg[v + 1],
                                        ((oh->nmesgs - 1) - v) * sizeof(H5O_mesg_t));

                            
                            
                            oh->nmesgs--;

                            
                            if ((result = H5O__remove_empty_chunks(f, oh)) < 0)
                                HGOTO_ERROR(H5E_OHDR, H5E_CANTPACK, FAIL, "can't remove empty chunk");
                            else if (result > 0)
                                
                                break;

                            
                            if (curr_msg->raw_size >= H5O_MESG_MAX_SIZE)
                                if (H5O__alloc_shrink_chunk(f, oh, curr_msg->chunkno) < 0)
                                    HGOTO_ERROR(H5E_OHDR, H5E_CANTPACK, FAIL, "unable to shrink chunk");

                            
                            break;
                        } 
                    }     
                }         

                
                if (merged_msg)
                    break;
            } 
        }     

        
        if (merged_msg)
            did_merging = true;
    } while (merged_msg);

    
    ret_value = (htri_t)did_merging;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5O__remove_empty_chunks(H5F_t *f, H5O_t *oh)
{
    bool   deleted_chunk;        
    bool   did_deleting = false; 
    htri_t ret_value    = FAIL;  

    FUNC_ENTER_PACKAGE

    
    assert(oh != NULL);

    
    do {
        H5O_mesg_t *null_msg; 
        H5O_mesg_t *cont_msg; 
        unsigned    u, v;     

        
        deleted_chunk = false;

        
        for (u = 0, null_msg = &oh->mesg[0]; u < oh->nmesgs; u++, null_msg++) {
            
            if (H5O_NULL_ID == null_msg->type->id && null_msg->chunkno > 0 &&
                ((size_t)H5O_SIZEOF_MSGHDR_OH(oh) + null_msg->raw_size) ==
                    (oh->chunk[null_msg->chunkno].size - H5O_SIZEOF_CHKHDR_OH(oh))) {
                H5O_mesg_t *curr_msg;        
                unsigned    null_msg_no;     
                unsigned    deleted_chunkno; 

                
                for (v = 0, cont_msg = &oh->mesg[0]; v < oh->nmesgs; v++, cont_msg++) {
                    if (H5O_CONT_ID == cont_msg->type->id) {
                        
                        H5O_LOAD_NATIVE(f, 0, oh, cont_msg, FAIL)

                        
                        
                        if (0 == ((H5O_cont_t *)(cont_msg->native))->chunkno) {
                            unsigned w; 

                            
                            for (w = 0; w < oh->nchunks; w++)
                                if (oh->chunk[w].addr == ((H5O_cont_t *)(cont_msg->native))->addr) {
                                    ((H5O_cont_t *)(cont_msg->native))->chunkno = w;
                                    break;
                                } 
                            assert(((H5O_cont_t *)(cont_msg->native))->chunkno > 0);
                        } 

                        
                        if (oh->chunk[null_msg->chunkno].addr == ((H5O_cont_t *)(cont_msg->native))->addr)
                            break;
                    } 
                }     
                
                assert(v < oh->nmesgs);
                assert(cont_msg);
                assert(((H5O_cont_t *)(cont_msg->native))->chunkno == null_msg->chunkno);

                
                null_msg_no     = u;
                deleted_chunkno = null_msg->chunkno;

                
                if (H5O__release_mesg(f, oh, cont_msg, true) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTDELETE, FAIL, "unable to convert into null message");

                

                
                oh->chunk[null_msg->chunkno].image =
                    H5FL_BLK_FREE(chunk_image, oh->chunk[null_msg->chunkno].image);

                
                if (null_msg->chunkno < (oh->nchunks - 1)) {
                    memmove(&oh->chunk[null_msg->chunkno], &oh->chunk[null_msg->chunkno + 1],
                            ((oh->nchunks - 1) - null_msg->chunkno) * sizeof(H5O_chunk_t));

                    
                    for (u = null_msg->chunkno; u < (oh->nchunks - 1); u++) {
                        unsigned chk_proxy_status = 0; 

                        
                        if (H5AC_get_entry_status(f, oh->chunk[u].addr, &chk_proxy_status) < 0)
                            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL,
                                        "unable to check metadata cache status for chunk proxy");

                        
                        if (chk_proxy_status & H5AC_ES__IN_CACHE) {
                            if (H5O__chunk_update_idx(f, oh, u) < 0)
                                HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL,
                                            "unable to update index for chunk proxy");
                        } 
                    }     
                }         

                
                
                oh->nchunks--;

                

                
                H5O__msg_free_mesg(null_msg);

                
                if (null_msg_no < (oh->nmesgs - 1))
                    memmove(&oh->mesg[null_msg_no], &oh->mesg[null_msg_no + 1],
                            ((oh->nmesgs - 1) - null_msg_no) * sizeof(H5O_mesg_t));

                
                
                oh->nmesgs--;

                
                for (u = 0, curr_msg = &oh->mesg[0]; u < oh->nmesgs; u++, curr_msg++) {
                    
                    assert(curr_msg->chunkno != deleted_chunkno);

                    
                    if (curr_msg->chunkno > deleted_chunkno)
                        curr_msg->chunkno--;

                    
                    if (H5O_CONT_ID == curr_msg->type->id) {
                        
                        H5O_LOAD_NATIVE(f, 0, oh, curr_msg, FAIL)

                        
                        
                        if (0 == ((H5O_cont_t *)(curr_msg->native))->chunkno) {
                            unsigned w; 

                            
                            for (w = 0; w < oh->nchunks; w++)
                                if (oh->chunk[w].addr == ((H5O_cont_t *)(curr_msg->native))->addr) {
                                    ((H5O_cont_t *)(curr_msg->native))->chunkno = w;
                                    break;
                                } 
                            assert(((H5O_cont_t *)(curr_msg->native))->chunkno > 0);
                        } 
                        else {
                            
                            if (((H5O_cont_t *)(curr_msg->native))->chunkno > deleted_chunkno)
                                ((H5O_cont_t *)(curr_msg->native))->chunkno--;
                        } 
                    }     
                }         

                
                deleted_chunk = true;
                break;
            } 
        }     

        
        if (deleted_chunk)
            did_deleting = true;
    } while (deleted_chunk);

    
    ret_value = (htri_t)did_deleting;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__condense_header(H5F_t *f, H5O_t *oh)
{
    bool   rescan_header;       
    htri_t result;              
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(oh != NULL);

    
    for (unsigned u = 0; oh->num_deleted_mesgs > 0 && u < oh->nmesgs;)
        if (oh->mesg[u].type->id == H5O_DELETED_ID) {
            if (u < (oh->nmesgs - 1))
                memmove(&oh->mesg[u], &oh->mesg[u + 1], ((oh->nmesgs - 1) - u) * sizeof(H5O_mesg_t));
            oh->nmesgs--;
            oh->num_deleted_mesgs--;
        }
        else
            u++;
    assert(oh->num_deleted_mesgs == 0);

    
    do {
        
        rescan_header = false;

        
        result = H5O__move_msgs_forward(f, oh);
        if (result < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTPACK, FAIL, "can't move header messages forward");
        if (result > 0)
            rescan_header = true;

        
        result = H5O__merge_null(f, oh);
        if (result < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTPACK, FAIL, "can't pack null header messages");
        if (result > 0)
            rescan_header = true;

        
        result = H5O__remove_empty_chunks(f, oh);
        if (result < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTPACK, FAIL, "can't remove empty chunk");
        if (result > 0)
            rescan_header = true;
    } while (rescan_header);
#ifdef H5O_DEBUG
    H5O__assert(oh);
#endif 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__alloc_shrink_chunk(H5F_t *f, H5O_t *oh, unsigned chunkno)
{
    H5O_chunk_t       *chunk     = &oh->chunk[chunkno];      
    H5O_chunk_proxy_t *chk_proxy = NULL;                     
    H5O_mesg_t        *curr_msg;                             
    uint8_t           *old_image = chunk->image;             
    size_t             old_size  = chunk->size;              
    size_t             new_size  = chunk->size - chunk->gap; 
    size_t             total_msg_size;                       
    size_t             min_chunk_size    = H5O_ALIGN_OH(oh, H5O_MIN_SIZE); 
    size_t             sizeof_chksum     = H5O_SIZEOF_CHKSUM_OH(oh);       
    size_t             sizeof_msghdr     = H5O_SIZEOF_MSGHDR_OH(oh);       
    uint8_t            new_size_flags    = 0;                              
    bool               adjust_size_flags = false; 
    size_t             less_prfx_size    = 0;     
    size_t             u;                         
    herr_t             ret_value = SUCCEED;       

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(oh);

    
    if (NULL == (chk_proxy = H5O__chunk_protect(f, oh, chunkno)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to protect object header chunk");

    
    for (u = oh->nmesgs - 1, curr_msg = &oh->mesg[u]; u < oh->nmesgs; u--, curr_msg--) {
        if ((H5O_NULL_ID == curr_msg->type->id) && (chunkno == curr_msg->chunkno)) {
            size_t shrink_size = curr_msg->raw_size + sizeof_msghdr; 

            
            if (curr_msg->raw + curr_msg->raw_size < old_image + new_size - sizeof_chksum) {
                unsigned    v; 
                H5O_mesg_t *curr_msg2;
                uint8_t    *src = curr_msg->raw + curr_msg->raw_size; 

                
                memmove(curr_msg->raw - sizeof_msghdr, src,
                        (size_t)(old_image + new_size - sizeof_chksum - src));

                
                for (v = 0, curr_msg2 = &oh->mesg[0]; v < oh->nmesgs; v++, curr_msg2++)
                    if ((chunkno == curr_msg2->chunkno) && (curr_msg2->raw > curr_msg->raw))
                        curr_msg2->raw -= shrink_size;
            } 

            
            new_size -= shrink_size;

            
            H5O__msg_free_mesg(curr_msg);

            
            if (u < (oh->nmesgs - 1))
                memmove(&oh->mesg[u], &oh->mesg[u + 1], ((oh->nmesgs - 1) - u) * sizeof(H5O_mesg_t));

            
            
            oh->nmesgs--;
        } 
    }     

    
    total_msg_size = new_size - (size_t)(chunkno == 0 ? H5O_SIZEOF_HDR(oh) : H5O_SIZEOF_CHKHDR_OH(oh));
    if (total_msg_size < min_chunk_size) {
        assert(oh->alloc_nmesgs > oh->nmesgs);
        oh->nmesgs++;

        
        curr_msg         = &oh->mesg[oh->nmesgs - 1];
        curr_msg->type   = H5O_MSG_NULL;
        curr_msg->dirty  = true;
        curr_msg->native = NULL;
        curr_msg->raw    = old_image + new_size + sizeof_msghdr - sizeof_chksum;
        curr_msg->raw_size =
            MAX(H5O_ALIGN_OH(oh, min_chunk_size - total_msg_size), sizeof_msghdr) - sizeof_msghdr;
        curr_msg->chunkno = chunkno;

        
        new_size += curr_msg->raw_size + sizeof_msghdr;
    } 

    
    if (oh->version > H5O_VERSION_1 && chunkno == 0) {
        uint64_t chunk0_newsize = new_size - (size_t)H5O_SIZEOF_HDR(oh); 
        size_t   orig_prfx_size = (size_t)1 << (oh->flags & H5O_HDR_CHUNK0_SIZE); 

        
        if (orig_prfx_size > 1 && chunk0_newsize <= 255) {
            less_prfx_size    = orig_prfx_size - 1;
            new_size_flags    = H5O_HDR_CHUNK0_1;
            adjust_size_flags = true;
        } 
        
        else if (orig_prfx_size > 2 && chunk0_newsize <= 65535) {
            less_prfx_size    = orig_prfx_size - 2;
            new_size_flags    = H5O_HDR_CHUNK0_2;
            adjust_size_flags = true;
        } 
        
        else if (orig_prfx_size > 4 && chunk0_newsize <= 4294967295) {
            less_prfx_size    = orig_prfx_size - 4;
            new_size_flags    = H5O_HDR_CHUNK0_4;
            adjust_size_flags = true;
        } 
    }     

    if (adjust_size_flags) {
        
        oh->flags = (uint8_t)(oh->flags & ~H5O_HDR_CHUNK0_SIZE);
        oh->flags |= new_size_flags;

        
        memmove(chunk->image + H5O_SIZEOF_HDR(oh) - sizeof_chksum,
                chunk->image + H5O_SIZEOF_HDR(oh) - sizeof_chksum + less_prfx_size,
                new_size - (size_t)H5O_SIZEOF_HDR(oh));

        
        new_size -= less_prfx_size;
    } 

    
    chunk->size  = new_size;
    chunk->image = H5FL_BLK_REALLOC(chunk_image, old_image, chunk->size);
    chunk->gap   = 0;
    if (NULL == oh->chunk[chunkno].image)
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    for (u = 0, curr_msg = &oh->mesg[0]; u < oh->nmesgs; u++, curr_msg++) {
        if (adjust_size_flags || (chunk->image != old_image))
            
            if (curr_msg->chunkno == chunkno)
                curr_msg->raw = chunk->image - less_prfx_size + (curr_msg->raw - old_image);

        
        
        if (chunkno > 0 && (H5O_CONT_ID == curr_msg->type->id) &&
            (((H5O_cont_t *)(curr_msg->native))->chunkno == chunkno)) {
            H5O_chunk_proxy_t *cont_chk_proxy; 

            
            if (NULL == (cont_chk_proxy = H5O__chunk_protect(f, oh, curr_msg->chunkno)))
                HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to protect object header chunk");

            
            assert(((H5O_cont_t *)(curr_msg->native))->size == old_size);
            ((H5O_cont_t *)(curr_msg->native))->size = chunk->size;

            
            curr_msg->dirty = true;

            
            if (H5O__chunk_unprotect(f, cont_chk_proxy, true) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to unprotect object header chunk");
        } 
    }     

    assert(new_size <= old_size);

    
    if (H5O__chunk_resize(oh, chk_proxy) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTRESIZE, FAIL, "unable to resize object header chunk");

    
    if (H5MF_xfree(f, H5FD_MEM_OHDR, chunk->addr + new_size, (hsize_t)(old_size - new_size)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTFREE, FAIL, "unable to shrink object header chunk");

done:
    
    if (chk_proxy && H5O__chunk_unprotect(f, chk_proxy, true) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to unprotect object header chunk");

    FUNC_LEAVE_NOAPI(ret_value)
} 
