#' @title Function to solve for the mean frequency effect size given target power for one-sample simulations.
#' @description
#' Computes the mean frequency difference parameter \eqn{\Delta} that achieves a specified power for a two-sided group sequential
#' test.
#'
#' @param K Number of analysis (including the final analysis).
#' @param timing Numeric vector of information fractions at each analysis, with values in \eqn{(0, 1)} and length equal to \code{K}.
#' @param alpha Overall Type I error.
#' @param power Target power of thw group sequenctial test.
#' @param Imax Maximum information at the final analysis.
#'
#' @importFrom gsDesign gsDesign
#' @importFrom mvtnorm pmvnorm
#' @importFrom stats uniroot
#'
#' @returns A numeric value giving the mean frequency effect size \eqn{\Delta} that achieves the specified power.
#' @export
#'
#' @examples
#' # Toy Example: n = 800, two interim analysi planned at 50% and
#' # 75% if maximum information, along with a final analysis
#' # at 100% maximum information. Assuming the maximum
#' # information is 100, overall Type I error at 0.05 and power at 0.8.
#' find.Delta.given.power(K = 3, timing = c(0.5, 0.75, 1),
#'  alpha = 0.05, power = 0.8, Imax = 100)
find.Delta.given.power <- function(K, timing, alpha, power, Imax){

  f = function(Delta){
    # Find the canonical boundary
    canonical <- gsDesign(k = K, test.type = 2, alpha = alpha/2, sfu = "OF",
                          timing = timing)
    canonical.bdry <- canonical$upper$bound

    mean.vec = sqrt(Imax)*sqrt(timing)*Delta
    canonical.corr <- sqrt(outer(timing, timing, "/"))
    canonical.corr[lower.tri(canonical.corr)] <- t(canonical.corr)[lower.tri(canonical.corr)]
    prob = pmvnorm(lower = -canonical.bdry, upper = canonical.bdry, mean = mean.vec, corr = canonical.corr)
    return(prob[1] - (1- power))
  }
  solution = uniroot(f, c(0, 10))$root
  return(solution)
}
