#' @title Function to calculate the Z statistics and variance for the time-to-first-event (TTFE) method.
#' @description
#' Computes a two-sample TTFE test statistics by retaining only the first observed event for each subject and
#' applying a standard log-rank test. This function is intended as a comparator or benchmark to the mean-frequency
#' based estimator implemented elsewhere in the package.
#'
#' @param data A data frame generated by \code{TwoSamaple.generate.sequential()}.
#'
#' @returns A list with:
#' \itemize{
#' \item \code{Z.stat}: The ordinary log-rank statistics on the Z-scale, defined as the square root of the chi-square
#' statistic from \code{survdiff()}.
#' \item \code{var}: The estimated variance of the test statistic. extracted from the log-rank test variance matrix.
#' }
#' @export
#' @importFrom dplyr group_by filter mutate slice
#' @importFrom survival Surv survdiff
TwoSample.Estimator.TTFE.sequential <- function(data){

  # keep the first event within each id
  original.data <- data %>%
    dplyr::group_by(.data$id) %>%
    dplyr::filter(!is.na(.data$status)) %>%
    dplyr::mutate(true_event_time = .data$event_time_cal - .data$e) %>%
    slice(1)

  fit <- survdiff(Surv(true_event_time, event) ~ group, data = original.data)

  Z.stat <- sqrt(fit$chisq)
  var <- fit$var[1,1]

  n <- length(unique(original.data$id))

  return(list(Z.stat = Z.stat, var = var ))

}
