% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/legend_keys.R
\name{legend_keys}
\alias{legend_keys}
\title{Customize Legend Key Appearance}
\usage{
legend_keys(
  size = NULL,
  alpha = NULL,
  shape = NULL,
  linewidth = NULL,
  linetype = NULL,
  fill = NULL,
  colour = NULL,
  color = NULL,
  stroke = NULL,
  aesthetic = c("colour", "fill")
)
}
\arguments{
\item{size}{Numeric. Size of point keys.}

\item{alpha}{Numeric. Alpha (transparency) of keys, between 0 and 1.}

\item{shape}{Shape of point keys. Can be:
\itemize{
\item Numeric (0-25): Standard ggplot2 shape codes
\item Character name: \code{"circle"}, \code{"square"}, \code{"diamond"},
\code{"triangle"}, \code{"triangle_down"}, \code{"plus"}, \code{"cross"},
\code{"asterisk"}, \code{"circle_open"}, \code{"square_open"},
\code{"diamond_open"}, \code{"triangle_open"}, \code{"circle_filled"},
\code{"square_filled"}, \code{"diamond_filled"}, \code{"triangle_filled"}
}
Shapes 21-25 (or names ending in \code{"_filled"}) support both outline
(\code{colour}) and fill (\code{fill}) colors.}

\item{linewidth}{Numeric. Width of line keys.}

\item{linetype}{Character or numeric. Line type for line keys.}

\item{fill}{Character. Fill color for filled shapes (shapes 21-25). For
shapes 0-20, use \code{colour} instead.}

\item{colour, color}{Character. Outline/stroke color for keys. For shapes
21-25, this controls the outline; for shapes 0-20, this is the main color.}

\item{stroke}{Numeric. Stroke width for point outlines (shapes 21-25).}

\item{aesthetic}{Character vector specifying which aesthetic(s) to modify.
Default is \code{c("colour", "fill")} which covers most common cases.
Use \code{"all"} to apply to all legends.}
}
\value{
A list of ggplot2 guide specifications.
}
\description{
Modify the appearance of legend keys (the symbols/glyphs in the legend)
without affecting the plot itself. This is a simpler alternative to using
\code{guide_legend(override.aes = list(...))}.
}
\details{
This function wraps \code{guide_legend(override.aes = ...)} to provide a cleaner
interface for common legend key modifications. It's particularly useful for:

\itemize{
\item Making small points more visible in the legend
\item Removing transparency from legend keys
\item Changing symbol shapes to improve clarity
\item Adding outlines to filled shapes for better visibility
}

\strong{Shape types:}
\itemize{
\item Shapes 0-14: Outline only (color from \code{colour})
\item Shapes 15-20: Filled solid (color from \code{colour})
\item Shapes 21-25: Outline + fill (outline from \code{colour}, interior
from \code{fill})
}

\strong{Important note for filled shapes (21-25):}

When using filled shapes with both outline and fill colors, the behavior
depends on which aesthetics are mapped in your original plot:

\itemize{
\item \strong{White fill, colored outline}: Works with \code{aes(color = var)}.
Use \code{legend_keys(shape = "circle_filled", fill = "white")}.
\item \strong{Colored fill, black outline}: Requires \code{aes(color = var, fill = var)}
in your plot. Then use \code{legend_keys(colour = "black")}.
}

This is because \code{override.aes} can only set static values; it cannot
inherit from mapped aesthetics. If you only map \code{color} and try to
override the outline to black, the fill will not have a color mapping to use.

Only non-NULL arguments are applied, so you can selectively modify specific
properties.
}
\examples{
library(ggplot2)

# Points get lost in legend - make them bigger
ggplot(mtcars, aes(mpg, wt, color = factor(cyl))) +
  geom_point(size = 1) +
  legend_keys(size = 4)

# Remove transparency from legend
ggplot(mtcars, aes(mpg, wt, color = factor(cyl))) +
  geom_point(alpha = 0.3, size = 3) +
  legend_keys(alpha = 1)

# Change shape using name
ggplot(mtcars, aes(mpg, wt, color = factor(cyl))) +
  geom_point(size = 3) +
  legend_keys(shape = "square")

# Filled shape with white fill and colored outline (shapes 21-25)
# Works because we set fill to a static color (white)
ggplot(mtcars, aes(mpg, wt, color = factor(cyl))) +
  geom_point(size = 3) +
  legend_keys(shape = "circle_filled", fill = "white", stroke = 1.5)

# Colored fill with black outline - MUST map both color AND fill in the plot
# This is a ggplot2 limitation: override.aes can only set static values,
# it cannot make fill "inherit" from color
ggplot(mtcars, aes(mpg, wt, color = factor(cyl), fill = factor(cyl))) +
  geom_point(size = 3, shape = 21, stroke = 1) +
  legend_keys(colour = "black", stroke = 1)

# Apply to fill aesthetic (e.g., for boxplots)
ggplot(mtcars, aes(factor(cyl), mpg, fill = factor(cyl))) +
  geom_boxplot(alpha = 0.5) +
  legend_keys(alpha = 1, aesthetic = "fill")

}
\seealso{
\code{\link{legend_style}} for styling legend text and background,
\code{\link{legend_order}} for reordering legend entries.
}
